/**
 * WordExporter - HTML转DOCX导出器
 * 基于html-docx-js库的封装，提供与PDF导出器一致的接口
 */
class WordExporter {
  constructor() {
    if (!window.htmlDocx) {
      console.error('[WordExporter] html-docx库未加载');
      throw new Error('html-docx库未加载');
    }
    
  }

  /**
   * 导出HTML元素为DOCX文档
   * @param {HTMLElement} element - 要导出的HTML元素
   * @param {Object} options - 导出选项
   * @returns {Promise<void>}
   */
  async exportToWord(element, options = {}) {
    try {
      // 准备HTML内容
      const htmlContent = this.prepareHtmlContent(element);
      
      // 配置选项
      const exportOptions = this.buildExportOptions(options);
      
      // 使用html-docx库生成DOCX
      const docxBlob = window.htmlDocx.asBlob(htmlContent, exportOptions);
      
      // 下载文件
      this.downloadBlob(docxBlob, options.fileName || 'document.docx');
      
      return {
        success: true,
        filename: options.fileName || 'document.docx'
      };
    } catch (error) {
      console.error('[WordExporter] 导出失败:', error);
      throw new Error(`Word文档导出失败: ${error.message}`);
    }
  }

  /**
   * 准备HTML内容，确保符合html-docx要求
   * @param {HTMLElement} element - HTML元素
   * @returns {string} - 格式化的HTML内容
   */
  prepareHtmlContent(element) {
    // 克隆元素以避免修改原始DOM
    const clonedElement = element.cloneNode(true);

    // 处理所有图片元素，确保尺寸正确应用
    const images = clonedElement.querySelectorAll('img');
    images.forEach(img => {
      const displayWidth = img.getAttribute('data-display-width');
      const displayHeight = img.getAttribute('data-display-height');

      if (displayWidth && displayHeight) {
        // 获取现有的style属性（如果有的话）
        const existingStyle = img.getAttribute('style') || '';

        // 创建新的样式，优先使用data-display-width/height
        const newStyle = [
          `width: ${displayWidth}px !important`,
          `height: ${displayHeight}px !important`,
          `max-width: ${displayWidth}px !important`,
          `max-height: ${displayHeight}px !important`,
          `display: block`,
          `margin: 15px auto`,
          `object-fit: contain`
        ].join('; ');

        // 设置新样式
        img.setAttribute('style', newStyle);

        // 同时设置HTML属性，确保Word能正确识别
        img.setAttribute('width', displayWidth);
        img.setAttribute('height', displayHeight);
      }
    });

    // 获取处理后的HTML内容
    let htmlContent = clonedElement.outerHTML;
    
    // html-docx需要完整的HTML文档结构
    const fullHtml = `
<!DOCTYPE html>
<html>
<head>
  <meta charset="utf-8">
  <title>Document</title>
  <style>
    body {
      font-family: 'Helvetica Neue', Helvetica, Arial, sans-serif;
      line-height: 1.6;
      color: #333;
      max-width: none;
      margin: 0;
      padding: 20px;
    }
    h1, h2, h3, h4, h5, h6 {
      color: #2c3e50;
      margin-top: 24px;
      margin-bottom: 16px;
    }
    p {
      margin-bottom: 16px;
    }
    pre {
      background-color: #f8f9fa;
      border: 1px solid #e9ecef;
      border-radius: 4px;
      padding: 16px;
      overflow-x: auto;
    }
    code {
      background-color: #f8f9fa;
      padding: 2px 4px;
      border-radius: 3px;
      font-family: 'Courier New', Courier, monospace;
    }
    blockquote {
      border-left: 4px solid #ddd;
      margin: 0 0 16px;
      padding: 0 16px;
      color: #666;
    }
    table {
      border-collapse: collapse;
      width: 100%;
      margin-bottom: 16px;
    }
    th, td {
      border: 1px solid #ddd;
      padding: 8px;
      text-align: left;
    }
    th {
      background-color: #f2f2f2;
    }
    img {
      /* 移除默认样式，让内联样式生效 */
      display: block;
      margin: 15px auto;
    }
  </style>
</head>
<body>
  ${htmlContent}
</body>
</html>`;
    
    return fullHtml;
  }

  /**
   * 构建导出选项
   * @param {Object} options - 用户选项
   * @returns {Object} - html-docx选项
   */
  buildExportOptions(options) {
    return {
      orientation: options.orientation || 'portrait',
      margins: {
        top: 720,    // 0.5 inch = 720 twips
        right: 720,
        bottom: 720,
        left: 720,
        header: 360, // 0.25 inch
        footer: 360
      }
    };
  }

  /**
   * 下载Blob文件
   * @param {Blob} blob - 文件Blob
   * @param {string} filename - 文件名
   */
  downloadBlob(blob, filename) {
    // 创建下载链接
    const url = window.URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = filename;
    
    // 触发下载
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    
    // 清理URL对象
    window.URL.revokeObjectURL(url);
  }
}

// 延迟检查和初始化，确保所有依赖都已加载
setTimeout(() => {
  if (window.htmlDocx) {
  } else {
    console.error('[WordExporter] 警告: html-docx库不可用!');
  }

  // 将WordExporter暴露到全局作用域
  window.WordExporter = WordExporter;

}, 100);