/**
 * 基于文本锚点的精确滚动同步
 * 
 * 核心思想：
 * 1. 提取多个唯一的文本片段作为锚点
 * 2. 建立文本片段与位置的精确映射
 * 3. 通过多个锚点三角定位，实现精确滚动
 * 4. 避免模糊匹配导致的跳动
 */

(function() {
  'use strict';
  
  class TextAnchorScrollSync {
    constructor() {
      this.config = {
        // 锚点配置
        anchorCount: 8,           // 同时追踪的锚点数量（增加从5到8）
        anchorMinLength: 12,       // 锚点最小长度（降低从20到12）
        anchorMaxLength: 120,      // 锚点最大长度（增加从100到120）
        uniquenessThreshold: 0.95, // 唯一性阈值（1.0表示完全唯一）
        
        // 性能配置
        debounceDelay: 100,        // 防抖延迟
        cacheTimeout: 5000,        // 缓存超时时间
        
        // 滚动配置
        smoothScroll: true,
        scrollDuration: 200,       // 降低时长，减少延迟感
        
        // 通信配置
        channelId: 'md2pdf-text-anchor-sync',
        
        // 调试
        debug: localStorage.getItem('md2pdf_anchor_debug') === 'true'
      };
      
      // 状态管理
      this.state = {
        isScrolling: false,
        scrollSource: null,
        lastUpdate: 0,
        scrollTimer: null
      };
      
      // 锚点缓存
      this.anchors = {
        local: new Map(),      // 本地文本 -> 位置映射
        remote: new Map(),     // 远程锚点缓存
        uniqueTexts: new Set() // 唯一文本集合
      };
      
      // 文本提取器（根据容器类型）
      this.textExtractor = null;
      
      // 滚动容器
      this.container = null;
      
      // 初始化
      this.init();
    }
    
    /**
     * 初始化
     */
    init() {
      console.log('[TextAnchorSync] 开始初始化，document.readyState:', document.readyState);
      if (document.readyState === 'loading') {
        console.log('[TextAnchorSync] 等待DOMContentLoaded事件...');
        document.addEventListener('DOMContentLoaded', () => {
          console.log('[TextAnchorSync] DOMContentLoaded事件触发，开始setup');
          this.setup();
        });
      } else {
        console.log('[TextAnchorSync] DOM已就绪，直接开始setup');
        this.setup();
      }
    }
    
    /**
     * 设置
     */
    setup() {
      console.log('[TextAnchorSync] 开始设置...');
      
      // 识别容器类型和设置提取器
      this.identifyContainer();
      
      if (!this.container) {
        console.log('[TextAnchorSync] ❌ 未找到合适的容器，500ms后重试...');
        // 延迟重试，可能是DOM还未完全加载
        setTimeout(() => {
          console.log('[TextAnchorSync] 重试容器识别...');
          this.identifyContainer();
          if (this.container) {
            this.setupAfterContainerFound();
          } else {
            console.log('[TextAnchorSync] ❌ 重试后仍未找到容器');
          }
        }, 500);
        return;
      }
      
      this.setupAfterContainerFound();
    }
    
    setupAfterContainerFound() {
      console.log('[TextAnchorSync] ✅ 找到容器:', this.container.tagName, this.container.id || this.container.className);
      console.log('[TextAnchorSync] 源ID:', this.getSourceId());
      
      // 设置监听器（先设置监听器）
      this.setupListeners();
      console.log('[TextAnchorSync] ✅ 监听器设置完成');
      
      // 延迟构建锚点索引，等待内容加载
      this.delayedBuildAnchors();
      console.log('[TextAnchorSync] ✅ 设置完成');
    }
    
    /**
     * 延迟构建锚点，等待内容加载
     */
    delayedBuildAnchors() {
      // 只有编辑器才构建锚点
      if (!this.isEditor) {
        console.log('[TextAnchorSync] 📺 预览区域，跳过锚点构建');
        return;
      }
      
      console.log('[TextAnchorSync] 开始延迟构建锚点索引...');
      // 立即尝试构建
      this.buildAnchorIndex();
      console.log('[TextAnchorSync] 第一次构建完成，锚点数量:', this.anchors.local.size);
      
      // 如果没有锚点，延迟后重试
      if (this.anchors.local.size === 0) {
        console.log('[TextAnchorSync] 没有锚点，1000ms后重试...');
        setTimeout(() => {
          this.buildAnchorIndex();
          console.log('[TextAnchorSync] 第二次构建完成，锚点数量:', this.anchors.local.size);
          
          if (this.anchors.local.size === 0) {
            console.log('[TextAnchorSync] 仍然没有锚点，2000ms后最后重试...');
            setTimeout(() => {
              this.buildAnchorIndex();
              console.log('[TextAnchorSync] 最终构建完成，锚点数量:', this.anchors.local.size);
            }, 2000);
          }
        }, 1000);
      }
    }
    
    /**
     * 识别容器并设置相应的文本提取器
     */
    identifyContainer() {
      console.log('[TextAnchorSync] 识别容器类型...');
      
      // 检查是否是编辑器（textarea）- 只有编辑器才生成锚点
      const textarea = document.querySelector('#markdown-input, textarea.markdown-editor');
      if (textarea) {
        console.log('[TextAnchorSync] ✅ 识别为编辑器，将生成锚点');
        this.container = textarea;
        this.textExtractor = new TextareaExtractor(textarea);
        this.isEditor = true;
        return;
      }
      
      // 检查是否是渲染器/预览区域 - 只负责搜索锚点
      const renderContainer = document.querySelector('#render-container');
      if (renderContainer) {
        console.log('[TextAnchorSync] 📺 识别为预览区域，只负责搜索锚点');
        this.container = renderContainer;
        this.textExtractor = new DOMExtractor(renderContainer);
        this.isEditor = false;
        return;
      }
      
      // 检查是否是其他预览区
      const preview = document.querySelector('.markdown-body, .preview-content');
      if (preview) {
        console.log('[TextAnchorSync] 📺 识别为预览区域，只负责搜索锚点');
        this.container = preview;
        this.textExtractor = new DOMExtractor(preview);
        this.isEditor = false;
        return;
      }
      
      // 默认情况，假设是编辑器环境
      console.log('[TextAnchorSync] ⚠️ 未找到已知容器，假设为编辑器环境');
      this.container = document.body;
      this.textExtractor = new DOMExtractor(document.body);
      this.isEditor = true;
    }
    
    /**
     * 构建锚点索引 - 超级简化版本
     */
    buildAnchorIndex() {
      console.log('[TextAnchorSync] 开始构建锚点索引...');
      const startTime = performance.now();
      
      // 清空旧索引
      this.anchors.local.clear();
      this.anchors.uniqueTexts.clear();
      
      // 获取所有行文本
      let lines = [];
      if (this.textExtractor.textarea) {
        // 针对textarea
        lines = this.textExtractor.textarea.value.split('\n');
      } else if (this.container) {
        // 针对DOM容器，提取文本内容
        lines = this.container.textContent.split('\n');
      } else {
        console.log('[TextAnchorSync] 无法获取文本内容');
        return;
      }
      console.log(`[TextAnchorSync] 总行数: ${lines.length}`);
      
      // 第1步：从每行随机获取文字片段
      const allFragments = [];
      
      lines.forEach((line, lineIndex) => {
        const cleanLine = line.trim();
        
        // 跳过空行、代码块、LaTeX、Mermaid - 降低过滤严格性
        if (cleanLine.length < 6) return;  // 降低最小长度要求
        if (cleanLine.startsWith('```')) return;
        if (cleanLine.startsWith('\\[') || cleanLine.startsWith('$$')) return;
        if (cleanLine.includes('mermaid') || cleanLine.includes('latex')) return;
        
        // 增加锚点密度：从整行中获取更多片段
        const minLength = 8;   // 降低最小长度
        const maxLength = 25;  // 降低最大长度
        const fragmentCount = Math.min(5, Math.floor(cleanLine.length / 15)); // 动态片段数量
        
        for (let i = 0; i < fragmentCount; i++) {
          const fragmentLength = minLength + Math.floor(Math.random() * (maxLength - minLength));
          
          if (cleanLine.length > fragmentLength) {
            const maxStart = cleanLine.length - fragmentLength;
            const startPos = Math.floor(Math.random() * maxStart);
            const fragment = cleanLine.substring(startPos, startPos + fragmentLength).trim();
            
            // 降低文字要求，增加锚点生成
            if (/[\u4e00-\u9fa5a-zA-Z]{4,}/.test(fragment) && fragment.length >= 6) {
              allFragments.push({
                text: fragment,
                line: lineIndex,
                originalLine: line
              });
            }
          }
        }
      });
      
      console.log(`[TextAnchorSync] 生成随机片段: ${allFragments.length}个`);
      
      // 第2步：统计每个片段的出现次数
      const fragmentCounts = new Map();
      allFragments.forEach(fragment => {
        const text = fragment.text;
        if (!fragmentCounts.has(text)) {
          fragmentCounts.set(text, []);
        }
        fragmentCounts.get(text).push(fragment);
      });
      
      // 第3步：只保留出现次数为1的唯一片段，并存储更精确的位置信息
      let uniqueCount = 0;
      fragmentCounts.forEach((occurrences, text) => {
        if (occurrences.length === 1) {
          const fragment = occurrences[0];
          
          // 计算更准确的字符偏移量
          const linesBeforeCurrent = lines.slice(0, fragment.line);
          const characterOffset = linesBeforeCurrent.join('\n').length + (linesBeforeCurrent.length > 0 ? 1 : 0);
          
          this.anchors.local.set(text, {
            line: fragment.line,
            offset: characterOffset,
            lineText: fragment.originalLine,  // 保存完整行文本用于调试
            fragmentStart: fragment.originalLine.indexOf(text) // 在行内的位置
          });
          uniqueCount++;
        }
      });
      
      const endTime = performance.now();
      console.log(`[TextAnchorSync] 锚点构建完成: ${uniqueCount}个唯一锚点，用时: ${(endTime - startTime).toFixed(2)}ms`);
      
      // 显示锚点分布
      if (uniqueCount > 0) {
        const lines = Array.from(this.anchors.local.values()).map(pos => pos.line).sort((a, b) => a - b);
        console.log(`[TextAnchorSync] 锚点分布在行: [${lines.join(', ')}]`);
      }
    }
    
    
    
    
    
    /**
     * 设置监听器
     */
    setupListeners() {
      // 滚动监听
      this.setupScrollListener();
      
      // 消息监听
      this.setupMessageListener();
      
      // 内容变化监听
      this.setupMutationObserver();
    }
    
    /**
     * 设置滚动监听
     */
    setupScrollListener() {
      const handleScroll = () => {
        console.log('[TextAnchorSync] 滚动事件触发，源:', this.getSourceId(), '滚动来源:', this.state.scrollSource);
        
        if (this.state.scrollSource === 'remote') {
          console.log('[TextAnchorSync] 忽略远程触发的滚动');
          return;
        }
        
        clearTimeout(this.state.scrollTimer);
        this.state.scrollTimer = setTimeout(() => {
          console.log('[TextAnchorSync] 防抖延迟后处理本地滚动事件');
          this.handleLocalScroll();
        }, this.config.debounceDelay);
      };
      
      if (this.container.tagName === 'TEXTAREA') {
        console.log('[TextAnchorSync] 设置textarea滚动监听器');
        this.container.addEventListener('scroll', handleScroll, { passive: true });
      } else {
        // 对于div容器，需要监听正确的滚动目标
        // 在iframe中应该监听window滚动
        let scrollTarget;
        if (this.container.id === 'render-container' || this.getSourceId() === 'renderer') {
          scrollTarget = window;  // iframe中监听window滚动
          console.log('[TextAnchorSync] 渲染器环境，监听window滚动');
        } else {
          scrollTarget = this.container === document.body ? window : this.container;
          console.log('[TextAnchorSync] 其他环境，监听目标:', scrollTarget === window ? 'window' : 'container');
        }
        
        scrollTarget.addEventListener('scroll', handleScroll, { passive: true });
        console.log('[TextAnchorSync] ✅ 滚动监听器已设置，目标:', scrollTarget === window ? 'window' : scrollTarget.tagName);
        
        // 保留测试函数供调试使用
        this.testScrollListener = () => {
          handleScroll();
        };
      }
    }
    
    /**
     * 处理本地滚动
     */
    handleLocalScroll() {
      // 只有编辑器才发送锚点同步
      if (!this.isEditor) {
        console.log('[TextAnchorSync] 📺 预览区域滚动，不发送同步消息');
        return;
      }
      
      console.log('[TextAnchorSync] 处理编辑器滚动，当前锚点数量:', this.anchors.local.size);
      this.state.scrollSource = 'local';
      
      // 获取当前可见区域的锚点
      const visibleAnchors = this.getVisibleAnchors();
      console.log('[TextAnchorSync] 找到可见锚点数量:', visibleAnchors.length);
      
      if (visibleAnchors.length === 0) {
        console.log('[TextAnchorSync] 没有可见锚点，跳过同步');
        return;
      }
      
      // 发送锚点信息
      this.sendAnchors(visibleAnchors);
      
      setTimeout(() => {
        this.state.scrollSource = null;
      }, 100);
    }
    
    /**
     * 获取可见区域的锚点
     */
    getVisibleAnchors() {
      const viewportInfo = this.textExtractor.getViewportInfo();
      console.log('[TextAnchorSync] 视窗信息:', viewportInfo);
      
      const visibleAnchors = [];
      let totalAnchors = 0;
      
      // 从缓存的锚点中找出可见的
      this.anchors.local.forEach((position, text) => {
        totalAnchors++;
        console.log(`[TextAnchorSync] 检查锚点[${totalAnchors}]: "${text.substring(0, 20)}...", 位置:`, position);
        
        if (this.textExtractor.isPositionVisible(position, viewportInfo)) {
          // 计算相对位置（0-1之间）
          const relativePosition = this.textExtractor.getRelativePosition(position, viewportInfo);
          
          visibleAnchors.push({
            text: text,
            position: relativePosition,
            absolute: position
          });
          console.log(`[TextAnchorSync] ✅ 锚点可见，相对位置: ${relativePosition.toFixed(3)}`);
        } else {
          console.log(`[TextAnchorSync] ❌ 锚点不可见`);
        }
      });
      
      console.log(`[TextAnchorSync] 检查了 ${totalAnchors} 个锚点，找到 ${visibleAnchors.length} 个可见锚点`);
      
      // 按位置排序
      visibleAnchors.sort((a, b) => a.position - b.position);
      
      // 只保留最有代表性的锚点
      return this.selectBestAnchors(visibleAnchors);
    }
    
    /**
     * 选择最佳锚点（分布均匀，避免重复）
     */
    selectBestAnchors(anchors) {
      if (anchors.length <= this.config.anchorCount) {
        return anchors;
      }
      
      // 选择分布最均匀的锚点
      const selected = [];
      const step = anchors.length / this.config.anchorCount;
      
      for (let i = 0; i < this.config.anchorCount; i++) {
        const index = Math.floor(i * step);
        selected.push(anchors[index]);
      }
      
      return selected;
    }
    
    /**
     * 发送锚点信息
     */
    sendAnchors(anchors) {
      const message = {
        type: this.config.channelId,
        action: 'sync',
        anchors: anchors.map(a => ({
          text: a.text.substring(0, 100), // 限制发送长度
          position: a.position
        })),
        timestamp: Date.now(),
        source: this.getSourceId()
      };
      
      console.log('[TextAnchorSync] 发送锚点同步消息:', message);
      this.broadcast(message);
      this.log('发送锚点', anchors.length);
    }
    
    /**
     * 广播消息
     */
    broadcast(message) {
      let broadcastCount = 0;
      
      // 1. PostMessage到父窗口和iframe
      if (window.parent !== window) {
        try {
          window.parent.postMessage(message, '*');
          broadcastCount++;
          console.log('[TextAnchorSync] 消息已发送到父窗口');
        } catch (e) {
          console.log('[TextAnchorSync] 发送到父窗口失败:', e.message);
        }
      }
      
      const iframes = document.querySelectorAll('iframe');
      console.log('[TextAnchorSync] 找到iframe数量:', iframes.length);
      
      iframes.forEach((iframe, index) => {
        try {
          iframe.contentWindow?.postMessage(message, '*');
          broadcastCount++;
          console.log(`[TextAnchorSync] 消息已发送到iframe[${index}]:`, iframe.id || iframe.src);
        } catch (e) {
          console.log(`[TextAnchorSync] 发送到iframe[${index}]失败:`, e.message);
        }
      });
      
      // 2. BroadcastChannel
      if (window.BroadcastChannel) {
        try {
          const channel = new BroadcastChannel(this.config.channelId);
          channel.postMessage(message);
          channel.close();
          broadcastCount++;
          console.log('[TextAnchorSync] 消息已通过BroadcastChannel发送');
        } catch (e) {
          console.log('[TextAnchorSync] BroadcastChannel发送失败:', e.message);
        }
      }
      
      // 3. Chrome扩展消息
      if (typeof chrome !== 'undefined' && chrome.runtime?.sendMessage) {
        try {
          chrome.runtime.sendMessage(message);
          broadcastCount++;
          console.log('[TextAnchorSync] 消息已通过Chrome扩展发送');
        } catch (e) {
          console.log('[TextAnchorSync] Chrome扩展发送失败:', e.message);
        }
      }
      
      console.log(`[TextAnchorSync] 消息广播完成，发送目标数量: ${broadcastCount}`);
    }
    
    /**
     * 设置消息监听
     */
    setupMessageListener() {
      // PostMessage监听
      window.addEventListener('message', (event) => {
        if (event.data?.type === this.config.channelId) {
          console.log('[TextAnchorSync] 收到同步消息:', event.data);
          this.handleRemoteMessage(event.data);
        }
      });
      
      // BroadcastChannel监听
      if (window.BroadcastChannel) {
        try {
          const channel = new BroadcastChannel(this.config.channelId);
          channel.addEventListener('message', (event) => {
            this.handleRemoteMessage(event.data);
          });
        } catch (e) {}
      }
      
      // Chrome扩展消息监听
      if (typeof chrome !== 'undefined' && chrome.runtime?.onMessage) {
        chrome.runtime.onMessage.addListener((message) => {
          if (message?.type === this.config.channelId) {
            this.handleRemoteMessage(message);
          }
        });
      }
    }
    
    /**
     * 处理远程消息
     */
    handleRemoteMessage(message) {
      console.log('[TextAnchorSync] 处理远程消息:', message);
      
      // 忽略自己的消息
      if (message.source === this.getSourceId()) {
        console.log('[TextAnchorSync] 忽略自己的消息');
        return;
      }
      
      if (message.action !== 'sync') {
        console.log('[TextAnchorSync] 消息action不是sync，忽略:', message.action);
        return;
      }
      
      console.log('[TextAnchorSync] 开始执行滚动同步，锚点数量:', message.anchors?.length);
      this.state.scrollSource = 'remote';
      
      // 执行滚动同步
      this.syncToAnchors(message.anchors);
      
      setTimeout(() => {
        this.state.scrollSource = null;
      }, this.config.scrollDuration + 100);
    }
    
    /**
     * 同步到锚点位置
     */
    syncToAnchors(remoteAnchors) {
      if (!remoteAnchors || remoteAnchors.length === 0) {
        return;
      }
      
      // 预览区域：直接在DOM中搜索锚点文本
      if (!this.isEditor) {
        this.searchAndScrollToAnchor(remoteAnchors);
        return;
      }
      
      // 编辑器：使用复杂的锚点匹配（保持原逻辑）
      const matches = [];
      
      remoteAnchors.forEach((remoteAnchor, index) => {
        // 在本地锚点中查找
        const localPosition = this.findAnchorPosition(remoteAnchor.text);
        
        if (localPosition !== null) {
          matches.push({
            remote: remoteAnchor.position,
            local: localPosition
          });
        }
      });
      
      if (matches.length === 0) {
        return;
      }
      
      // 根据多个锚点计算目标位置
      const targetPosition = this.calculateTargetPosition(matches);
      
      if (targetPosition !== null) {
        this.scrollToPosition(targetPosition);
      }
    }
    
    /**
     * 预览区域：搜索并滚动到锚点（简化版本）
     */
    searchAndScrollToAnchor(remoteAnchors) {
      console.log('[TextAnchorSync] 📺 预览区域搜索锚点:', remoteAnchors.length);
      
      // 尝试匹配多个锚点以提高精确度
      const matches = [];
      
      remoteAnchors.forEach(anchor => {
        const searchText = anchor.text;
        console.log(`[TextAnchorSync] 搜索文本: "${searchText}"`);
        
        // 在DOM中搜索文本
        const walker = document.createTreeWalker(
          this.container,
          NodeFilter.SHOW_TEXT,
          null,
          false
        );
        
        let node;
        while (node = walker.nextNode()) {
          const nodeText = node.textContent;
          const index = nodeText.indexOf(searchText);
          
          if (index !== -1) {
            console.log('[TextAnchorSync] ✅ 找到匹配文本');
            
            // 获取包含文本的元素
            const element = node.parentElement;
            const rect = element.getBoundingClientRect();
            
            // 计算文档中的绝对位置
            const absoluteTop = window.pageYOffset + rect.top;
            
            matches.push({
              element: element,
              position: absoluteTop,
              remotePosition: anchor.position,
              text: searchText
            });
            break;  // 找到第一个匹配就停止
          }
        }
      });
      
      if (matches.length === 0) {
        console.log('[TextAnchorSync] ❌ 未找到任何匹配的文本');
        return;
      }
      
      console.log(`[TextAnchorSync] 找到 ${matches.length} 个匹配`);
      
      // 使用最佳匹配进行滚动
      const bestMatch = matches[0]; // 使用第一个匹配
      
      // 改进的滚动位置计算
      const containerHeight = this.container.scrollHeight || document.documentElement.scrollHeight;
      const viewportHeight = window.innerHeight;
      
      // 基于远程位置计算目标滚动位置
      let targetScrollTop;
      
      if (bestMatch.remotePosition <= 0.1) {
        // 接近顶部，直接滚动到匹配位置
        targetScrollTop = Math.max(0, bestMatch.position - viewportHeight * 0.1);
      } else if (bestMatch.remotePosition >= 0.9) {
        // 接近底部，调整到视窗底部
        targetScrollTop = Math.max(0, bestMatch.position - viewportHeight * 0.9);
      } else {
        // 中间位置，根据相对位置调整
        const relativeOffset = bestMatch.remotePosition * viewportHeight;
        targetScrollTop = Math.max(0, bestMatch.position - relativeOffset);
      }
      
      // 确保不超过最大滚动范围
      const maxScrollTop = containerHeight - viewportHeight;
      targetScrollTop = Math.min(targetScrollTop, maxScrollTop);
      
      console.log(`[TextAnchorSync] 滚动计算: 
        匹配位置: ${bestMatch.position}px
        远程相对位置: ${(bestMatch.remotePosition * 100).toFixed(1)}%
        目标滚动位置: ${targetScrollTop}px
        容器高度: ${containerHeight}px
        视窗高度: ${viewportHeight}px`);
      
      // 执行滚动
      window.scrollTo({
        top: targetScrollTop,
        behavior: 'smooth'
      });
      
      console.log(`[TextAnchorSync] ✅ 滚动执行完成`);
    }
    
    /**
     * 查找锚点位置 - 简化版
     */
    findAnchorPosition(text) {
      // 直接精确匹配
      if (this.anchors.local.has(text)) {
        return this.anchors.local.get(text);
      }
      
      // 简单的包含匹配（用于跨端兼容）
      for (const [localText, position] of this.anchors.local) {
        if (localText.includes(text) || text.includes(localText)) {
          return position;
        }
      }
      
      return null;
    }
    
    /**
     * 检查文本相似度
     */
    isSimilarText(text1, text2) {
      // 如果一个包含另一个，认为相似
      if (text1.includes(text2) || text2.includes(text1)) {
        return true;
      }
      
      // 计算编辑距离（简化版）
      const minLen = Math.min(text1.length, text2.length);
      const maxLen = Math.max(text1.length, text2.length);
      
      // 长度差异太大，不相似
      if (maxLen > minLen * 1.3) {
        return false;
      }
      
      // 检查开头和结尾是否匹配
      const headMatch = text1.substring(0, 20) === text2.substring(0, 20);
      const tailMatch = text1.substring(text1.length - 20) === text2.substring(text2.length - 20);
      
      return headMatch || tailMatch;
    }
    
    /**
     * 根据多个锚点计算目标位置
     */
    calculateTargetPosition(matches) {
      if (matches.length === 0) return null;
      
      // 如果只有一个匹配，直接使用
      if (matches.length === 1) {
        const match = matches[0];
        const viewportInfo = this.textExtractor.getViewportInfo();
        return this.textExtractor.calculateAbsolutePosition(match.local, match.remote, viewportInfo);
      }
      
      // 多个匹配，使用加权平均
      const viewportInfo = this.textExtractor.getViewportInfo();
      let weightedSum = 0;
      let totalWeight = 0;
      
      matches.forEach((match, index) => {
        // 权重：中间的锚点权重更高
        const weight = 1 + (matches.length - Math.abs(index - matches.length / 2));
        
        const position = this.textExtractor.calculateAbsolutePosition(
          match.local, 
          match.remote, 
          viewportInfo
        );
        
        weightedSum += position * weight;
        totalWeight += weight;
      });
      
      return weightedSum / totalWeight;
    }
    
    /**
     * 滚动到指定位置
     */
    scrollToPosition(position) {
      // 特殊处理：如果是render-container，应该滚动window而不是容器本身
      let scrollTarget;
      if (this.container.tagName === 'TEXTAREA') {
        scrollTarget = this.container;
      } else if (this.container.id === 'render-container' || this.getSourceId() === 'renderer') {
        // 在renderer中，内容在iframe里，应该滚动window
        scrollTarget = window;
      } else if (this.container === document.body) {
        scrollTarget = window;
      } else {
        scrollTarget = this.container;
      }
      
      
      if (this.config.smoothScroll) {
        this.smoothScroll(scrollTarget, position);
      } else {
        if (scrollTarget === window) {
          window.scrollTo(0, position);
        } else {
          scrollTarget.scrollTop = position;
        }
      }
      
    }
    
    /**
     * 平滑滚动
     */
    smoothScroll(target, position) {
      const start = target === window ? window.pageYOffset : target.scrollTop;
      const distance = position - start;
      const startTime = performance.now();
      
      const animation = (currentTime) => {
        const elapsed = currentTime - startTime;
        const progress = Math.min(elapsed / this.config.scrollDuration, 1);
        const easeProgress = this.easeInOutQuad(progress);
        
        const currentPosition = start + distance * easeProgress;
        
        if (target === window) {
          window.scrollTo(0, currentPosition);
        } else {
          target.scrollTop = currentPosition;
        }
        
        if (progress < 1) {
          requestAnimationFrame(animation);
        }
      };
      
      requestAnimationFrame(animation);
    }
    
    /**
     * 缓动函数
     */
    easeInOutQuad(t) {
      return t < 0.5 ? 2 * t * t : -1 + (4 - 2 * t) * t;
    }
    
    /**
     * 设置内容变化监听
     */
    setupMutationObserver() {
      // Textarea监听输入事件
      if (this.container.tagName === 'TEXTAREA') {
        this.container.addEventListener('input', () => {
          // 延迟重建索引
          this.debounceRebuild();
        });
        return;
      }
      
      // DOM容器监听变化
      if (!window.MutationObserver) return;
      
      const observer = new MutationObserver(() => {
        this.debounceRebuild();
      });
      
      observer.observe(this.container, {
        childList: true,
        subtree: true,
        characterData: true
      });
    }
    
    /**
     * 防抖重建索引
     */
    debounceRebuild() {
      clearTimeout(this.rebuildTimer);
      this.rebuildTimer = setTimeout(() => {
        this.buildAnchorIndex();
      }, 500);
    }
    
    /**
     * 获取源ID
     */
    getSourceId() {
      // 编辑器页面的textarea
      if (document.querySelector('#markdown-input')) return 'editor';
      
      // 渲染器页面的容器
      if (document.querySelector('#render-container')) return 'renderer';
      
      // 其他预览容器
      if (document.querySelector('.markdown-body')) return 'preview';
      
      // 默认使用路径
      return location.pathname.split('/').pop() || 'unknown';
    }
    
    /**
     * 调试日志
     */
    log(...args) {
      if (this.config.debug) {
        //console.log('[TextAnchorSync]', ...args);
      }
    }
  }
  
  /**
   * Textarea文本提取器
   */
  class TextareaExtractor {
    constructor(textarea) {
      this.textarea = textarea;
    }
    
    extractAllTexts() {
      const texts = [];
      const lines = this.textarea.value.split('\n');
      
      console.log(`[TextareaExtractor] 开始提取文本，总行数: ${lines.length}`);
      let includedLines = 0;
      let excludedLines = 0;
      
      lines.forEach((line, index) => {
        const trimmed = line.trim();
        if (trimmed.length > 5) {  // 降低从10到5
          texts.push({
            text: line,
            position: {
              line: index,
              offset: lines.slice(0, index).join('\n').length
            }
          });
          includedLines++;
          console.log(`[TextareaExtractor] ✅ 行${index}: "${trimmed.substring(0, 30)}..." (长度: ${trimmed.length})`);
        } else {
          excludedLines++;
          if (trimmed.length > 0) {
            console.log(`[TextareaExtractor] ❌ 行${index}: "${trimmed}" (长度: ${trimmed.length}, 太短)`);
          }
        }
      });
      
      console.log(`[TextareaExtractor] 提取完成: 包含${includedLines}行, 排除${excludedLines}行`);
      return texts;
    }
    
    getViewportInfo() {
      const lineHeight = this.getLineHeight();
      const scrollTop = this.textarea.scrollTop;
      const clientHeight = this.textarea.clientHeight;
      const totalLines = this.textarea.value.split('\n').length;
      
      // 计算可见区域的行号范围
      const startLine = Math.max(0, Math.floor(scrollTop / lineHeight));
      const endLine = Math.min(totalLines - 1, Math.ceil((scrollTop + clientHeight) / lineHeight));
      
      const viewportInfo = {
        startLine: startLine,
        endLine: endLine,
        scrollTop,
        clientHeight,
        lineHeight,
        totalLines
      };
      
      console.log('[TextareaExtractor] 视窗详情:', {
        lineHeight: lineHeight,
        scrollTop: scrollTop,
        clientHeight: clientHeight,
        startLine: startLine,
        endLine: endLine,
        totalLines: totalLines,
        visibleLineCount: endLine - startLine + 1
      });
      
      return viewportInfo;
    }
    
    isPositionVisible(position, viewportInfo) {
      const isVisible = position.line >= viewportInfo.startLine && 
                       position.line <= viewportInfo.endLine;
      
      console.log(`[TextareaExtractor] 位置可见性检查: 行${position.line} (范围: ${viewportInfo.startLine}-${viewportInfo.endLine}) = ${isVisible ? '✅可见' : '❌不可见'}`);
      
      return isVisible;
    }
    
    getRelativePosition(position, viewportInfo) {
      const visibleLines = viewportInfo.endLine - viewportInfo.startLine;
      const relativeLinePos = position.line - viewportInfo.startLine;
      return relativeLinePos / visibleLines;
    }
    
    calculateAbsolutePosition(localPos, remoteRelativePos, viewportInfo) {
      // 计算目标滚动位置：让本地锚点显示在与远程相同的相对位置
      // localPos.line 是锚点所在的行号
      // remoteRelativePos 是远程锚点在视口中的相对位置 (0-1)
      
      // 计算目标滚动位置，使得本地锚点出现在视口的相对位置处
      const targetScrollTop = (localPos.line * viewportInfo.lineHeight) - 
                             (remoteRelativePos * viewportInfo.clientHeight);
      
      // 确保在有效范围内
      const maxScrollTop = Math.max(0, this.textarea.scrollHeight - this.textarea.clientHeight);
      return Math.max(0, Math.min(targetScrollTop, maxScrollTop));
    }
    
    getContext(position) {
      const lines = this.textarea.value.split('\n');
      return {
        before: lines[position.line - 1]?.substring(0, 30) || '',
        after: lines[position.line + 1]?.substring(0, 30) || ''
      };
    }
    
    getLineHeight() {
      const style = window.getComputedStyle(this.textarea);
      const lineHeight = parseFloat(style.lineHeight);
      return isNaN(lineHeight) ? parseFloat(style.fontSize) * 1.2 : lineHeight;
    }
  }
  
  /**
   * DOM文本提取器
   */
  class DOMExtractor {
    constructor(container) {
      this.container = container;
    }
    
    extractAllTexts() {
      const texts = [];
      const walker = document.createTreeWalker(
        this.container,
        NodeFilter.SHOW_TEXT,
        {
          acceptNode: (node) => {
            const text = node.textContent.trim();
            if (text.length > 10 && 
                !node.parentElement.matches('script, style, noscript')) {
              return NodeFilter.FILTER_ACCEPT;
            }
            return NodeFilter.FILTER_REJECT;
          }
        }
      );
      
      let node;
      while (node = walker.nextNode()) {
        const rect = this.getTextNodeRect(node);
        texts.push({
          text: node.textContent,
          position: {
            top: rect.top + window.pageYOffset,
            node: node
          }
        });
      }
      
      return texts;
    }
    
    getTextNodeRect(textNode) {
      const range = document.createRange();
      range.selectNodeContents(textNode);
      return range.getBoundingClientRect();
    }
    
    getViewportInfo() {
      const scrollTop = window.pageYOffset || this.container.scrollTop;
      const clientHeight = window.innerHeight || this.container.clientHeight;
      
      return {
        scrollTop,
        clientHeight,
        scrollBottom: scrollTop + clientHeight
      };
    }
    
    isPositionVisible(position, viewportInfo) {
      return position.top >= viewportInfo.scrollTop && 
             position.top <= viewportInfo.scrollBottom;
    }
    
    getRelativePosition(position, viewportInfo) {
      return (position.top - viewportInfo.scrollTop) / viewportInfo.clientHeight;
    }
    
    calculateAbsolutePosition(localPos, remoteRelativePos, viewportInfo) {
      // 目标：让本地锚点出现在与远程相同的相对位置
      // 在iframe中，需要滚动到的是相对于文档顶部的绝对位置
      const targetScrollTop = localPos.top - (remoteRelativePos * viewportInfo.clientHeight);
      
      // 确保位置在有效范围内
      const maxScroll = Math.max(0, document.documentElement.scrollHeight - window.innerHeight);
      return Math.max(0, Math.min(targetScrollTop, maxScroll));
    }
    
    getContext(position) {
      if (!position.node) return { before: '', after: '' };
      
      const prev = position.node.previousSibling;
      const next = position.node.nextSibling;
      
      return {
        before: prev?.textContent?.substring(0, 30) || '',
        after: next?.textContent?.substring(0, 30) || ''
      };
    }
  }
  
  // 自动初始化
  console.log('[TextAnchorSync] 开始加载模块...');
  const instance = new TextAnchorScrollSync();
  console.log('[TextAnchorSync] 模块实例创建完成，开始初始化...');
  
  // 暴露到全局
  window.TextAnchorScrollSync = instance;
  console.log('[TextAnchorSync] 模块已暴露到全局 window.TextAnchorScrollSync');
  
  // 调试工具
  window.toggleAnchorDebug = function() {
    const current = localStorage.getItem('md2pdf_anchor_debug') === 'true';
    localStorage.setItem('md2pdf_anchor_debug', !current ? 'true' : 'false');
    console.log('锚点调试模式:', !current ? '开启' : '关闭');
    location.reload();
  };
  
  // 添加测试工具
  window.testAnchorSync = function() {
    console.log('=== 测试锚点同步 ===');
    console.log('当前窗口:', window === window.parent ? '主窗口' : 'iframe');
    console.log('容器类型:', instance.container?.tagName);
    console.log('锚点数量:', instance.anchors.local.size);
    console.log('源ID:', instance.getSourceId());
    
    // 如果是textarea且为空，添加测试内容
    if (instance.container?.tagName === 'TEXTAREA' && !instance.container.value) {
      console.log('⚠️ Textarea为空，添加测试内容...');
      instance.container.value = `# 测试文档

这是第一段测试内容，用于验证滚动同步功能。

## 第二章节

这里有更多的内容，包含一些独特的文本用于锚点匹配。

### 子章节

Lorem ipsum dolor sit amet, consectetur adipiscing elit.

${Array(30).fill('这是填充文本，让内容足够长可以滚动。').join('\n\n')}

## 结尾

文档结束。`;
      
      // 触发input事件
      instance.container.dispatchEvent(new Event('input', { bubbles: true }));
      
      // 重建锚点索引
      setTimeout(() => {
        instance.buildAnchorIndex();
        console.log('✅ 已添加测试内容并重建锚点，锚点数量:', instance.anchors.local.size);
      }, 100);
    }
    
    // 手动发送测试消息
    instance.broadcast({
      type: 'md2pdf-text-anchor-sync',
      action: 'sync',
      anchors: [{
        text: '测试锚点',
        position: 0.5
      }],
      timestamp: Date.now(),
      source: 'test'
    });
    
    console.log('测试消息已发送，请查看另一个窗口的控制台');
  };
  
  // 添加手动重建锚点的工具
  window.rebuildAnchors = function() {
    console.log('🔨 手动重建锚点索引...');
    instance.buildAnchorIndex();
    console.log('✅ 重建完成，锚点数量:', instance.anchors.local.size);
    return instance.anchors.local.size;
  };
  
  // 添加调试工具：检查通信状态
  window.checkSyncStatus = function() {
    console.log('=== 滚动同步状态检查 ===');
    console.log('当前窗口:', window === window.parent ? '主窗口' : 'iframe');
    console.log('容器:', instance.container?.tagName, instance.container?.id);
    console.log('锚点数:', instance.anchors.local.size);
    console.log('源ID:', instance.getSourceId());
    
    // 检查iframe
    const iframes = document.querySelectorAll('iframe');
    console.log('找到的iframe数量:', iframes.length);
    iframes.forEach((iframe, i) => {
      console.log(`  iframe[${i}]: id=${iframe.id}, src=${iframe.src}`);
      try {
        if (iframe.contentWindow?.TextAnchorScrollSync) {
          const iframeSync = iframe.contentWindow.TextAnchorScrollSync;
          console.log(`    - 同步模块: ✅ 已加载`);
          console.log(`    - 锚点数: ${iframeSync.anchors.local.size}`);
          console.log(`    - 源ID: ${iframeSync.getSourceId()}`);
        } else {
          console.log(`    - 同步模块: ❌ 未加载`);
        }
      } catch (e) {
        console.log(`    - 无法访问: ${e.message}`);
      }
    });
    
    return '检查完成';
  };
  
  // 添加手动测试滚动监听器的工具
  window.testIframeScroll = function() {
    console.log('=== 测试iframe滚动监听器 ===');
    
    // 获取iframe
    const iframes = document.querySelectorAll('iframe');
    if (iframes.length === 0) {
      console.log('❌ 没有找到iframe');
      return;
    }
    
    const iframe = iframes[0];
    try {
      if (iframe.contentWindow?.TextAnchorScrollSync) {
        const iframeSync = iframe.contentWindow.TextAnchorScrollSync;
        console.log('✅ 找到iframe中的同步模块');
        
        // 测试滚动监听器
        if (iframeSync.testScrollListener) {
          console.log('🧪 调用测试函数...');
          iframeSync.testScrollListener();
        } else {
          console.log('❌ 测试函数不可用');
        }
        
        // 手动触发滚动
        const currentScroll = iframe.contentWindow.pageYOffset;
        console.log(`当前滚动位置: ${currentScroll}`);
        
        // 滚动一点
        iframe.contentWindow.scrollTo(0, currentScroll + 50);
        console.log('📜 手动滚动了50px');
        
      } else {
        console.log('❌ iframe中没有同步模块');
      }
    } catch (e) {
      console.log('❌ 无法访问iframe:', e.message);
    }
  };
  
})();