/**
 * SVG转换工具 - 最简化版本，回滚到基础功能
 */

class SvgConverter {
  constructor(options = {}) {
    this.options = {
      format: 'png',
      quality: 0.95,            // 高质量
      scaleFactor: 3,           // 高倍放大提升Word中图片清晰度
      backgroundColor: '#ffffff',
      // 移除maxWidth和maxHeight限制，保持图片完整性
      ...options
    };
  }

  /**
   * 计算Word文档中图片的最佳尺寸（单页适配，按比例缩放）
   * @param {number} originalWidth - 原始图片宽度(px)
   * @param {number} originalHeight - 原始图片高度(px)
   * @returns {Object} 包含最佳宽度和高度的对象
   */
  calculateOptimalWordImageSize(originalWidth, originalHeight) {
    // A4纸张尺寸和边距设置
    const A4_WIDTH_INCHES = 8.27;
    const A4_HEIGHT_INCHES = 11.7;
    const MARGIN_INCHES = 1;
    
    // 可用空间（英寸）- 保留一些空间给文档其他内容
    const availableWidthInches = A4_WIDTH_INCHES - (MARGIN_INCHES * 2); // 6.27英寸
    const availableHeightInches = A4_HEIGHT_INCHES - (MARGIN_INCHES * 2) - 1.5; // 8.2英寸（减去标题和其他内容空间）
    
    // 转换为像素（假设96 DPI）
    const DPI = 96;
    const availableWidthPx = availableWidthInches * DPI; // 约602px
    const availableHeightPx = availableHeightInches * DPI; // 约787px
    
    let targetWidth, targetHeight;
    
    // 图片必须放在单页中，按比例缩放，不超过页面尺寸
    if (originalWidth <= availableWidthPx && originalHeight <= availableHeightPx) {
      // 图片本身就适合单页，保持原始尺寸
      targetWidth = originalWidth;
      targetHeight = originalHeight;
    } else {
      // 需要缩放以适配单页，选择合适的缩放比例保持比例
      const widthRatio = availableWidthPx / originalWidth;
      const heightRatio = availableHeightPx / originalHeight;
      const ratio = Math.min(widthRatio, heightRatio); // 选择较小的比例确保不超出页面
      
      targetWidth = Math.round(originalWidth * ratio);
      targetHeight = Math.round(originalHeight * ratio);
    }
    
    return {
      width: Math.max(100, targetWidth), // 最小100px
      height: Math.max(100, targetHeight), // 最小100px
      widthInches: targetWidth / DPI,
      heightInches: targetHeight / DPI,
      scale: targetWidth / originalWidth
    };
  }

  /**
   * 将SVG元素转换为图片数据URL
   */
  async convertSvgToImage(svgElement, options = {}) {
    const rect = svgElement.getBoundingClientRect();
    const opts = { ...this.options, ...options };

    try {
      // 如果传入了 width 和 height，使用它们；否则自动计算
      let width, height;

      if (opts.width && opts.height) {
        width = opts.width;
        height = opts.height;
      } else {
        // 获取SVG的基本尺寸
        const dims = this.getSvgDimensions(svgElement, opts);
        width = dims.width;
        height = dims.height;
      }

      // 统一使用htmlToImage转换
      const result = await this.convertSvgWithHtmlToImage(svgElement, opts, width, height);

      return result;

    } catch (error) {
      console.error('SVG转换失败:', error.message);
      throw error;
    }
  }

  /**
   * 获取SVG尺寸（修复版 - 正确处理百分比值）
   */
  getSvgDimensions(svgElement, opts) {
    let width = 0;
    let height = 0;
    
    // 优先使用实际渲染尺寸（最准确的方法）
    const rect = svgElement.getBoundingClientRect();
    
    if (rect.width > 0 && rect.height > 0) {
      width = rect.width;
      height = rect.height;
      return { width, height };
    }
    
    // 从SVG属性获取
    const svgWidth = svgElement.getAttribute('width');
    const svgHeight = svgElement.getAttribute('height');
    
    // 检查是否为有效的像素值
    const isValidPixelValue = (value) => {
      if (!value) return false;
      const str = value.toString().trim();
      // 排除百分比和相对单位
      if (str.includes('%') || str.includes('em') || str.includes('rem') || 
          str.includes('vw') || str.includes('vh') || str.includes('vmin') || 
          str.includes('vmax') || str.includes('auto') || str.includes('inherit')) {
        return false;
      }
      const num = parseFloat(str);
      return !isNaN(num) && num > 0;
    };
    
    if (svgWidth && svgHeight && isValidPixelValue(svgWidth) && isValidPixelValue(svgHeight)) {
      width = parseFloat(svgWidth);
      height = parseFloat(svgHeight);
    }
    
    // 从viewBox获取
    if ((!width || !height) && svgElement.getAttribute('viewBox')) {
      const viewBoxAttr = svgElement.getAttribute('viewBox');
      
      const viewBox = viewBoxAttr.split(' ');
      if (viewBox.length >= 4) {
        const vbWidth = parseFloat(viewBox[2]) - parseFloat(viewBox[0]);
        const vbHeight = parseFloat(viewBox[3]) - parseFloat(viewBox[1]);
        if (vbWidth > 0 && vbHeight > 0) {
          width = vbWidth;
          height = vbHeight;
        }
      }
    }
    
    // 方法3：从style属性获取
    if (!width || !height) {
      const styleWidth = svgElement.style.width;
      const styleHeight = svgElement.style.height;
      
      if (styleWidth && styleHeight) {
        const sw = parseFloat(styleWidth);
        const sh = parseFloat(styleHeight);
        if (sw > 0 && sh > 0) {
          width = sw;
          height = sh;
        }
      }
    }
    
    // 🎯 方法4：临时容器获取实际尺寸
    if (!width || !height || width < 10 || height < 10) {
      
      try {
        const tempDiv = document.createElement('div');
        tempDiv.style.cssText = `
          position: fixed !important;
          top: 0 !important;
          left: -9999px !important;
          visibility: hidden !important;
          pointer-events: none !important;
          z-index: -99999 !important;
          width: auto !important;
          height: auto !important;
          overflow: visible !important;
        `;
        
        const clonedSvg = svgElement.cloneNode(true);
        clonedSvg.style.cssText = '';
        clonedSvg.removeAttribute('width');
        clonedSvg.removeAttribute('height');
        
        tempDiv.appendChild(clonedSvg);
        document.body.appendChild(tempDiv);
        
        // 等待渲染（使用requestAnimationFrame避免阻塞主线程）
        void clonedSvg.offsetHeight;
        
        // 使用更短的等待时间，避免阻塞
        const startTime = Date.now();
        while (Date.now() - startTime < 10) {} // 减少等待时间到10ms
        
        
        const rect = clonedSvg.getBoundingClientRect();
        
        if (rect.width > 0 && rect.height > 0) {
          width = rect.width;
          height = rect.height;
        } else {
        }
        
        document.body.removeChild(tempDiv);
      } catch (error) {
      }
    }
    
    // 方法5：从原始getBoundingClientRect获取
    if (!width || !height || width < 10 || height < 10) {
      const rect = svgElement.getBoundingClientRect();
      if (rect.width > 0 && rect.height > 0) {
        width = rect.width;
        height = rect.height;
      }
    }
    
    // 兜底默认值
    if (!width || width < 10) {
      const viewBox = svgElement.getAttribute('viewBox');
      if (viewBox) {
        const vb = viewBox.split(' ');
        width = vb.length >= 4 ? Math.max(parseFloat(vb[2]) || 800, 400) : 800;
      } else {
        width = 800;
      }
    }
    
    if (!height || height < 10) {
      const viewBox = svgElement.getAttribute('viewBox');
      if (viewBox) {
        const vb = viewBox.split(' ');
        height = vb.length >= 4 ? Math.max(parseFloat(vb[3]) || 600, 300) : 600;
      } else {
        height = Math.round(width * 0.75);
      }
    }
    
    return { width, height };
  }

  /**
   * 使用htmlToImage转换SVG
   */
  async convertSvgWithHtmlToImage(svgElement, opts, width, height) {
    // 检查htmlToImage库是否可用
    if (typeof htmlToImage === 'undefined') {
      console.error('[htmlToImage] ❌ htmlToImage库未加载');
      throw new Error('htmlToImage库未加载');
    }
    
    const tempContainer = document.createElement('div');
    // 增强CSS隔离，避免网页样式污染，使用明确尺寸
    tempContainer.style.cssText = `
      /* 先重置，再设置 */
      all: initial !important;
      /* 重新设置所有必需的属性 */
      position: fixed !important;
      left: -9999px !important;
      top: 0 !important;
      /* 使用明确的容器尺寸 */
      width: ${width + 40}px !important;
      height: ${height + 40}px !important;
      background-color: ${opts.backgroundColor} !important;
      padding: 20px !important;
      z-index: -9999 !important;
      opacity: 1 !important;
      visibility: visible !important;
      pointer-events: none !important;
      display: block !important;
      /* 重设字体和颜色 */
      font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif !important;
      line-height: 1.5 !important;
      color: #000000 !important;
      box-sizing: border-box !important;
    `;
    
    const clonedSvg = svgElement.cloneNode(true);
    
    // 设置明确的SVG尺寸
    clonedSvg.setAttribute('width', width);
    clonedSvg.setAttribute('height', height);
    clonedSvg.style.cssText = `
      width: ${width}px !important;
      height: ${height}px !important;
      display: block !important;
      max-width: none !important;
      max-height: none !important;
    `;
    
    tempContainer.appendChild(clonedSvg);
    document.body.appendChild(tempContainer);
    
    try {
      // 等待SVG渲染完成
      await this.waitForSvgRender(clonedSvg);
      
      // 多次尝试转换，提高成功率
      let lastError = null;
      for (let attempt = 1; attempt <= 3; attempt++) {
        try {
          
          // 确保容器可见性
          tempContainer.style.opacity = '1';
          tempContainer.style.visibility = 'visible';
          
          // 等待DOM更新并检查实际尺寸
          await new Promise(resolve => setTimeout(resolve, 50));
          
          const containerRect = tempContainer.getBoundingClientRect();
          const svgRect = clonedSvg.getBoundingClientRect();
          const captureWidth = width + 40;   
          const captureHeight = height + 40; 
          
          
          
          const canvas = await htmlToImage.toCanvas(tempContainer, {
            pixelRatio: opts.scaleFactor,
            backgroundColor: opts.backgroundColor,
            width: captureWidth,
            height: captureHeight,
            cacheBust: true
          });
          
          
          const mimeType = opts.format === 'jpeg' ? 'image/jpeg' : 'image/png';
          const dataUrl = canvas.toDataURL(mimeType, opts.quality);
          
          // 验证转换结果
          if (dataUrl && dataUrl.length > 100) {
            return dataUrl;
          } else {
            throw new Error('生成的图片数据无效');
          }
          
        } catch (error) {
          lastError = error;
          
          if (attempt < 3) {
            // 等待一段时间后重试
            await new Promise(resolve => setTimeout(resolve, 100 * attempt));
          }
        }
      }
      
      // 所有尝试都失败，抛出最后的错误
      throw lastError || new Error('SVG转换失败');
      
    } finally {
      try {
        if (tempContainer && tempContainer.parentNode) {
          document.body.removeChild(tempContainer);
        }
      } catch (cleanupError) {
        // 静默处理清理错误
      }
    }
  }

  /**
   * 等待SVG渲染完成
   */
  async waitForSvgRender(svgElement, timeout = 2000) {
    return new Promise((resolve) => {
      // 简单等待50ms让DOM更新
      setTimeout(() => {
        resolve();
      }, 50);
    });
  }

  /**
   * 批量转换DOM元素中的SVG（供renderer使用）
   */
  async convertSvgElements(container, options = {}) {
    console.log('[SvgConverter] 开始转换SVG元素');
    
    try {
      const svgElements = container.querySelectorAll('svg');
      
      if (svgElements.length === 0) {
        console.log('[SvgConverter] 没有找到SVG元素');
        return;
      }
      
      console.log(`[SvgConverter] 找到 ${svgElements.length} 个SVG元素`);
      
      // 转换统计
      let successCount = 0;
      let failCount = 0;
      
      for (let i = 0; i < svgElements.length; i++) {
        const svg = svgElements[i];
        
        try {
          const imageDataUrl = await this.convertSvgToImage(svg, options);
          
          if (imageDataUrl) {
            // 创建img元素替换svg
            const img = document.createElement('img');
            img.src = imageDataUrl;
            img.alt = 'Converted SVG';
            img.style.cssText = svg.style.cssText || '';
            
            // 如果需要设置尺寸
            if (options.maxWidth) {
              img.style.maxWidth = `${options.maxWidth}px`;
            }
            if (options.maxHeight) {
              img.style.maxHeight = `${options.maxHeight}px`;
            }
            
            // 替换SVG元素
            svg.parentNode.replaceChild(img, svg);
            successCount++;
            console.log(`[SvgConverter] 成功转换第 ${i + 1} 个SVG`);
          }
        } catch (error) {
          console.error(`[SvgConverter] 转换第 ${i + 1} 个SVG失败:`, error);
          failCount++;
        }
      }
      
      console.log(`[SvgConverter] 转换完成: 成功 ${successCount}, 失败 ${failCount}`);
      
    } catch (error) {
      console.error('[SvgConverter] SVG转换过程出错:', error);
      throw error;
    }
  }

  /**
   * 批量转换HTML中的SVG元素
   */
  async convertSvgElementsInHtml(htmlContent, options = {}) {
    
    // 创建隔离的临时容器
    const tempDiv = document.createElement('div');
    tempDiv.innerHTML = htmlContent;
    tempDiv.style.cssText = `
      position: fixed !important;
      left: -9999px !important;
      top: 0 !important;
      visibility: hidden !important;
      pointer-events: none !important;
      z-index: -99999 !important;
      /* 防止继承网页样式 */
      all: initial !important;
      display: block !important;
      position: fixed !important;
      left: -9999px !important;
    `;
    document.body.appendChild(tempDiv);
    
    try {
      const svgElements = tempDiv.querySelectorAll('svg');
      
      if (svgElements.length === 0) {
        return htmlContent;
      }
      
      // 转换统计
      let successCount = 0;
      let failCount = 0;
      
      for (let i = 0; i < svgElements.length; i++) {
        const svg = svgElements[i];
        
        // 记录SVG信息
        const svgInfo = {
          index: i + 1,
          id: svg.id || 'no-id',
          className: svg.className || 'no-class',
          width: svg.getAttribute('width'),
          height: svg.getAttribute('height'),
          viewBox: svg.getAttribute('viewBox'),
          childrenCount: svg.children.length
        };
        
        try {
          const imageDataUrl = await this.convertSvgToImage(svg, options);
          
          // 验证转换结果
          if (!imageDataUrl || imageDataUrl.length < 100) {
            throw new Error('转换后的图片数据无效');
          }
          
          const img = document.createElement('img');
          img.src = imageDataUrl;
          
          // 检查是否是Word导出模式并应用智能尺寸计算
          if (options.wordExport) {
            // 获取SVG原始尺寸信息
            const originalWidth = svgInfo.width ? parseFloat(svgInfo.width) : 
                                svg.getBoundingClientRect().width || 600;
            const originalHeight = svgInfo.height ? parseFloat(svgInfo.height) : 
                                 svg.getBoundingClientRect().height || 400;
            
            // 计算最佳尺寸（单页适配，按比例缩放）
            const optimalSize = this.calculateOptimalWordImageSize(originalWidth, originalHeight);
            
            
            // 应用计算出的最佳尺寸
            img.style.cssText = `
              width: ${optimalSize.widthInches.toFixed(2)}in !important;
              height: ${optimalSize.heightInches.toFixed(2)}in !important;
              max-width: 6.27in !important;
              max-height: 8.7in !important;
              display: block !important;
              margin: 12pt auto !important;
              object-fit: contain !important;
            `;
            
            // 设置HTML属性（用于html-docx-js处理）
            img.setAttribute('width', Math.round(optimalSize.width));
            img.setAttribute('height', Math.round(optimalSize.height));
            img.setAttribute('data-word-optimized', 'true');
            img.setAttribute('data-original-size', `${originalWidth}x${originalHeight}`);
            img.setAttribute('data-optimal-size', `${optimalSize.width}x${optimalSize.height}`);
            
          } else {
            img.style.cssText = `
              max-width: 100% !important;
              height: auto !important;
              display: block !important;
              margin: 0 auto !important;
            `;
          }
          img.alt = `Mermaid Diagram ${i + 1}`;
          
          // 保留原始SVG的类名或ID，便于调试
          if (svg.id) img.setAttribute('data-original-id', svg.id);
          if (svg.className) img.setAttribute('data-original-class', svg.className);
          
          svg.replaceWith(img);
          successCount++;
          
        } catch (error) {
          failCount++;
          console.error(`[SVG转换] 第 ${i + 1} 个SVG转换失败:`, error);
          console.error('[SVG转换] 失败的SVG信息:', svgInfo);
          
          // 保留原始SVG，但添加错误标记
          svg.setAttribute('data-conversion-failed', 'true');
          svg.setAttribute('data-conversion-error', error.message);
        }
      }
      
      const result = tempDiv.innerHTML;
      return result;
      
    } catch (error) {
      console.error('[SVG转换] 批量转换过程出错:', error);
      return htmlContent;
    } finally {
      document.body.removeChild(tempDiv);
    }
  }
}

// 创建全局实例
if (typeof window !== 'undefined') {
  window.SvgConverter = SvgConverter;
  window.svgConverter = new SvgConverter();
}

if (typeof module !== 'undefined' && module.exports) {
  module.exports = SvgConverter;
}