/**
 * 进度条管理系统 V3 - 极简版本
 * 设计原则：
 * 1. 简单直接，没有复杂的权重计算
 * 2. 进度直接设置，不做动画平滑
 * 3. 确保100%后自动隐藏
 * 创建日期：2025-01
 */

(function() {
  'use strict';

  class SimpleProgressManager {
    constructor() {
      this.container = null;
      this.progressBar = null;
      this.progressText = null;
      this.titleText = null;
      this.hideTimer = null;
      this.cachedTexts = {}; // 缓存国际化文本
    }

    // 获取国际化文本
    async getI18nText(key) {
      // 如果已经缓存过，直接返回
      if (this.cachedTexts[key]) {
        return this.cachedTexts[key];
      }

      // 优先使用 window.MD2PDF_I18N
      if (typeof window !== 'undefined' && window.MD2PDF_I18N) {
        try {
          // 尝试使用异步方法
          if (typeof window.MD2PDF_I18N.getText === 'function') {
            const text = await window.MD2PDF_I18N.getText(key);
            this.cachedTexts[key] = text;
            return text;
          }
          // 如果异步方法不可用，尝试使用同步方法
          else if (typeof window.MD2PDF_I18N.getCachedText === 'function') {
            const text = window.MD2PDF_I18N.getCachedText(key);
            this.cachedTexts[key] = text;
            return text;
          }
        } catch (error) {
          console.warn('Failed to get i18n text:', error);
        }
      }

      // 如果window.MD2PDF_I18N不可用，尝试从Chrome storage或localStorage获取语言设置
      if (typeof window !== 'undefined') {
        let storedLang = null;

        // 优先尝试Chrome storage（扩展环境）
        if (typeof chrome !== 'undefined' && chrome.storage && chrome.storage.sync) {
          try {
            const result = await new Promise((resolve) => {
              chrome.storage.sync.get(['language'], resolve);
            });
            storedLang = result.language;
          } catch (error) {
            console.warn('Failed to read from Chrome storage:', error);
          }
        }

        // 如果Chrome storage没有，尝试localStorage
        if (!storedLang && window.localStorage) {
          try {
            storedLang = localStorage.getItem('language');
          } catch (error) {
            console.warn('Failed to read from localStorage:', error);
          }
        }

        // 如果找到了语言设置
        if (storedLang && storedLang !== 'auto') {
          // 定义简单的翻译映射
          const simpleTranslations = {
            'zh_CN': {
              processing: '处理中',
              convertingToPDF: 'PDF转换中',
              convertingToWord: 'Word转换中',
              completed: '完成',
              failed: '失败',
              pdfGenerated: 'PDF生成成功',
              wordGenerated: 'Word文档生成成功'
            },
            'zh_TW': {
              processing: '處理中',
              convertingToPDF: 'PDF轉換中',
              convertingToWord: 'Word轉換中',
              completed: '完成',
              failed: '失敗',
              pdfGenerated: 'PDF生成成功',
              wordGenerated: 'Word文檔生成成功'
            },
            'ja': {
              processing: '処理中',
              convertingToPDF: 'PDF変換中',
              convertingToWord: 'Word変換中',
              completed: '完了',
              failed: '失敗',
              pdfGenerated: 'PDF生成成功',
              wordGenerated: 'Wordドキュメント生成成功'
            },
            'ko': {
              processing: '처리 중',
              convertingToPDF: 'PDF 변환 중',
              convertingToWord: 'Word 변환 중',
              completed: '완료',
              failed: '실패',
              pdfGenerated: 'PDF 생성 성공',
              wordGenerated: 'Word 문서 생성 성공'
            },
            'es': {
              processing: 'Procesando',
              convertingToPDF: 'Convirtiendo a PDF',
              convertingToWord: 'Convirtiendo a Word',
              completed: 'Completado',
              failed: 'Fallido',
              pdfGenerated: 'PDF generado exitosamente',
              wordGenerated: 'Documento Word generado exitosamente'
            },
            'fr': {
              processing: 'Traitement en cours',
              convertingToPDF: 'Conversion PDF en cours',
              convertingToWord: 'Conversion Word en cours',
              completed: 'Terminé',
              failed: 'Échec',
              pdfGenerated: 'PDF généré avec succès',
              wordGenerated: 'Document Word généré avec succès'
            },
            'de': {
              processing: 'Verarbeitung läuft',
              convertingToPDF: 'PDF-Konvertierung läuft',
              convertingToWord: 'Word-Konvertierung läuft',
              completed: 'Abgeschlossen',
              failed: 'Fehlgeschlagen',
              pdfGenerated: 'PDF erfolgreich erstellt',
              wordGenerated: 'Word-Dokument erfolgreich erstellt'
            },
            'ru': {
              processing: 'Обработка',
              convertingToPDF: 'Преобразование в PDF',
              convertingToWord: 'Преобразование в Word',
              completed: 'Завершено',
              failed: 'Ошибка',
              pdfGenerated: 'PDF успешно создан',
              wordGenerated: 'Документ Word успешно создан'
            },
            'pt': {
              processing: 'Processando',
              convertingToPDF: 'Convertendo para PDF',
              convertingToWord: 'Convertendo para Word',
              completed: 'Concluído',
              failed: 'Falhou',
              pdfGenerated: 'PDF gerado com sucesso',
              wordGenerated: 'Documento Word gerado com sucesso'
            },
            'it': {
              processing: 'Elaborazione in corso',
              convertingToPDF: 'Conversione PDF in corso',
              convertingToWord: 'Conversione Word in corso',
              completed: 'Completato',
              failed: 'Fallito',
              pdfGenerated: 'PDF generato con successo',
              wordGenerated: 'Documento Word generato con successo'
            },
            'ar': {
              processing: 'جاري المعالجة',
              convertingToPDF: 'جاري تحويل PDF',
              convertingToWord: 'جاري تحويل Word',
              completed: 'مكتمل',
              failed: 'فشل',
              pdfGenerated: 'تم إنشاء PDF بنجاح',
              wordGenerated: 'تم إنشاء مستند Word بنجاح'
            },
            'hi': {
              processing: 'प्रसंस्करण में',
              convertingToPDF: 'PDF रूपांतरण में',
              convertingToWord: 'Word रूपांतरण में',
              completed: 'पूर्ण',
              failed: 'असफल',
              pdfGenerated: 'PDF सफलतापूर्वक जेनरेट किया गया',
              wordGenerated: 'Word दस्तावेज़ सफलतापूर्वक जेनरेट किया गया'
            }
          };

          const translations = simpleTranslations[storedLang];
          if (translations && translations[key]) {
            this.cachedTexts[key] = translations[key];
            return translations[key];
          }
        }
      }

      // Fallback 文本
      const fallbackTexts = {
        processing: 'Processing...',
        convertingToPDF: 'Converting to PDF...',
        convertingToWord: 'Converting to Word...',
        completed: 'Completed',
        failed: 'Failed',
        pdfGenerated: 'PDF generated successfully',
        wordGenerated: 'Word document generated successfully'
      };

      const text = fallbackTexts[key] || key;
      this.cachedTexts[key] = text;
      return text;
    }

    // 清除缓存
    clearCache() {
      this.cachedTexts = {};
    }

    // 创建进度条UI
    async createUI() {
      // 如果已存在，先清理
      this.destroy();

      // 获取处理中的文本
      const processingText = await this.getI18nText('processing');

      // 创建容器
      this.container = document.createElement('div');
      this.container.id = 'md2pdf-progress-simple';
      this.container.innerHTML = `
        <div class="progress-card">
          <div class="progress-header">
            <span class="progress-title">${processingText}</span>
            <button class="progress-close">×</button>
          </div>
          <div class="progress-body">
            <div class="progress-bar">
              <div class="progress-fill"></div>
            </div>
            <span class="progress-text">0%</span>
          </div>
        </div>
      `;

      // 注入样式
      if (!document.getElementById('md2pdf-progress-simple-styles')) {
        const styles = document.createElement('style');
        styles.id = 'md2pdf-progress-simple-styles';
        styles.textContent = `
          #md2pdf-progress-simple {
            position: fixed;
            top: 20px;
            right: 20px;
            z-index: 2147483647;
            font-family: system-ui, -apple-system, sans-serif;
            opacity: 0;
            transform: translateX(400px);
            transition: all 0.3s ease;
          }

          #md2pdf-progress-simple.visible {
            opacity: 1;
            transform: translateX(0);
          }

          .progress-card {
            background: white;
            border-radius: 8px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            width: 320px;
            overflow: hidden;
          }

          .progress-header {
            padding: 12px 16px;
            background: #f5f5f5;
            display: flex;
            justify-content: space-between;
            align-items: center;
            border-bottom: 1px solid #e0e0e0;
          }

          .progress-title {
            font-size: 14px;
            font-weight: 500;
            color: #333;
          }

          .progress-close {
            background: none;
            border: none;
            font-size: 20px;
            cursor: pointer;
            color: #666;
            padding: 0;
            width: 24px;
            height: 24px;
            display: flex;
            align-items: center;
            justify-content: center;
          }

          .progress-close:hover {
            color: #333;
          }

          .progress-body {
            padding: 16px;
            display: flex;
            align-items: center;
            gap: 12px;
          }

          .progress-bar {
            height: 6px;
            background: #e0e0e0;
            border-radius: 3px;
            overflow: hidden;
            flex: 1;
          }

          .progress-fill {
            height: 100%;
            background: linear-gradient(90deg, #4CAF50, #45a049);
            border-radius: 3px;
            width: 0;
            transition: width 0.3s ease;
          }

          .progress-text {
            font-size: 12px;
            color: #666;
            flex-shrink: 0;
            min-width: 35px;
            text-align: right;
          }

          .progress-card.success .progress-fill {
            background: linear-gradient(90deg, #4CAF50, #45a049);
          }

          .progress-card.error .progress-fill {
            background: linear-gradient(90deg, #f44336, #da190b);
          }
        `;
        document.head.appendChild(styles);
      }

      // 添加到页面
      document.body.appendChild(this.container);

      // 缓存元素引用
      this.progressBar = this.container.querySelector('.progress-fill');
      this.progressText = this.container.querySelector('.progress-text');
      this.titleText = this.container.querySelector('.progress-title');

      // 绑定关闭按钮
      this.container.querySelector('.progress-close').addEventListener('click', () => {
        this.hide();
      });
    }

    // 显示进度条
    async show(title) {
      if (!this.container) {
        await this.createUI();
      }

      // 清除之前的隐藏定时器
      if (this.hideTimer) {
        clearTimeout(this.hideTimer);
        this.hideTimer = null;
      }

      // 如果没有提供标题，使用默认的处理中
      if (!title) {
        title = await this.getI18nText('processing');
      }

      this.titleText.textContent = title;
      this.setProgress(0);

      // 显示动画
      setTimeout(() => {
        this.container.classList.add('visible');
      }, 10);
    }

    // 设置进度（0-100）
    setProgress(percent) {
      if (!this.container) return;

      // 确保在0-100范围内
      percent = Math.max(0, Math.min(100, percent));

      // 直接设置，不做复杂计算
      this.progressBar.style.width = `${percent}%`;
      this.progressText.textContent = `${Math.round(percent)}%`;

    }

    // 完成
    async complete(message) {

      this.setProgress(100);

      // 如果没有提供消息，使用默认的完成消息
      if (!message) {
        message = await this.getI18nText('completed');
      }

      this.titleText.textContent = '✅ ' + message;
      this.container.querySelector('.progress-card').classList.add('success');

      // 2秒后自动隐藏
      this.hideTimer = setTimeout(() => {
        this.hide();
      }, 2000);
    }

    // 错误
    async error(message) {

      // 如果没有提供消息，使用默认的失败消息
      if (!message) {
        message = await this.getI18nText('failed');
      }

      this.titleText.textContent = '❌ ' + message;
      this.container.querySelector('.progress-card').classList.add('error');

      // 3秒后自动隐藏
      this.hideTimer = setTimeout(() => {
        this.hide();
      }, 3000);
    }

    // 隐藏
    hide() {
      if (!this.container) return;


      this.container.classList.remove('visible');

      // 动画结束后移除
      setTimeout(() => {
        if (this.container && this.container.parentNode) {
          this.container.remove();
          this.container = null;
        }
      }, 300);

      // 清除定时器
      if (this.hideTimer) {
        clearTimeout(this.hideTimer);
        this.hideTimer = null;
      }
    }

    // 销毁
    destroy() {
      this.hide();
      // 移除样式
      const styles = document.getElementById('md2pdf-progress-simple-styles');
      if (styles) {
        styles.remove();
      }
    }

    // === 简化的API，兼容V2接口 ===

    async start(type = 'PDF', title = null) {
      // 如果没有提供标题，根据类型获取默认标题
      if (!title) {
        if (type === 'PDF') {
          title = await this.getI18nText('convertingToPDF');
        } else {
          title = await this.getI18nText('convertingToWord');
        }
      }

      await this.show(title);
      this.currentType = type;
    }

    // 简化的步骤更新 - 直接映射到百分比
    async nextStep(stepId) {

      // PDF步骤映射
      const pdfSteps = {
        'init': 10,
        'parse': 20,
        'render_html': 35,
        'process_math': 50,
        'process_diagrams': 65,
        'process_images': 75,
        'capture': 85,
        'generate_pdf': 95,
        'finalize': 100
      };

      // Word步骤映射
      const wordSteps = {
        'init': 10,
        'parse': 25,
        'render_html': 40,
        'process_math': 60,
        'process_diagrams': 75,
        'build_docx': 90,
        'finalize': 100
      };

      const steps = this.currentType === 'PDF' ? pdfSteps : wordSteps;
      const percent = steps[stepId] || 0;

      this.setProgress(percent);

      // 如果是finalize，不需要等待complete调用，直接完成
      if (stepId === 'finalize') {
        // 稍微延迟以确保100%显示
        setTimeout(async () => {
          let successMessage;
          if (this.currentType === 'PDF') {
            successMessage = await this.getI18nText('pdfGenerated');
          } else {
            successMessage = await this.getI18nText('wordGenerated');
          }
          this.complete(successMessage);
        }, 100);
      }
    }

    // 子进度更新 - 简化处理
    updateSubProgress(stepId, current, total) {
      // 简单地调用nextStep
      this.nextStep(stepId);
    }
  }

  // 创建全局实例
  window.MD2PDF_Progress = new SimpleProgressManager();


})();