/**
 * 预览专用iframe管理器
 * 专门为editor预览功能提供iframe渲染支持，确保预览与PDF输出完全一致
 */
class PreviewIframeManager {
  constructor() {
    this.iframe = null;
    this.isInitialized = false;
    this.pendingRequests = new Map();
    this.requestId = 0;
    this.eventListeners = new Map();
    this.currentContent = null; // 缓存当前内容，避免重复渲染
  }
  
  // 简单的事件发射器
  on(event, listener) {
    if (!this.eventListeners.has(event)) {
      this.eventListeners.set(event, []);
    }
    this.eventListeners.get(event).push(listener);
  }
  
  once(event, listener) {
    const onceWrapper = (...args) => {
      this.off(event, onceWrapper);
      listener(...args);
    };
    this.on(event, onceWrapper);
  }
  
  off(event, listener) {
    if (!this.eventListeners.has(event)) return;
    const listeners = this.eventListeners.get(event);
    const index = listeners.indexOf(listener);
    if (index > -1) {
      listeners.splice(index, 1);
    }
  }
  
  emit(event, ...args) {
    if (!this.eventListeners.has(event)) return;
    const listeners = this.eventListeners.get(event).slice();
    listeners.forEach(listener => {
      try {
        listener(...args);
      } catch (error) {
        console.error(`[PreviewIframe] Event listener error for ${event}:`, error);
      }
    });
  }

  /**
   * 初始化预览iframe
   * @param {HTMLElement} container - 预览容器元素
   */
  async initialize(container) {
    if (this.isInitialized) return;
    
    try {
      // 创建iframe
      this.iframe = document.createElement('iframe');
      this.iframe.id = 'md2pdf-preview-iframe';
      
      // 设置iframe样式
      this.iframe.style.cssText = `
        width: 100%;
        height: 100%;
        border: none;
        background: white;
        display: block;
      `;
      
      // 设置iframe源
      if (typeof chrome !== 'undefined' && chrome.runtime && chrome.runtime.getURL) {
        this.iframe.src = chrome.runtime.getURL('renderer/renderer.html');
        this.extensionOrigin = chrome.runtime.getURL('').slice(0, -1);
        console.log('[PreviewIframe] Chrome扩展环境，iframe.src:', this.iframe.src);
      } else {
        // 开发环境或直接访问时的路径处理
        const currentPath = window.location.pathname;
        if (currentPath.includes('/editor/')) {
          this.iframe.src = '../renderer/renderer.html';
        } else {
          this.iframe.src = './renderer/renderer.html';
        }
        this.extensionOrigin = window.location.origin;
      }
      
      // 设置消息监听器
      this.setupMessageListener();
      
      
      // 添加到容器
      container.appendChild(this.iframe);
      
      // 等待iframe加载完成
      await new Promise((resolve, reject) => {
        let hasResolved = false;
        
        const cleanup = () => {
          clearTimeout(timeout);
          hasResolved = true;
        };
        
        const timeout = setTimeout(() => {
          if (!hasResolved) {
            console.error('[PreviewIframe] 初始化超时，iframe状态:', {
              src: this.iframe.src,
              readyState: this.iframe.readyState,
              contentWindow: !!this.iframe.contentWindow,
              parentNode: !!this.iframe.parentNode,
              contentDocumentReady: this.iframe.contentDocument ? this.iframe.contentDocument.readyState : 'N/A'
            });
            
            // 尝试直接检查iframe内容
            try {
              if (this.iframe.contentWindow && this.iframe.contentWindow.renderer) {
                console.log('[PreviewIframe] 发现renderer对象存在，可能是消息监听问题');
              }
            } catch (e) {
              console.log('[PreviewIframe] 无法访问iframe内容，可能是跨域限制');
            }
            
            reject(new Error('预览iframe初始化超时'));
          }
        }, 30000); // 30秒超时
        
        // 监听渲染器准备就绪
        this.once('rendererReady', () => {
          if (!hasResolved) {
            cleanup();
            this.isInitialized = true;
            resolve();
          }
        });
        
        // iframe加载错误处理
        this.iframe.onerror = (error) => {
          if (!hasResolved) {
            console.error('[PreviewIframe] iframe加载错误:', error);
            cleanup();
            reject(new Error('预览iframe加载失败'));
          }
        };
        
        // iframe加载完成事件
        this.iframe.onload = () => {
          // 立即尝试握手，不等待
          const attemptHandshake = () => {
            try {
              if (this.iframe.contentWindow) {
                const message = {
                  source: 'md2pdf-iframe-manager',
                  action: 'requestReady',
                  data: {}
                };
                this.iframe.contentWindow.postMessage(message, '*');
              }
            } catch (e) {
              console.log('[PreviewIframe] 握手失败:', e.message);
            }
          };
          
          // 立即尝试一次，然后每2秒重试
          attemptHandshake();
          
          // 延迟检查渲染器是否就绪（给脚本执行时间）
          setTimeout(() => {
            if (!hasResolved && !this.isInitialized) {
              attemptHandshake(); // 再次尝试握手
              
              // 再等3秒后最后一次尝试
              setTimeout(() => {
                if (!hasResolved && !this.isInitialized) {
                  attemptHandshake();
                  
                  // 最终检查
                  try {
                    if (this.iframe.contentWindow && this.iframe.contentWindow.renderer) {
                      console.log('[PreviewIframe] 发现renderer对象，强制完成初始化');
                      cleanup();
                      this.isInitialized = true;
                      resolve();
                    }
                  } catch (e) {
                    console.log('[PreviewIframe] 最终检查失败:', e.message);
                  }
                }
              }, 3000);
            }
          }, 2000);
        };
      });
      
      // 滚动监听器已在主消息处理器中处理，无需额外设置
      
      
    } catch (error) {
      console.error('[PreviewIframe] 初始化失败:', error);
      this.cleanup();
      throw error;
    }
  }

  /**
   * 设置消息监听器
   */
  setupMessageListener() {
    this.messageHandler = (event) => {
      if (!this.iframe || event.source !== this.iframe.contentWindow) {
        return;
      }
      
      if (!this.isValidMessageOrigin(event.origin)) {
        console.warn('[PreviewIframe] 消息来源无效:', event.origin);
        return;
      }
      
      // 处理renderer.js发送的消息格式
      if (event.data.source === 'md2pdf-renderer') {
        const { action, data } = event.data;
        // 处理渲染器准备就绪消息
        if (action === 'rendererReady') {
          this.emit('rendererReady');
          return;
        }
        
        // 处理渲染完成消息
        if (action === 'renderComplete') {
          const { requestId, result, error } = data;
          
          const request = this.pendingRequests.get(requestId);
          if (request) {
            this.pendingRequests.delete(requestId);
            if (error) {
              request.reject(new Error(error));
            } else {
              request.resolve(result);
            }
          }
          return;
        }
        
        // 处理滚动同步消息
        if (action === 'scrollSync') {
          const { percentage } = data;
          this.emit('scrollSync', percentage);
          return;
        }
        
        // 发射其他事件
        this.emit(action, data);
        return;
      }
      
      // 处理简化格式的消息（兼容性）
      const { type, requestId, data, error } = event.data;
      
      if (type === 'rendererReady') {
        this.emit('rendererReady');
        return;
      }
      
      if (type === 'renderComplete' && requestId) {
        const request = this.pendingRequests.get(requestId);
        if (request) {
          this.pendingRequests.delete(requestId);
          if (error) {
            request.reject(new Error(error));
          } else {
            request.resolve(data);
          }
        }
        return;
      }
      
      // 发射其他事件
      this.emit(type, data);
    };
    
    window.addEventListener('message', this.messageHandler);
  }

  /**
   * 验证消息来源
   */
  isValidMessageOrigin(origin) {
    if (this.extensionOrigin && origin === this.extensionOrigin) {
      return true;
    }
    
    if (origin === window.location.origin) {
      return true;
    }
    
    return false;
  }

  /**
   * 渲染markdown内容
   * @param {string} markdown - Markdown文本
   * @param {Object} options - 渲染选项
   */
  async renderMarkdown(markdown, options = {}) {
    if (!this.isInitialized) {
      throw new Error('预览iframe未初始化');
    }
    
    // 检查内容是否变化，避免重复渲染
    const contentHash = this.hashContent(markdown, options);
    if (this.currentContent === contentHash) {
      return; // 内容未变化，跳过渲染
    }
    
    const requestId = ++this.requestId;
    
    return new Promise((resolve, reject) => {
      // 添加到待处理请求
      this.pendingRequests.set(requestId, { resolve, reject });
      
      // 设置超时
      const timeout = setTimeout(() => {
        this.pendingRequests.delete(requestId);
        reject(new Error('渲染超时'));
      }, 30000);
      
      // 修改promise处理以清除超时
      const originalResolve = resolve;
      const originalReject = reject;
      
      this.pendingRequests.set(requestId, {
        resolve: (data) => {
          clearTimeout(timeout);
          this.currentContent = contentHash; // 缓存成功渲染的内容
          originalResolve(data);
        },
        reject: (error) => {
          clearTimeout(timeout);
          originalReject(error);
        }
      });
      
      // 发送渲染请求（使用renderer.js期望的消息格式）
      const message = {
        source: 'md2pdf-iframe-manager',
        action: 'renderPreview',
        data: {
          markdown,
          requestId,
          options: {
            ...options,
            preview: true // 标记为预览模式
          }
        }
      };
      
      this.iframe.contentWindow.postMessage(message, '*');
    });
  }

  /**
   * 生成内容哈希，用于检测变化
   */
  hashContent(markdown, options) {
    const content = JSON.stringify({ markdown, options });
    let hash = 0;
    for (let i = 0; i < content.length; i++) {
      const char = content.charCodeAt(i);
      hash = ((hash << 5) - hash) + char;
      hash = hash & hash; // 转换为32位整数
    }
    return hash.toString();
  }

  /**
   * 设置缩放比例
   * @param {number} scale - 缩放比例
   */
  setScale(scale = 1) {
    if (this.iframe) {
      this.iframe.style.transform = `scale(${scale})`;
      this.iframe.style.transformOrigin = 'top left';
      
      // 调整容器尺寸以适应缩放
      if (scale !== 1) {
        this.iframe.style.width = `${100 / scale}%`;
        this.iframe.style.height = `${100 / scale}%`;
      }
    }
  }

  /**
   * 同步滚动到指定百分比
   * @param {number} percentage - 滚动百分比 (0-1)
   */
  syncScroll(percentage) {
    if (!this.isInitialized || !this.iframe) return;
    
    // 发送滚动同步消息到iframe
    const message = {
      source: 'md2pdf-iframe-manager',
      action: 'syncScroll',
      data: { percentage }
    };
    
    this.iframe.contentWindow.postMessage(message, '*');
  }
  
  /**
   * 智能滚动到指定元素（用于智能滚动同步）
   * @param {Object} options - 滚动选项
   * @param {string} options.elementId - 目标元素ID
   * @param {string} options.strategy - 滚动策略
   * @param {boolean} options.smooth - 是否平滑滚动
   * @param {number} options.duration - 滚动持续时间
   * @param {Object} options.blockInfo - 源码块信息
   */
  scrollToElement(options) {
    if (!this.isInitialized || !this.iframe) return;
    
    const message = {
      source: 'md2pdf-iframe-manager',
      action: 'scrollToElement',
      data: options
    };
    
    this.iframe.contentWindow.postMessage(message, '*');
  }

  /**
   * 设置iframe滚动监听
   */
  setupIframeScrollListener() {
    if (!this.isInitialized) return;
    
    // 监听来自iframe的滚动事件
    const scrollHandler = (event) => {
      if (event.data.source === 'md2pdf-renderer' && event.data.action === 'scrollSync') {
        const { percentage } = event.data.data;
        
        // 触发外部的滚动同步回调
        this.emit('scrollSync', percentage);
      }
    };
    
    window.addEventListener('message', scrollHandler);
    
    // 保存引用以便清理
    this.scrollHandler = scrollHandler;
  }

  /**
   * 获取iframe内容高度
   */
  async getContentHeight() {
    if (!this.isInitialized) return 0;
    
    return new Promise((resolve) => {
      const requestId = ++this.requestId;
      
      const timeout = setTimeout(() => {
        this.pendingRequests.delete(requestId);
        resolve(0); // 超时返回0
      }, 1000);
      
      this.pendingRequests.set(requestId, {
        resolve: (data) => {
          clearTimeout(timeout);
          resolve(data.height || 0);
        },
        reject: () => {
          clearTimeout(timeout);
          resolve(0);
        }
      });
      
      this.iframe.contentWindow.postMessage({
        source: 'md2pdf-iframe-manager',
        action: 'getContentHeight',
        data: { requestId }
      }, '*');
    });
  }

  /**
   * 清理资源
   */
  cleanup() {
    if (this.messageHandler) {
      window.removeEventListener('message', this.messageHandler);
      this.messageHandler = null;
    }
    
    if (this.scrollHandler) {
      window.removeEventListener('message', this.scrollHandler);
      this.scrollHandler = null;
    }
    
    if (this.iframe && this.iframe.parentNode) {
      this.iframe.parentNode.removeChild(this.iframe);
    }
    
    this.iframe = null;
    this.isInitialized = false;
    this.pendingRequests.clear();
    this.eventListeners.clear();
    this.currentContent = null;
  }

  /**
   * 检查是否已初始化
   */
  get ready() {
    return this.isInitialized;
  }
}

// 导出类
if (typeof module !== 'undefined' && module.exports) {
  module.exports = PreviewIframeManager;
} else {
  window.PreviewIframeManager = PreviewIframeManager;
}