// 标准化的PDF生成器类

class PDFGenerator {
  constructor(options = {}) {
    // 获取统一的PDF配置
    this.config = StyleConfig.getPDFConfig(options);
    this.loading = null;
    this.container = null;
    this.styleTag = null;
    this.customFontSizes = options.customFontSizes || StyleConfig.getUserFontSizes();
  }

  // 显示加载提示
  showLoading(message = '正在生成PDF，请稍候...') {
    if (typeof showLoading === 'function') {
      this.loading = showLoading();
    }
    return this.loading;
  }

  // 隐藏加载提示
  hideLoading() {
    if (this.loading && this.loading.parentNode) {
      this.loading.remove();
      this.loading = null;
    }
  }

  // 添加页脚到内容
  addFooterToContent(content, options = {}) {
    // 检查是否禁用页脚
    if (options.noFooter === true) {
      return content;
    }
    
    // 获取页脚文本，默认为 md-pdf-word.com
    const footerText = options.footerText || 'md-pdf-word.com';
    
    // 创建页脚HTML
    const footerHtml = `
      <div class="pdf-footer" style="
        margin-top: 30px;
        padding-top: 15px;
        border-top: 1px solid #eee;
        text-align: center;
        font-size: 12px;
        color: #888;
        font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
      ">
        ${footerText}
      </div>
    `;
    
    return content + footerHtml;
  }

  // 创建隔离容器
  createContainer(content, options = {}) {
    // 添加页脚
    const contentWithFooter = this.addFooterToContent(content, options);
    
    const result = StyleConfig.createIsolatedContainer({
      ...options,
      content: contentWithFooter,
      width: options.width || this.config.container.width,
      padding: options.padding || this.config.container.padding
    });
    
    this.container = result.container;
    this.cleanup = result.cleanup;
    
    // 添加到DOM
    document.body.appendChild(this.container);
    
    // 应用样式
    this.styleTag = StyleConfig.applyStylesToIsolatedContainer(
      this.container.id,
      options.theme || 'default',
      this.customFontSizes
    );
    
    return this.container;
  }

  // 等待渲染完成
  async waitForRender() {
    // 等待字体加载
    if (document.fonts && document.fonts.ready) {
      try {
        await Promise.race([
          document.fonts.ready,
          new Promise(resolve => setTimeout(resolve, this.config.timing.fontLoadTimeout))
        ]);
      } catch (e) {
        console.warn('PDFGenerator: 字体加载超时或失败', e);
      }
    }
    
    // 等待渲染
    await new Promise(resolve => setTimeout(resolve, this.config.timing.renderDelay));
    
    // 确保样式完全应用
    await new Promise(resolve => {
      requestAnimationFrame(() => {
        requestAnimationFrame(() => {
          setTimeout(resolve, this.config.timing.styleApplyDelay);
        });
      });
    });
  }

  // 渲染到Canvas
  async renderToCanvas(element = null, options = {}) {
    const targetElement = element || this.container;
    if (!targetElement) {
      throw new Error('PDFGenerator: 没有要渲染的元素');
    }
    
    // 使用统一配置，允许覆盖特定选项
    const canvasOptions = {
      ...this.config.render,
      ...options,
      windowWidth: targetElement.scrollWidth,
      windowHeight: targetElement.scrollHeight
    };
    
    // 如果提供了自定义scale，使用它
    if (options.scale !== undefined) {
      canvasOptions.scale = options.scale;
    }
    
    return await htmlToImage.toCanvas(targetElement, {
      pixelRatio: canvasOptions.scale || 2,
      backgroundColor: canvasOptions.backgroundColor || '#ffffff',
      width: targetElement.scrollWidth,
      height: targetElement.scrollHeight,
      cacheBust: true
    });
  }

  // 从Canvas生成PDF
  async generatePDF(canvas, filename = null) {
    // 使用统一的PDF配置
    const pdfConfig = await PDFConfig.createPDFConfig(canvas, this.config.render.scale);
    const dimensions = await PDFConfig.calculatePDFDimensionsFromCanvas(canvas, this.config.render.scale);
    
    // 创建PDF
    const pdf = new jspdf.jsPDF(pdfConfig);
    
    // 转换图片格式
    const canvasSize = canvas.width * canvas.height * 4;
    const imageFormat = canvasSize > 10 * 1024 * 1024 ? 'JPEG' : 'PNG';
    const quality = imageFormat === 'JPEG' ? 0.92 : 1;
    const imgData = canvas.toDataURL(imageFormat === 'JPEG' ? 'image/jpeg' : 'image/png', quality);
    
    // 添加图片到PDF（添加调试信息）
    console.log(`[PDF-Generator] PDF尺寸信息:`, {
      pdfWidth: dimensions.pdfWidth,
      pdfHeight: dimensions.pdfHeight,
      contentWidth: dimensions.contentWidth,
      contentHeight: dimensions.contentHeight,
      margin: dimensions.margin,
      canvasSize: `${canvas.width}x${canvas.height}`
    });
    
    pdf.addImage(imgData, imageFormat, dimensions.margin, dimensions.margin, dimensions.contentWidth, dimensions.contentHeight);
    
    // 生成文件名
    if (!filename) {
      const now = new Date();
      const timestamp = now.getFullYear() + 
                       String(now.getMonth() + 1).padStart(2, '0') + 
                       String(now.getDate()).padStart(2, '0') + '_' +
                       String(now.getHours()).padStart(2, '0') + 
                       String(now.getMinutes()).padStart(2, '0');
      filename = 'document_' + timestamp + '.pdf';
    }
    
    // 保存PDF
    pdf.save(filename);
    
    return pdf;
  }

  // 从Markdown生成PDF
  async generateFromMarkdown(markdownText, options = {}) {
    if (!markdownText) {
      throw new Error('PDFGenerator: 没有提供Markdown内容');
    }
    
    try {
      // 显示加载提示
      this.showLoading(options.loadingMessage || '正在转换Markdown到PDF...');
      
      // 检查marked库
      if (typeof marked === 'undefined') {
        throw new Error('Markdown解析器不可用');
      }
      
      // 配置marked
      marked.setOptions({
        breaks: true,
        gfm: true,
        headerIds: false,
        mangle: false
      });
      
      // 转换Markdown为HTML
      const htmlContent = marked.parse(markdownText);
      
      // 创建容器并设置内容
      const content = `<div class="markdown-content">${htmlContent}</div>`;
      this.createContainer(content, options);
      
      // 等待渲染
      await this.waitForRender();
      
      // 渲染到Canvas
      const canvas = await this.renderToCanvas();
      
      // 生成PDF
      const pdf = await this.generatePDF(canvas, options.filename);
      
      return pdf;
      
    } catch (error) {
      console.error('PDFGenerator: 生成PDF失败', error);
      throw error;
    } finally {
      this.cleanup();
    }
  }

  // 从HTML元素生成PDF
  async generateFromElement(element, options = {}) {
    if (!element) {
      throw new Error('PDFGenerator: 没有提供要转换的元素');
    }
    
    try {
      // 显示加载提示
      this.showLoading(options.loadingMessage || '正在生成PDF...');
      
      // 检查是否需要创建隔离容器
      if (options.useIsolation !== false) {
        // 克隆元素到隔离容器
        const clonedElement = element.cloneNode(true);
        this.createContainer('', options);
        this.container.appendChild(clonedElement);
        
        // 等待渲染
        await this.waitForRender();
        
        // 渲染到Canvas
        const canvas = await this.renderToCanvas();
        
        // 生成PDF
        const pdf = await this.generatePDF(canvas, options.filename);
        
        return pdf;
      } else {
        // 直接从元素生成（不使用隔离容器）
        const canvas = await this.renderToCanvas(element, options);
        const pdf = await this.generatePDF(canvas, options.filename);
        return pdf;
      }
      
    } catch (error) {
      console.error('PDFGenerator: 生成PDF失败', error);
      throw error;
    } finally {
      this.cleanup();
    }
  }

  // 清理资源
  cleanup() {
    // 隐藏加载提示
    this.hideLoading();
    
    // 调用容器清理函数
    if (this.cleanup && typeof this.cleanup === 'function') {
      this.cleanup();
    }
    
    // 清理样式标签
    if (this.styleTag && this.styleTag.parentNode) {
      this.styleTag.parentNode.removeChild(this.styleTag);
      this.styleTag = null;
    }
    
    // 清理容器
    if (this.container && this.container.parentNode) {
      this.container.parentNode.removeChild(this.container);
      this.container = null;
    }
  }
}

// 导出模块
if (typeof module !== 'undefined' && module.exports) {
  module.exports = PDFGenerator;
}

// 为浏览器环境创建全局对象
if (typeof window !== 'undefined') {
  window.PDFGenerator = PDFGenerator;
}