// PDF配置模块 - 统一管理PDF纸张尺寸和设置

// 标准纸张尺寸定义（单位：mm）
const PAGE_SIZES = {
  'a4': { width: 210, height: 297, label: 'A4' },
  'letter': { width: 215.9, height: 279.4, label: 'Letter' },
  'legal': { width: 215.9, height: 355.6, label: 'Legal' }
};

// 统一的尺寸常量（基于96 DPI）
const DPI = 96;
const MM_TO_PX = DPI / 25.4;  // 1mm = 3.7795px at 96dpi

// 默认设置
const DEFAULT_SETTINGS = {
  pageSize: 'a4',
  margin: 10 // 10mm边距
};

// 获取内容区域的像素宽度（统一尺寸计算）
function getContentAreaWidthPx(pageSize = 'a4', marginMM = 10) {
  const pageDimensions = PAGE_SIZES[pageSize] || PAGE_SIZES.a4;
  const pageWidthPx = pageDimensions.width * MM_TO_PX;  // A4: 794px
  const marginPx = marginMM * MM_TO_PX;  // 10mm = 37.8px
  return Math.round(pageWidthPx - (marginPx * 2));  // A4内容区域: 718px
}

// 获取内容区域的毫米宽度
function getContentAreaWidthMM(pageSize = 'a4', marginMM = 10) {
  const pageDimensions = PAGE_SIZES[pageSize] || PAGE_SIZES.a4;
  return pageDimensions.width - (marginMM * 2);  // A4内容区域: 190mm
}

// 获取用户的PDF设置
async function getPDFSettings() {
  return new Promise((resolve) => {
    // 检查是否在扩展环境中
    if (typeof chrome !== 'undefined' && chrome.storage && chrome.storage.sync) {
      chrome.storage.sync.get(['pdfPageSize'], (result) => {
        resolve({
          pageSize: result.pdfPageSize || DEFAULT_SETTINGS.pageSize,
          margin: DEFAULT_SETTINGS.margin
        });
      });
    } else {
      // 非扩展环境，使用localStorage
      try {
        const savedPageSize = localStorage.getItem('pdfPageSize');
        resolve({
          pageSize: savedPageSize || DEFAULT_SETTINGS.pageSize,
          margin: DEFAULT_SETTINGS.margin
        });
      } catch (e) {
        console.error('Failed to read PDF settings from localStorage:', e);
        resolve(DEFAULT_SETTINGS);
      }
    }
  });
}

// 保存用户的PDF设置
async function savePDFSettings(pageSize) {
  return new Promise((resolve) => {
    if (typeof chrome !== 'undefined' && chrome.storage && chrome.storage.sync) {
      chrome.storage.sync.set({ pdfPageSize: pageSize }, () => {
        resolve();
      });
    } else {
      // 非扩展环境，使用localStorage
      try {
        localStorage.setItem('pdfPageSize', pageSize);
        resolve();
      } catch (e) {
        console.error('Failed to save PDF settings to localStorage:', e);
        resolve(); // 继续执行，不要阻塞
      }
    }
  });
}

// 计算PDF尺寸（优化版 - 统一尺寸计算）
// 规则：宽度固定为纸张宽度，高度最小为纸张高度，内容超出时自动延伸
function calculatePDFDimensions(contentWidthMM, contentHeightMM, pageSize) {
  const pageDimensions = PAGE_SIZES[pageSize] || PAGE_SIZES.a4;
  const margin = DEFAULT_SETTINGS.margin;
  
  // PDF宽度固定为纸张宽度
  const pdfWidth = pageDimensions.width;
  
  // 内容区域宽度（毫米）
  const contentAreaWidth = getContentAreaWidthMM(pageSize, margin);
  
  // 优化的缩放逻辑：如果内容已经适配容器，避免不必要的缩放
  let finalContentWidth = contentWidthMM;
  let finalContentHeight = contentHeightMM;
  
  // 允许的误差范围（1mm），用于判断内容是否已经适配
  const tolerance = 1.0;
  
  if (contentWidthMM > contentAreaWidth + tolerance) {
    // 只有明显超出时才缩放
    const scale = contentAreaWidth / contentWidthMM;
    finalContentWidth = contentAreaWidth;
    finalContentHeight = contentHeightMM * scale;
    console.log(`[PDF-Config] 内容宽度缩放: ${contentWidthMM}mm -> ${finalContentWidth}mm (缩放比例: ${(scale * 100).toFixed(1)}%)`);
  } else if (Math.abs(contentWidthMM - contentAreaWidth) <= tolerance) {
    // 如果内容宽度已经接近内容区域宽度，直接使用内容区域宽度
    finalContentWidth = contentAreaWidth;
    finalContentHeight = contentHeightMM;
    console.log(`[PDF-Config] 内容宽度已适配: ${finalContentWidth}mm (无缩放)`);
  } else {
    // 内容宽度小于内容区域，保持原宽度
    console.log(`[PDF-Config] 内容宽度保持原样: ${finalContentWidth}mm`);
  }
  
  // PDF高度：最小为纸张高度，内容超出时自动延伸
  const requiredHeight = finalContentHeight + (margin * 2);
  const pdfHeight = Math.max(pageDimensions.height, requiredHeight);
  
  return {
    pdfWidth,
    pdfHeight,
    contentWidth: finalContentWidth,
    contentHeight: finalContentHeight,
    margin,
    pageSize: pageDimensions
  };
}

// 根据canvas计算PDF尺寸
function calculatePDFDimensionsFromCanvas(canvas, scale = 1) {
  // 将canvas尺寸转换为mm（假设96 DPI）
  const contentWidthMM = (canvas.width / scale) * 25.4 / 96;
  const contentHeightMM = (canvas.height / scale) * 25.4 / 96;
  
  return getPDFSettings().then(settings => {
    return calculatePDFDimensions(contentWidthMM, contentHeightMM, settings.pageSize);
  });
}

// 创建统一的PDF配置
async function createPDFConfig(canvas, scale = 1) {
  const settings = await getPDFSettings();
  const dimensions = await calculatePDFDimensionsFromCanvas(canvas, scale);
  
  return {
    orientation: 'portrait', // 始终使用portrait，通过自定义尺寸实现效果
    unit: 'mm',
    format: [dimensions.pdfWidth, dimensions.pdfHeight],
    compress: true
  };
}

// 生成PDF文件名
function generatePDFFilename(content) {
  // 如果content是自定义文件名，直接使用
  if (content && typeof content === 'string' && content.trim() && !content.includes('\n') && content.length < 100) {
    const filename = content.trim().replace(/[<>:"/\\|?*]/g, '').substring(0, 50);
    if (filename) {
      return filename.endsWith('.pdf') ? filename : filename + '.pdf';
    }
  }
  
  // 否则从内容中提取标题或使用时间戳
  let filename = '';
  
  if (content && typeof content === 'string') {
    // 尝试从内容中提取第一个标题
    const titleMatch = content.match(/^#\s+(.+)$/m) || content.match(/^(.+)$/m);
    if (titleMatch && titleMatch[1]) {
      filename = titleMatch[1].trim()
        .replace(/[<>:"/\\|?*]/g, '')  // 移除非法字符
        .replace(/\s+/g, '_')          // 空格替换为下划线
        .substring(0, 50);             // 限制长度
    }
  }
  
  // 如果没有找到合适的标题，使用默认名称
  if (!filename) {
    const now = new Date();
    const timestamp = now.getFullYear() + 
                     String(now.getMonth() + 1).padStart(2, '0') + 
                     String(now.getDate()).padStart(2, '0') + '_' +
                     String(now.getHours()).padStart(2, '0') + 
                     String(now.getMinutes()).padStart(2, '0');
    filename = 'markdown_' + timestamp;
  }
  
  return filename.endsWith('.pdf') ? filename : filename + '.pdf';
}

// 导出模块
if (typeof module !== 'undefined' && module.exports) {
  module.exports = {
    PAGE_SIZES,
    DEFAULT_SETTINGS,
    DPI,
    MM_TO_PX,
    getContentAreaWidthPx,
    getContentAreaWidthMM,
    getPDFSettings,
    savePDFSettings,
    calculatePDFDimensions,
    calculatePDFDimensionsFromCanvas,
    createPDFConfig,
    generatePDFFilename
  };
}

// 为浏览器环境创建全局对象
if (typeof window !== 'undefined') {
  window.PDFConfig = {
    PAGE_SIZES,
    DEFAULT_SETTINGS,
    DPI,
    MM_TO_PX,
    getContentAreaWidthPx,
    getContentAreaWidthMM,
    getPDFSettings,
    savePDFSettings,
    calculatePDFDimensions,
    calculatePDFDimensionsFromCanvas,
    createPDFConfig,
    generatePDFFilename
  };
}