/**
 * 智能分页核心模块 - 极简接口设计
 * 文件：lib/pagination/smart-pagination.js
 * 
 * 设计原则：
 * - 单一入口函数：SmartPagination.process(canvas)
 * - 配置自动从localStorage获取
 * - 返回格式标准化
 * - 错误处理优雅降级
 * - 接口稳定向后兼容
 */

class SmartPagination {
  /**
   * 智能分页核心接口 - 唯一对外方法
   * @param {Canvas} canvas - 待分页的Canvas对象
   * @param {Object} options - 可选参数（保持向后兼容）
   * @param {number} options.pageHeight - 动态页面高度（像素），优先级高于localStorage
   * @returns {Object} 标准分页结果
   */
  static process(canvas, options = {}) {
    try {
      // 输入验证
      if (!canvas || typeof canvas !== 'object' || !canvas.getContext) {
        return {
          success: false,
          error: 'Invalid canvas input: Canvas object required',
          pages: [],
          totalPages: 0,
          fallbackPages: this._createFallbackPages(canvas)
        };
      }

      // 强制返回单页结果（单页模式）
      console.log('[SmartPagination] 单页模式：跳过所有分页逻辑');
      return this._createSinglePageResult(canvas);

      // 以下是原有的分页逻辑，现在已经禁用
      /*
      // 从localStorage获取配置
      let config = this._loadConfiguration();

      // 如果传入了动态页面高度，使用传入的值（优先级更高）
      if (options.pageHeight && typeof options.pageHeight === 'number' && options.pageHeight > 0) {
        config = {
          ...config,
          pageHeight: options.pageHeight
        };
        console.log('[SmartPagination] 使用动态页面高度:', options.pageHeight);
      }

      // 如果分页未启用，返回单页结果
      if (!config.enabled) {
        return this._createSinglePageResult(canvas);
      }

      // 执行智能分页处理
      const processor = new PaginationProcessor(config);
      return processor.processCanvas(canvas);
      */

    } catch (error) {
      // 优雅错误处理，提供fallback
      console.error('[SmartPagination] 处理错误:', error.message);
      return {
        success: false,
        error: `Canvas processing failed: ${error.message}`,
        pages: [],
        totalPages: 0,
        fallbackPages: this._createFallbackPages(canvas)
      };
    }
  }

  /**
   * 从localStorage加载配置
   * @private
   */
  static _loadConfiguration() {
    const getItem = (key, defaultValue) => {
      try {
        const value = localStorage?.getItem(key);
        return value !== null ? value : defaultValue;
      } catch (e) {
        return defaultValue;
      }
    };

    return {
      enabled: getItem('md2pdf_pagination_enabled', 'false') === 'true',  // 默认禁用分页
      strategy: getItem('md2pdf_pagination_strategy', 'balanced'),
      avoidOrphans: getItem('md2pdf_avoid_orphans', 'false') === 'true',  // 默认禁用孤行避免
      maxBacktrack: parseInt(getItem('md2pdf_max_backtrack', '100'), 10),
      pageHeight: parseInt(getItem('md2pdf_page_height', '999999'), 10)  // 设置超大页面高度
    };
  }

  /**
   * 创建单页结果（当分页未启用时）
   * @private
   */
  static _createSinglePageResult(canvas) {
    if (!canvas || !canvas.height) {
      return {
        success: false,
        error: 'Invalid canvas for single page',
        pages: [],
        totalPages: 0
      };
    }

    return {
      success: true,
      pages: [{
        pageNumber: 1,
        startY: 0,
        endY: canvas.height,
        height: canvas.height
      }],
      totalPages: 1
    };
  }

  /**
   * 创建备用分页方案（错误时使用）
   * @private
   */
  static _createFallbackPages(canvas) {
    if (!canvas || typeof canvas.height !== 'number') {
      return [];
    }

    const defaultPageHeight = 800;
    const totalHeight = canvas.height;
    const pageCount = Math.ceil(totalHeight / defaultPageHeight);
    const fallbackPages = [];

    for (let i = 0; i < pageCount; i++) {
      const startY = i * defaultPageHeight;
      const endY = Math.min(startY + defaultPageHeight, totalHeight);
      
      fallbackPages.push({
        pageNumber: i + 1,
        startY: startY,
        endY: endY,
        height: endY - startY,
        isFallback: true
      });
    }

    return fallbackPages;
  }
}

/**
 * 内部分页处理器类
 * @private
 */
class PaginationProcessor {
  constructor(config) {
    this.config = config;
  }

  /**
   * 处理Canvas分页
   */
  processCanvas(canvas) {
    try {
      // 检测Canvas上下文是否可用 - 使用willReadFrequently优化性能
      const ctx = canvas.getContext('2d', { willReadFrequently: true });
      if (!ctx) {
        throw new Error('Canvas context not available');
      }

      // 缓存context以避免重复获取
      this.cachedContext = ctx;
      this.cachedCanvas = canvas;

      const { strategy, pageHeight, avoidOrphans, maxBacktrack } = this.config;

      // 检测实际内容高度，避免空白区域
      const actualContentHeight = this._detectActualContentHeight(canvas);
      const totalHeight = Math.min(actualContentHeight, canvas.height);
      const canvasWidth = canvas.width;

      // 如果实际内容高度很小，可能是空文档
      if (totalHeight < 50) {
        console.warn('[SmartPagination] 检测到内容过小，可能是空文档');
        return this._createEmptyResult();
      }
      
      // 如果内容高度小于一页，返回单页
      if (totalHeight <= pageHeight) {
        return {
          success: true,
          pages: [{
            pageNumber: 1,
            startY: 0,
            endY: totalHeight,
            height: totalHeight
          }],
          totalPages: 1,
          actualContentHeight: totalHeight
        };
      }

      // 根据策略执行分页
      let pages;
      switch (strategy) {
        case 'conservative':
          pages = this._conservativePagination(canvas, pageHeight, maxBacktrack);
          break;
        case 'compact':
          pages = this._compactPagination(canvas, pageHeight);
          break;
        case 'balanced':
        default:
          pages = this._balancedPagination(canvas, pageHeight, maxBacktrack);
          break;
      }

      // 应用孤行避免规则
      if (avoidOrphans && pages.length > 1) {
        const orphanResult = this._applyOrphanAvoidance(pages, canvas);
        // 如果孤行避免返回的是特殊格式（如空文档），使用其结果
        if (orphanResult && typeof orphanResult === 'object' && 'pages' in orphanResult) {
          return orphanResult;
        }
        pages = orphanResult;
      }

      return {
        success: true,
        pages: pages,
        totalPages: pages.length,
        actualContentHeight: totalHeight
      };

    } catch (error) {
      throw new Error(`PaginationProcessor error: ${error.message}`);
    }
  }

  /**
   * 保守分页策略 - 避免切分任何内容
   */
  _conservativePagination(canvas, pageHeight, maxBacktrack) {
    const pages = [];
    const totalHeight = canvas.height;
    let currentY = 0;
    let pageNumber = 1;
    const maxIterations = Math.ceil(totalHeight / pageHeight) * 2; // 防止无限循环
    let iterations = 0;

    while (currentY < totalHeight && iterations < maxIterations) {
      iterations++;
      let targetEndY = Math.min(currentY + pageHeight, totalHeight);

      // 查找安全的分页点
      const safeEndY = this._findSafeBreakPoint(canvas, currentY, targetEndY, maxBacktrack);

      const currentPageHeight = safeEndY - currentY;

      // 只添加高度大于0的页面
      if (currentPageHeight > 0) {
        pages.push({
          pageNumber: pageNumber++,
          startY: currentY,
          endY: safeEndY,
          height: pageHeight,
          avoidsCutting: true
        });
      }

      currentY = safeEndY;

      // 如果页面高度为0，增加Y位置避免无限循环
      if (pageHeight <= 0) {
        currentY += Math.min(10, totalHeight - currentY);
      }
    }

    return pages;
  }

  /**
   * 紧凑分页策略 - 最大化页面利用率
   */
  _compactPagination(canvas, pageHeight) {
    const pages = [];
    const totalHeight = canvas.height;
    const pageCount = Math.ceil(totalHeight / pageHeight);

    for (let i = 0; i < pageCount; i++) {
      const startY = i * pageHeight;
      const endY = Math.min(startY + pageHeight, totalHeight);
      const height = endY - startY;

      // 只添加有效的页面，且高度必须大于0
      if (height > 0 && height >= 20) { // 最小高度阈值
        pages.push({
          pageNumber: i + 1,
          startY: startY,
          endY: endY,
          height: height
        });
      }
    }

    // 如果最后一页高度太小，尝试合并到前一页
    if (pages.length > 1) {
      const lastPage = pages[pages.length - 1];
      if (lastPage.height > 0 && lastPage.height < pageHeight * 0.1) { // 小于10%的页面
        const prevPage = pages[pages.length - 2];
        prevPage.endY = lastPage.endY;
        prevPage.height = prevPage.endY - prevPage.startY;
        pages.pop(); // 移除最后一页
      }
    }

    return pages;
  }

  /**
   * 平衡分页策略 - 在避免切分和页面利用率间平衡
   */
  _balancedPagination(canvas, pageHeight, maxBacktrack) {
    const pages = [];
    const totalHeight = canvas.height;
    let currentY = 0;
    let pageNumber = 1;
    const maxIterations = Math.ceil(totalHeight / pageHeight) * 2; // 防止无限循环
    let iterations = 0;

    while (currentY < totalHeight && iterations < maxIterations) {
      iterations++;
      let targetEndY = Math.min(currentY + pageHeight, totalHeight);

      // 尝试找到更好的分页点，但限制回退距离
      const betterEndY = this._findBetterBreakPoint(canvas, currentY, targetEndY, maxBacktrack);

      const currentPageHeight = betterEndY - currentY;

      // 只添加高度大于0的页面
      if (currentPageHeight > 0) {
        pages.push({
          pageNumber: pageNumber++,
          startY: currentY,
          endY: betterEndY,
          height: pageHeight
        });
      }

      currentY = betterEndY;

      // 如果页面高度为0，增加Y位置避免无限循环
      if (pageHeight <= 0) {
        currentY += Math.min(10, totalHeight - currentY);
      }
    }

    return pages;
  }

  /**
   * 查找安全的分页点（保守策略用）
   */
  _findSafeBreakPoint(canvas, startY, targetEndY, maxBacktrack) {
    // 向上搜索空白行，避开已知的内容区域
    for (let searchY = targetEndY; searchY >= Math.max(startY + 50, targetEndY - maxBacktrack); searchY -= 5) {
      if (this._isRowEmpty(canvas, searchY) && !this._isInContentArea(searchY)) {
        return searchY;
      }
    }
    
    // 如果在内容区域，退到区域前面
    if (this._isInContentArea(targetEndY)) {
      // 找到内容区域的开始位置，退到前面
      for (let searchY = targetEndY; searchY >= startY + 50; searchY -= 5) {
        if (!this._isInContentArea(searchY)) {
          return searchY;
        }
      }
    }
    
    // 最后的保护：确保不在明显内容中间
    return Math.max(startY + 50, targetEndY - 20);
  }

  /**
   * 检测位置是否在已知内容区域
   */
  _isInContentArea(y) {
    // 在实际使用中，我们不知道内容区域的具体位置
    // 所以这个方法应该返回false，让其他逻辑判断
    return false;
  }

  /**
   * 查找更好的分页点（平衡策略用）
   */
  _findBetterBreakPoint(canvas, startY, targetEndY, maxBacktrack) {
    // 简化实现：在小范围内寻找更好的分页点
    const searchRange = Math.min(maxBacktrack, 50);
    
    for (let searchY = targetEndY; searchY >= targetEndY - searchRange; searchY -= 3) {
      if (this._isRowEmpty(canvas, searchY)) {
        return searchY;
      }
    }
    
    return targetEndY; // 找不到更好的点时使用目标位置
  }

  /**
   * 检测Canvas行是否为空白
   */
  _isRowEmpty(canvas, y, tolerance = 10) {
    try {
      // 使用缓存的context，如果没有则获取（带优化标志）
      const ctx = this.cachedContext || canvas.getContext('2d', { willReadFrequently: true });
      
      // 性能优化：采样检测而不是检查整行
      const sampleStep = 10; // 每10个像素采样一次
      const sampleWidth = Math.min(canvas.width, 200); // 最多检查200像素宽度
      const imageData = ctx.getImageData(0, y, sampleWidth, 1);
      const data = imageData.data;
      
      // 简化检测：采样检查像素
      let whitePixels = 0;
      let sampledPixels = 0;
      
      for (let i = 0; i < data.length; i += sampleStep * 4) {
        const r = data[i];
        const g = data[i + 1];
        const b = data[i + 2];
        
        sampledPixels++;
        if (r > 240 && g > 240 && b > 240) {
          whitePixels++;
        }
      }
      
      const whiteRatio = sampledPixels > 0 ? whitePixels / sampledPixels : 1;
      return whiteRatio > 0.9; // 90%以上为白色认为是空白行
      
    } catch (error) {
      return false; // 检测失败时保守处理
    }
  }

  /**
   * 应用孤行避免规则
   */
  _applyOrphanAvoidance(pages, canvas) {
    // 首先过滤掉高度为0的页面
    const filteredPages = pages.filter(page => {
      if (page.height <= 0) {
        console.warn('[SmartPagination] 过滤高度为0的页面:', page);
        return false;
      }
      return true;
    });

    // 如果过滤后没有页面了，返回空文档结果
    if (filteredPages.length === 0) {
      return {
        success: true,
        pages: [],
        totalPages: 0,
        isEmpty: true
      };
    }

    // 如果只有一个页面且高度有效，直接返回
    if (filteredPages.length === 1) {
      return filteredPages;
    }

    const minPageHeight = 100;
    const lastPageThreshold = 0.15; // 最后一页如果小于15%高度，考虑合并
    const adjustedPages = [];

    for (let i = 0; i < filteredPages.length; i++) {
      const page = filteredPages[i];

      // 如果页面太小，尝试与前一页合并
      if (page.height < minPageHeight && adjustedPages.length > 0) {
        const lastPage = adjustedPages[adjustedPages.length - 1];

        // 检查合并后是否超出页面高度限制
        const combinedHeight = lastPage.height + page.height;
        const maxHeight = this.config.pageHeight || 800;

        // 只有当页面高度大于0时才合并
        if (page.height > 0 && combinedHeight <= maxHeight) {
          // 可以合并
          lastPage.endY = page.endY;
          lastPage.height = combinedHeight;
          console.log('[SmartPagination] 合并小页面到前一页:', {
            smallPageHeight: page.height,
            newHeight: combinedHeight
          });
        } else {
          // 不能合并，但页面太小，可能需要调整分页点
          const betterBreakPoint = this._findBetterBreakPointForSmallPage(canvas, page, minPageHeight);
          if (betterBreakPoint && betterBreakPoint.height > 0) {
            adjustedPages.push(betterBreakPoint);
          } else {
            // 实在无法处理，保留原页面（但确保高度大于0）
            if (page.height > 0) {
              adjustedPages.push({
                ...page,
                pageNumber: adjustedPages.length + 1
              });
            }
          }
        }
      } else {
        // 只添加有效的页面
        if (page.height > 0) {
          adjustedPages.push({
            ...page,
            pageNumber: adjustedPages.length + 1
          });
        }
      }
    }

    // 处理最后一页特殊情况
    if (adjustedPages.length > 1) {
      const lastPage = adjustedPages[adjustedPages.length - 1];
      const avgPageHeight = adjustedPages.reduce((sum, p) => sum + p.height, 0) / adjustedPages.length;

      // 如果最后一页高度小于平均高度的15%，尝试合并到前一页
      if (lastPage.height < avgPageHeight * lastPageThreshold && lastPage.height > 0) {
        const secondLastPage = adjustedPages[adjustedPages.length - 2];

        // 检查是否可以合并
        const combinedHeight = secondLastPage.height + lastPage.height;
        const maxHeight = this.config.pageHeight || 800;

        if (combinedHeight <= maxHeight * 1.1) { // 允许10%的超出
          secondLastPage.endY = lastPage.endY;
          secondLastPage.height = combinedHeight;
          adjustedPages.pop(); // 移除最后一页
          console.log('[SmartPagination] 合并最后一页到前一页:', {
            lastPageHeight: lastPage.height,
            avgHeight: avgPageHeight,
            newHeight: combinedHeight
          });
        }
      }
    }

    // 验证并移除无效页面
    const validPages = this._validatePages(adjustedPages, canvas);

    // 如果验证后没有有效页面，返回空文档结果
    if (validPages.length === 0) {
      console.warn('[SmartPagination] 所有页面都被验证过滤，返回空文档');
      return {
        success: true,
        pages: [],
        totalPages: 0,
        isEmpty: true
      };
    }

    return validPages;
  }

  /**
   * 为小页面寻找更好的分页点
   */
  _findBetterBreakPointForSmallPage(canvas, page, minHeight) {
    try {
      // 从页面结束位置向上查找最近的合适分页点
      const startY = page.startY;
      const endY = page.endY;
      const searchRange = Math.min(endY - startY - minHeight, 100);

      if (searchRange <= 0) {
        return null;
      }

      // 查找空白行
      for (let y = endY; y >= startY + minHeight; y -= 5) {
        if (this._isRowEmpty(canvas, y)) {
          return {
            ...page,
            startY: startY,
            endY: y,
            height: y - startY,
            pageNumber: 0 // 稍后会被重新分配
          };
        }
      }

      // 找不到空白行，返回null
      return null;

    } catch (error) {
      console.error('[SmartPagination] 寻找更好分页点失败:', error);
      return null;
    }
  }

  /**
   * 验证页面有效性并移除无效页面
   */
  _validatePages(pages, canvas) {
    if (!pages || pages.length === 0) {
      return [];
    }

    const validPages = [];
    const minPageHeight = 50; // 提高最小页面高度阈值，从20到50像素
    const canvasHeight = canvas.height;

    for (let i = 0; i < pages.length; i++) {
      const page = pages[i];

      // 检查页面是否有效 - 添加更严格的验证
      if (page.height >= minPageHeight &&
          page.startY >= 0 &&
          page.startY < canvasHeight &&
          page.endY <= canvasHeight &&
          page.startY < page.endY &&
          page.height > 0) { // 确保高度大于0

        // 确保页面不超出Canvas范围
        const adjustedPage = {
          ...page,
          startY: Math.max(0, page.startY),
          endY: Math.min(canvasHeight, page.endY),
          height: Math.min(page.endY - page.startY, canvasHeight - page.startY)
        };

        // 再次检查调整后的高度
        if (adjustedPage.height >= minPageHeight) {
          // 检查是否与前一个页面重复
          const lastPage = validPages[validPages.length - 1];
          if (!lastPage ||
              Math.abs(adjustedPage.startY - lastPage.startY) > 5 ||
              Math.abs(adjustedPage.height - lastPage.height) > 5) {
            validPages.push(adjustedPage);
          }
        }
      } else {
        console.warn('[SmartPagination] 过滤无效页面:', {
          height: page.height,
          startY: page.startY,
          endY: page.endY,
          canvasHeight: canvasHeight,
          reason: page.height <= 0 ? '高度为0或负数' :
                  page.height < minPageHeight ? '高度过小' :
                  page.startY >= page.endY ? '起始位置大于结束位置' : '其他原因'
        });
      }
    }

    // 重新分配页码
    validPages.forEach((page, index) => {
      page.pageNumber = index + 1;
    });

    // 如果所有页面都被过滤掉了，返回空数组而不是无效页面
    if (validPages.length === 0 && pages.length > 0) {
      console.warn('[SmartPagination] 所有页面都被过滤，可能是内容检测问题');
    }

    return validPages;
  }

  /**
   * 检测Canvas中的实际内容高度，避免空白区域
   * @param {Canvas} canvas - Canvas对象
   * @returns {number} 实际内容高度（像素）
   * @private
   */
  _detectActualContentHeight(canvas) {
    try {
      const ctx = canvas.getContext('2d', { willReadFrequently: true });
      const width = canvas.width;
      const height = canvas.height;

      // 从底部向上扫描，找到最后一个非空白行
      let lastContentY = height;
      const scanRange = Math.min(500, height); // 增加扫描范围到500像素

      for (let y = height - 1; y >= Math.max(0, height - scanRange); y -= 5) {
        const imageData = ctx.getImageData(0, y, width, 1);
        const data = imageData.data;
        let hasContent = false;
        let contentPixels = 0;
        let totalSampled = 0;

        // 采样检查，提高性能
        const sampleStep = 20; // 每20个像素采样一次
        for (let i = 0; i < data.length; i += sampleStep * 4) {
          const r = data[i];
          const g = data[i + 1];
          const b = data[i + 2];

          totalSampled++;
          // 降低阈值，从250降到240，更容易检测到浅色内容
          if (r < 240 || g < 240 || b < 240) {
            hasContent = true;
            contentPixels++;
          }
        }

        // 需要至少有一定比例的内容像素才认为该行有内容
        if (hasContent && contentPixels / totalSampled > 0.05) { // 至少5%的像素有内容
          lastContentY = y;
          break;
        }
      }

      // 添加余量，但避免过多的空白
      let actualHeight = lastContentY + 30; // 减少余量从50到30

      // 如果检测到的内容高度太小，可能是检测错误
      if (actualHeight < height * 0.1) {
        actualHeight = height * 0.1; // 至少保留10%的高度
      }

      // 调试信息
      if (height - actualHeight > 100) {
        console.log('[SmartPagination] 检测到空白区域:', {
          canvasHeight: height,
          actualHeight: actualHeight,
          blankArea: height - actualHeight
        });
      }

      return Math.min(actualHeight, height);

    } catch (error) {
      console.error('[SmartPagination] 检测内容高度失败:', error);
      return canvas.height;
    }
  }

  /**
   * 创建空文档结果
   * @returns {Object} 空文档的分页结果
   * @private
   */
  _createEmptyResult() {
    return {
      success: true,
      pages: [], // 返回空数组而不是包含高度为0的页面
      totalPages: 0,
      isEmpty: true
    };
  }
}

// 兼容浏览器和Node.js环境
(function() {
  if (typeof module !== 'undefined' && typeof module.exports !== 'undefined') {
    module.exports = SmartPagination;
  } else if (typeof window !== 'undefined') {
    window.SmartPagination = SmartPagination;
  }
})();