/**
 * 分页配置管理器
 * 文件：lib/pagination/pagination-config.js
 * 
 * 功能：
 * - 统一管理分页相关配置
 * - 提供默认值和配置验证
 * - 支持运行时配置更新
 */

class PaginationConfig {
  /**
   * 默认配置
   */
  static get DEFAULTS() {
    return {
      enabled: true,
      strategy: 'balanced', // conservative | balanced | compact
      pageHeight: 800,
      avoidOrphans: true,
      maxBacktrack: 100,
      minPageHeight: 100,
      tolerance: 10
    };
  }

  /**
   * localStorage键名
   */
  static get STORAGE_KEYS() {
    return {
      enabled: 'md2pdf_pagination_enabled',
      strategy: 'md2pdf_pagination_strategy', 
      pageHeight: 'md2pdf_page_height',
      avoidOrphans: 'md2pdf_avoid_orphans',
      maxBacktrack: 'md2pdf_max_backtrack',
      minPageHeight: 'md2pdf_min_page_height',
      tolerance: 'md2pdf_pagination_tolerance'
    };
  }

  /**
   * 有效的策略选项
   */
  static get VALID_STRATEGIES() {
    return ['conservative', 'balanced', 'compact'];
  }

  /**
   * 获取当前完整配置
   * @returns {Object} 配置对象
   */
  static getConfig() {
    const config = {};
    
    Object.keys(this.DEFAULTS).forEach(key => {
      const storageKey = this.STORAGE_KEYS[key];
      const defaultValue = this.DEFAULTS[key];
      const storedValue = this._getStorageValue(storageKey, defaultValue);
      
      config[key] = this._validateConfigValue(key, storedValue);
    });
    
    return config;
  }

  /**
   * 设置配置值
   * @param {string} key - 配置键
   * @param {*} value - 配置值
   */
  static setConfig(key, value) {
    if (!(key in this.DEFAULTS)) {
      console.warn(`[PaginationConfig] 未知配置键: ${key}`);
      return false;
    }
    
    const validatedValue = this._validateConfigValue(key, value);
    const storageKey = this.STORAGE_KEYS[key];
    
    try {
      localStorage?.setItem(storageKey, validatedValue.toString());
      return true;
    } catch (error) {
      console.error(`[PaginationConfig] 配置保存失败: ${error.message}`);
      return false;
    }
  }

  /**
   * 批量设置配置
   * @param {Object} configObj - 配置对象
   */
  static setConfigs(configObj) {
    const results = {};
    
    Object.keys(configObj).forEach(key => {
      results[key] = this.setConfig(key, configObj[key]);
    });
    
    return results;
  }

  /**
   * 重置为默认配置
   */
  static resetToDefaults() {
    Object.keys(this.DEFAULTS).forEach(key => {
      this.setConfig(key, this.DEFAULTS[key]);
    });
  }

  /**
   * 获取配置的用户界面选项
   */
  static getUIOptions() {
    return {
      strategies: this.VALID_STRATEGIES.map(strategy => ({
        value: strategy,
        label: this._getStrategyLabel(strategy),
        description: this._getStrategyDescription(strategy)
      })),
      pageHeightOptions: [
        { value: 600, label: '小页面 (600px)', description: '适合移动设备预览' },
        { value: 800, label: '标准页面 (800px)', description: '推荐的默认设置' },
        { value: 1000, label: '大页面 (1000px)', description: '减少页数，内容更紧凑' },
        { value: 1200, label: '超大页面 (1200px)', description: '最少页数' }
      ]
    };
  }

  /**
   * 验证配置完整性
   */
  static validateConfig(config = null) {
    const configToValidate = config || this.getConfig();
    const errors = [];
    const warnings = [];

    // 验证启用状态
    if (typeof configToValidate.enabled !== 'boolean') {
      errors.push('enabled 必须是布尔值');
    }

    // 验证策略
    if (!this.VALID_STRATEGIES.includes(configToValidate.strategy)) {
      errors.push(`strategy 必须是以下之一: ${this.VALID_STRATEGIES.join(', ')}`);
    }

    // 验证页面高度
    if (configToValidate.pageHeight < 200 || configToValidate.pageHeight > 2000) {
      errors.push('pageHeight 必须在 200-2000 之间');
    }

    // 验证最大回退距离
    if (configToValidate.maxBacktrack < 10 || configToValidate.maxBacktrack > 500) {
      warnings.push('maxBacktrack 建议在 10-500 之间');
    }

    // 验证最小页面高度
    if (configToValidate.minPageHeight > configToValidate.pageHeight / 2) {
      warnings.push('minPageHeight 不应超过 pageHeight 的一半');
    }

    return {
      valid: errors.length === 0,
      errors,
      warnings
    };
  }

  /**
   * 从localStorage获取值
   * @private
   */
  static _getStorageValue(storageKey, defaultValue) {
    try {
      const value = localStorage?.getItem(storageKey);
      if (value === null) return defaultValue;
      
      // 尝试解析为合适的类型
      if (typeof defaultValue === 'boolean') {
        return value === 'true';
      } else if (typeof defaultValue === 'number') {
        const numValue = Number(value);
        return isNaN(numValue) ? defaultValue : numValue;
      } else {
        return value;
      }
    } catch (error) {
      console.warn(`[PaginationConfig] 读取配置失败: ${error.message}`);
      return defaultValue;
    }
  }

  /**
   * 验证配置值
   * @private
   */
  static _validateConfigValue(key, value) {
    const defaultValue = this.DEFAULTS[key];
    
    switch (key) {
      case 'enabled':
      case 'avoidOrphans':
        return typeof value === 'boolean' ? value : Boolean(value);
        
      case 'strategy':
        return this.VALID_STRATEGIES.includes(value) ? value : 'balanced';
        
      case 'pageHeight':
        const height = Number(value);
        return (height >= 200 && height <= 2000) ? height : 800;
        
      case 'maxBacktrack':
        const backtrack = Number(value);
        return (backtrack >= 10 && backtrack <= 500) ? backtrack : 100;
        
      case 'minPageHeight':
        const minHeight = Number(value);
        return (minHeight >= 50 && minHeight <= 300) ? minHeight : 100;
        
      case 'tolerance':
        const tolerance = Number(value);
        return (tolerance >= 1 && tolerance <= 50) ? tolerance : 10;
        
      default:
        return defaultValue;
    }
  }

  /**
   * 获取策略标签
   * @private
   */
  static _getStrategyLabel(strategy) {
    const labels = {
      conservative: '保守分页',
      balanced: '平衡分页',
      compact: '紧凑分页'
    };
    return labels[strategy] || strategy;
  }

  /**
   * 获取策略描述
   * @private
   */
  static _getStrategyDescription(strategy) {
    const descriptions = {
      conservative: '最大程度避免内容被切分，可能产生更多页面',
      balanced: '在避免切分和页面利用率间平衡，推荐使用',
      compact: '最大化页面利用率，可能在非关键位置切分内容'
    };
    return descriptions[strategy] || '';
  }

  /**
   * 导出配置到JSON
   */
  static exportConfig() {
    const config = this.getConfig();
    const validation = this.validateConfig(config);
    
    return {
      config,
      validation,
      exportTime: new Date().toISOString(),
      version: '1.0.0'
    };
  }

  /**
   * 从JSON导入配置
   */
  static importConfig(configJson) {
    try {
      const imported = typeof configJson === 'string' ? 
        JSON.parse(configJson) : configJson;
      
      if (!imported.config) {
        throw new Error('配置数据格式无效');
      }
      
      const results = this.setConfigs(imported.config);
      const validation = this.validateConfig();
      
      return {
        success: Object.values(results).every(r => r),
        results,
        validation
      };
    } catch (error) {
      return {
        success: false,
        error: error.message
      };
    }
  }
}

module.exports = PaginationConfig;