/**
 * Mermaid工具方法模块
 * 提供通用的辅助功能
 */

class MermaidUtils {
  constructor() {
    // 移除了文本测量相关的属性
  }

  /**
   * HTML转义
   * @param {string} text - 要转义的文本
   * @returns {string} 转义后的文本
   */
  escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
  }

  /**
   * 等待Mermaid渲染完成
   * @param {Element} container - 容器元素
   * @param {number} timeout - 超时时间（毫秒）
   * @returns {Promise<void>}
   */
  async waitForMermaidRender(container, timeout = 5000) {
    return new Promise((resolve, reject) => {
      const startTime = Date.now();
      
      const checkRender = () => {
        const svgElements = container.querySelectorAll('.mermaid-container svg');
        const allRendered = Array.from(svgElements).every(svg => 
          svg.getAttribute('viewBox') && svg.querySelector('g')
        );
        
        if (allRendered) {
          resolve();
        } else if (Date.now() - startTime > timeout) {
          reject(new Error('Mermaid渲染超时'));
        } else {
          setTimeout(checkRender, 100);
        }
      };
      
      checkRender();
    });
  }

  /**
   * 等待SVG元素渲染完成
   * @param {Element} container - 容器元素
   * @param {number} timeout - 超时时间（毫秒）
   * @returns {Promise<void>}
   */
  async waitForSvgRender(container, timeout = 2000) {
    return new Promise((resolve) => {
      const startTime = Date.now();
      
      const checkRender = () => {
        const svgElements = container.querySelectorAll('svg');
        const allRendered = Array.from(svgElements).every(svg => {
          const hasViewBox = svg.hasAttribute('viewBox');
          const hasSize = svg.getBoundingClientRect().width > 0;
          const hasContent = svg.childNodes.length > 0;
          return hasViewBox || (hasSize && hasContent);
        });
        
        if (allRendered || Date.now() - startTime > timeout) {
          resolve();
        } else {
          requestAnimationFrame(checkRender);
        }
      };
      
      // 等待两个动画帧以确保DOM更新
      requestAnimationFrame(() => {
        requestAnimationFrame(checkRender);
      });
    });
  }


  /**
   * 获取支持的图表类型
   * @returns {Array<string>} 支持的图表类型列表
   */
  getSupportedDiagramTypes() {
    return [
      'graph', 'flowchart',
      'sequenceDiagram', 'sequence',
      'classDiagram', 'class',
      'stateDiagram', 'state',
      'erDiagram', 'er',
      'journey', 'userJourney',
      'gantt',
      'pie',
      'gitGraph', 'git',
      'mindmap',
      'quadrantChart'
    ];
  }

  /**
   * 快速检查HTML内容是否包含SVG元素
   * @param {string} htmlContent - HTML内容
   * @returns {boolean} 是否包含SVG
   */
  containsSvgElements(htmlContent) {
    return /<svg[^>]*>[\s\S]*?<\/svg>/i.test(htmlContent);
  }

  /**
   * 清理临时DOM元素
   */
  cleanupTempElements() {
    const tempElements = document.querySelectorAll('[id^="mermaid-temp-"]');
    tempElements.forEach(element => {
      if (element.parentNode) {
        element.parentNode.removeChild(element);
      }
    });
  }
}

// 导出工具类
if (typeof window !== 'undefined') {
  window.MermaidUtils = MermaidUtils;
}