/**
 * Mermaid语法修复模块
 * 负责验证和修复Mermaid代码语法
 */

class MermaidSyntaxFixer {
  constructor() {
    this.supportedTypes = [
      'graph', 'flowchart', 'sequenceDiagram', 'classDiagram', 
      'stateDiagram', 'erDiagram', 'journey', 'gantt', 'pie', 
      'gitGraph', 'mindmap', 'quadrantChart'
    ];
  }

  /**
   * 验证Mermaid语法
   * @param {string} code - Mermaid代码
   * @returns {Promise<boolean>} 验证结果
   */
  async validateMermaidSyntax(code) {
    if (!code || typeof code !== 'string') {
      throw new Error('无效的Mermaid代码');
    }

    // 基本语法检查
    const trimmedCode = code.trim();
    if (!trimmedCode) {
      throw new Error('Mermaid代码为空');
    }

    // 检查是否包含危险的HTML或JavaScript
    if (/<script/i.test(trimmedCode) || /javascript:/i.test(trimmedCode)) {
      throw new Error('Mermaid代码包含不安全的内容');
    }

    // 基础语法结构检查
    const diagramType = this.detectDiagramType(trimmedCode);
    if (!diagramType) {
      throw new Error('无法识别图表类型，请检查语法是否正确');
    }

    return true;
  }

  /**
   * 检测图表类型
   * @param {string} code - Mermaid代码
   * @returns {string|null} 图表类型
   */
  detectDiagramType(code) {
    for (const type of this.supportedTypes) {
      if (code.includes(type)) {
        return type;
      }
    }
    
    // 检查是否是默认的flowchart语法（没有明确声明类型）
    if (/^[A-Za-z0-9_]+(\s*-->\s*[A-Za-z0-9_]+|\s*\[.*?\]|\s*\{.*?\}|\s*\(.*?\))/.test(code.trim())) {
      return 'flowchart';
    }
    
    return null;
  }

  /**
   * 智能语法修复
   * @param {string} code - 原始Mermaid代码
   * @returns {string} 修复后的代码
   */
  attemptSyntaxFix(code) {
    console.log('[语法修复] 尝试修复Mermaid语法:', code.substring(0, 100) + '...');

    let fixedCode = code;

    // 检查是否包含复杂语法，如果有则使用更保守的修复策略
    const hasComplexSyntax = (
      code.includes('&') ||  // 多路径汇聚
      code.includes('<br>') || // HTML标签
      /\|[^|]*\|/.test(code) || // 带标签的连接线
      code.includes(':::') ||  // 样式定义
      code.split('\n').length > 15 // 复杂图表
    );

    // 修复规则集合
    const fixes = [];

    // 首先修复 subgraph 标题（这个修复优先级最高）
    fixes.push({
      name: 'subgraph 标题修复',
      pattern: /subgraph\s+([^\n]*?)(\s*\n|$)/gi,
      replacement: (match, title, newline) => {
        // 检查标题是否已经被引号包围
        if ((title.startsWith('"') && title.endsWith('"')) ||
            (title.startsWith("'") && title.endsWith("'"))) {
          return match; // 已经被引号包围，不需要修复
        }

        // 检查标题是否包含需要引号的字符（中文、括号、空格等）
        if (/[\u4e00-\u9fa5()（）\s]/.test(title)) {
          return `subgraph "${title}"${newline}`;
        }

        return match;
      }
    });

    if (hasComplexSyntax) {
      // 对于复杂语法，只应用最基本的修复
      console.log('[语法修复] 检测到复杂语法，使用保守修复策略');
      fixes.push(
        // 修复括号兼容性问题
        {
          name: '通用节点文本修复',
          pattern: /([a-zA-Z0-9_]+)\s*\[([^"][^\]]*)\]/g,
          replacement: '$1["$2"]'
        },
        // 只修复明显的特殊字符问题
        {
          name: '引号标准化',
          pattern: /[""]/g,
          replacement: '"'
        },
        {
          name: '单引号标准化',
          pattern: /['']/g,
          replacement: "'"
        }
      );
    } else {
      // 对于简单语法，使用完整的修复规则
      fixes.push(
        // 修复括号兼容性问题
        {
          name: '通用节点文本修复',
          pattern: /([a-zA-Z0-9_]+)\s*\[([^"][^\]]*)\]/g,
          replacement: '$1["$2"]'
        },

        // 修复中文圆括号问题
        {
          name: '中文圆括号修复',
          pattern: /([A-Za-z0-9_]+)\s*\(([^"]*[\u4e00-\u9fa5][^"]*)\)/g,
          replacement: '$1["$2"]'
        },

        // 修复节点定义中的中文标签
        {
          name: '节点标签修复',
          pattern: /([A-Za-z0-9_]+)\s*\[([^"]*[\u4e00-\u9fa5][^"]*)\]/g,
          replacement: '$1["$2"]'
        },

        // 修复不规范的箭头语法
        {
          name: '箭头语法修复',
          pattern: /--\^/g,
          replacement: '-->'
        },

        // 修复特殊字符
        {
          name: '引号标准化',
          pattern: /[""]/g,
          replacement: '"'
        },

        {
          name: '单引号标准化',
          pattern: /['']/g,
          replacement: "'"
        }
      );
    }
    
    // 应用修复规则
    fixes.forEach(fix => {
      const beforeFix = fixedCode;
      if (typeof fix.replacement === 'function') {
        fixedCode = fixedCode.replace(fix.pattern, fix.replacement);
      } else {
        fixedCode = fixedCode.replace(fix.pattern, fix.replacement);
      }
      if (beforeFix !== fixedCode) {
        console.log(`[语法修复] 应用规则: ${fix.name}`);
      }
    });
    
    // 对于复杂语法，不进行空白清理，保持原有格式
    if (!hasComplexSyntax) {
      // 清理多余的空白
      fixedCode = fixedCode.replace(/\s+/g, ' ').trim();
    }
    
    return fixedCode;
  }

  /**
   * 分类错误类型
   * @param {Error} error - 错误对象
   * @returns {string} 错误类型
   */
  classifyError(error) {
    const errorMessage = (error.message || error.toString()).toLowerCase();
    
    if (errorMessage.includes('parse error') || errorMessage.includes('expecting')) {
      return 'SYNTAX_ERROR';
    }
    
    if (errorMessage.includes('chinese') || /[\u4e00-\u9fa5]/.test(errorMessage)) {
      return 'CHINESE_CHAR_ERROR';
    }
    
    if (errorMessage.includes('undefined') || errorMessage.includes('not found')) {
      return 'REFERENCE_ERROR';
    }
    
    return 'UNKNOWN_ERROR';
  }

  /**
   * 获取错误修复建议
   * @param {string} errorType - 错误类型
   * @param {string} code - 原始代码
   * @returns {string} 修复建议
   */
  getErrorSuggestions(errorType, code) {
    switch (errorType) {
      case 'SYNTAX_ERROR':
        return '检查流程图语法，确保节点定义和连接正确。中文文本建议用双引号包围。';
      case 'CHINESE_CHAR_ERROR':
        return '中文字符可能导致解析问题，请将中文文本用双引号包围，如：A["中文文本"]';
      case 'REFERENCE_ERROR':
        return '检查节点引用是否正确，确保所有引用的节点都已定义。';
      default:
        return '请参考Mermaid官方文档检查语法格式。';
    }
  }

  /**
   * 获取支持的图表类型
   * @returns {Array<string>} 支持的图表类型列表
   */
  getSupportedDiagramTypes() {
    return [...this.supportedTypes];
  }
}

// 导出语法修复器
if (typeof window !== 'undefined') {
  window.MermaidSyntaxFixer = MermaidSyntaxFixer;
}