/**
 * Mermaid渲染器模块
 * 负责核心渲染逻辑
 */

class MermaidRenderer {
  constructor(parser, syntaxFixer, errorHandler) {
    this.parser = parser;
    this.syntaxFixer = syntaxFixer;
    this.errorHandler = errorHandler;
    this.renderQueue = [];
    this.processingQueue = false;
  }

  /**
   * 包装mermaid.render方法以防止错误DOM注入
   */
  wrapMermaidRender() {
    if (typeof mermaid !== 'undefined' && !mermaid._isWrapped) {
      const originalRender = mermaid.render;
      const self = this;
      
      mermaid.render = async function(id, definition, element) {
        try {
          // 调用原始render方法
          const result = await originalRender.call(this, id, definition, element);
          return result;
        } catch (error) {
          // 渲染失败时立即清理可能创建的DOM元素
          setTimeout(() => {
            self.errorHandler.immediateCleanupErrorElements(id);
          }, 0);
          
          // 重新抛出错误，让外层catch处理
          throw error;
        }
      };
      
      mermaid._isWrapped = true;
    }
  }

  /**
   * 在HTML中渲染Mermaid图表
   * @param {string} htmlContent - HTML内容
   * @param {Array} mermaidBlocks - Mermaid代码块数组
   * @param {Object} options - 渲染选项
   * @returns {Promise<string>} 渲染后的HTML
   */
  async renderMermaidInHTML(htmlContent, mermaidBlocks, options = {}) {
    if (mermaidBlocks.length === 0) {
      return htmlContent;
    }

    // 创建临时容器用于处理
    const tempDiv = document.createElement('div');
    tempDiv.innerHTML = htmlContent;
    
    // PDF模式下需要设置正确的容器宽度
    const isPdfMode = options.pdfExport === true;
    let containerWidth = '';
    if (isPdfMode) {
      // PDF模式：使用A4纸的标准宽度 (794px)
      containerWidth = 'width: 794px !important;';
    }
    
    // 添加CSS隔离，避免外部样式影响
    tempDiv.style.cssText = `
      position: absolute !important;
      left: -9999px !important;
      visibility: hidden !important;
      ${containerWidth}
      color: #333 !important;
    `;
    // 添加类名以便识别
    tempDiv.className = 'mermaid-render-container';
    document.body.appendChild(tempDiv);

    try {
      // 替换占位符为实际的Mermaid容器（应用classDef增强）
      for (const block of mermaidBlocks) {
        const placeholder = tempDiv.querySelector(`[data-mermaid-id="${block.id}"]`);
        if (placeholder) {
          const mermaidDiv = document.createElement('div');
          mermaidDiv.className = 'mermaid-diagram';
          // 移除所有样式设置，让Mermaid使用默认样式
          
          // 使用原始Mermaid代码，不进行样式增强
          mermaidDiv.textContent = block.code;
          
          placeholder.replaceWith(mermaidDiv);
        }
      }

      // 渲染所有Mermaid图表
      const mermaidElements = tempDiv.querySelectorAll('.mermaid-diagram');
      if (mermaidElements.length > 0) {
        await this.renderMermaidElements(mermaidElements);
        
        // 后处理：直接调整SVG节点尺寸以确保文字完整显示
        const isPdfMode = options.pdfExport === true;
        await this.postProcessSvgNodes(tempDiv, isPdfMode);
      }

      // 如果启用了SVG转换（用于Word导出），在DOM中直接转换
      if (options.convertSvg && window.svgConverter) {
        await this.convertSvgToImages(tempDiv, options);
      }

      const result = tempDiv.innerHTML;
      
      // 在返回结果前清理错误信息
      setTimeout(() => this.errorHandler.cleanupMermaidErrors(), 100);
      
      return result;
    } catch (error) {
      console.error('Mermaid渲染失败:', error);
      // 降级处理：返回包含错误信息的HTML
      return this.errorHandler.handleRenderError(htmlContent, mermaidBlocks, error);
    } finally {
      // 清理临时容器
      this.handleDebugMode(tempDiv, options, mermaidBlocks);
      
      // 清理可能遗留的错误信息
      setTimeout(() => this.errorHandler.cleanupMermaidErrors(), 200);
    }
  }

  /**
   * 渲染Mermaid元素（增强版）
   */
  async renderMermaidElements(elements) {
    const elementsArray = Array.from(elements);
    
    for (let i = 0; i < elementsArray.length; i++) {
      const element = elementsArray[i];
      const originalCode = element.textContent;
      const chartId = `mermaid-chart-${Date.now()}-${i}`;
      // 第一层：尝试直接渲染原始代码
      try {
        await this.syntaxFixer.validateMermaidSyntax(originalCode);
        const { svg } = await mermaid.render(chartId, originalCode);
        this.createSuccessfulRender(element, svg);
        continue;
      } catch (originalError) {
        // 立即清理Mermaid可能创建的错误DOM元素
        this.errorHandler.immediateCleanupErrorElements(chartId);
      }
      
      // 第二层：尝试语法修复后渲染
      try {
        const fixedCode = this.syntaxFixer.attemptSyntaxFix(originalCode);
        if (fixedCode !== originalCode) {
          await this.syntaxFixer.validateMermaidSyntax(fixedCode);
          const { svg } = await mermaid.render(chartId + '-fixed', fixedCode);
          this.createSuccessfulRender(element, svg, '已自动修复语法');
          continue;
        }
      } catch (fixError) {
        // 立即清理Mermaid可能创建的错误DOM元素
        this.errorHandler.immediateCleanupErrorElements(chartId + '-fixed');
      }
      
      // 第三层：最终降级处理 - 显示原始代码
      this.errorHandler.createFallbackRender(element, originalCode);
    }
  }

  /**
   * 创建成功的渲染结果
   */
  createSuccessfulRender(element, svg, message = null) {
    const wrapper = document.createElement('div');
    wrapper.className = 'mermaid-container';
    // 移除所有样式设置
    
    if (message) {
      const messageDiv = document.createElement('div');
      // 移除样式设置
      messageDiv.textContent = `✓ ${message}`;
      wrapper.appendChild(messageDiv);
    }
    
    wrapper.innerHTML += svg;
    
    // 移除所有SVG样式设置，让Mermaid使用默认样式
    
    element.replaceWith(wrapper);
  }

  /**
   * 后处理SVG节点
   * @param {Element} container - 包含SVG的容器
   * @param {boolean} isPdfMode - 是否为PDF模式
   */
  async postProcessSvgNodes(container, isPdfMode = false) {
    const svgElements = container.querySelectorAll('svg');
    
    // 动态计算最大宽度，基于容器实际可用宽度
    let maxWidth;
    if (isPdfMode) {
      // PDF模式：使用固定的A4宽度
      const containerWidth = 794; // A4宽度固定值
      const padding = 40; // PDF容器的padding
      maxWidth = containerWidth - padding * 2; // 实际可用宽度: 714px
    } else {
      // 预览模式：保持原有逻辑
      maxWidth = 600;
    }
    
    for (const svg of svgElements) {
      // 获取SVG的实际尺寸 - 多种方式确保准确性
      const bbox = svg.getBoundingClientRect();
      const currentWidth = Math.max(
        bbox.width || 0,
        parseFloat(svg.getAttribute('width')) || 0,
        parseFloat(svg.style.width) || 0
      );
      const currentHeight = Math.max(
        bbox.height || 0,
        parseFloat(svg.getAttribute('height')) || 0,
        parseFloat(svg.style.height) || 0
      );
      
      
      // SVG尺寸处理已移除，让Mermaid使用默认尺寸
    }
  }

  /**
   * 将SVG转换为图片（用于Word导出）
   */
  async convertSvgToImages(tempDiv, options) {
    // 等待SVG完全渲染
    await new Promise(resolve => setTimeout(resolve, 100));
    
    // 确保所有SVG元素都已经完全渲染
    const svgElements = tempDiv.querySelectorAll('svg');
    
    // 再次等待以确保SVG内容完全加载
    if (svgElements.length > 0) {
      // 使用svgConverter的waitForSvgRender方法，如果存在
      if (window.svgConverter && typeof window.svgConverter.waitForSvgRender === 'function') {
        await window.svgConverter.waitForSvgRender(svgElements[0]);
      } else {
        // 备用：简单等待
        await new Promise(resolve => setTimeout(resolve, 200));
      }
    }
    
    for (let i = 0; i < svgElements.length; i++) {
      const svg = svgElements[i];
      try {
        // 检查SVG是否有实际内容
        if (!svg.hasChildNodes() || !svg.hasAttribute('viewBox')) {
          await new Promise(resolve => setTimeout(resolve, 50));
        }
        
        const imageDataUrl = await window.svgConverter.convertSvgToImage(svg, options.svgOptions || {});
        
        const img = document.createElement('img');
        img.src = imageDataUrl;
        img.alt = `Mermaid Diagram ${i + 1}`;
        
        // 不设置任何尺寸属性，让htmlDocx库完全自主处理图片尺寸
        console.log(`[Mermaid渲染器] 不设置HTML尺寸属性，保持图片原始状态`);
        
        svg.replaceWith(img);
      } catch (error) {
        // SVG转换失败，跳过此元素
      }
    }
    
    // 验证转换结果
    const finalSvgCount = tempDiv.querySelectorAll('svg').length;
    const finalImgCount = tempDiv.querySelectorAll('img[src^="data:image"]').length;
  }

  /**
   * 处理调试模式
   */
  handleDebugMode(tempDiv, options, mermaidBlocks = []) {
    // 简化逻辑：始终清理临时容器，不再显示调试窗口
    // Mermaid渲染已经在iframe的renderer中完成，不需要额外的调试容器
    if (tempDiv && tempDiv.parentNode) {
      document.body.removeChild(tempDiv);
    }
  }
}

// 导出渲染器
if (typeof window !== 'undefined') {
  window.MermaidRenderer = MermaidRenderer;
}