/**
 * Mermaid解析器模块
 * 负责解析和预处理Markdown中的Mermaid代码
 */

class MermaidParser {
  constructor() {
    this.mermaidBlockId = 0;
  }

  /**
   * 检测Markdown内容中是否包含Mermaid图表
   * @param {string} content - Markdown内容
   * @returns {boolean} 是否包含Mermaid内容
   */
  detectMermaidContent(content) {
    if (typeof content !== 'string') return false;
    // 使用与extractMermaidBlocks一致的正则表达式
    const hasMermaid = /```mermaid\s*\n((?:[^`]|`(?!``))*?)```/gi.test(content);
    return hasMermaid;
  }

  /**
   * 提取Markdown中的Mermaid代码块
   * @param {string} content - Markdown内容
   * @returns {Array} Mermaid代码块数组
   */
  extractMermaidBlocks(content) {
    if (typeof content !== 'string') return [];
    
    // 改进的正则表达式：确保不会跨越代码块边界
    // 使用负向前瞻确保不会匹配到三个连续的反引号
    const mermaidRegex = /```mermaid\s*\n((?:[^`]|`(?!``))*?)```/gi;
    const blocks = [];
    let match;
    
    while ((match = mermaidRegex.exec(content)) !== null) {
      const extractedCode = match[1].trim();
      
      // 安全检查：确保提取的内容不包含其他代码块标记
      if (extractedCode.includes('```')) {
        console.warn('[MermaidParser] 警告：提取的内容包含代码块标记，可能存在问题:', {
          preview: extractedCode.substring(0, 100) + '...'
        });
        // 尝试修复：只取第一个代码块结束之前的内容
        const fixedCode = extractedCode.split('```')[0].trim();
        if (fixedCode) {
          const block = {
            original: match[0],
            code: fixedCode,
            index: match.index,
            id: `mermaid-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`
          };
          blocks.push(block);
        }
      } else {
        const block = {
          original: match[0],
          code: extractedCode,
          index: match.index,
          id: `mermaid-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`
        };
        blocks.push(block);
      }
    }
    
    return blocks;
  }

  /**
   * 预处理Markdown内容，将Mermaid代码块替换为占位符
   * @param {string} content - Markdown内容
   * @returns {Object} 包含处理后内容和Mermaid代码块的对象
   */
  preprocessMarkdown(content) {
    if (!this.detectMermaidContent(content)) {
      return { processedContent: content, mermaidBlocks: [] };
    }

    const mermaidBlocks = this.extractMermaidBlocks(content);
    let processedContent = content;
    
    // 从后往前替换，避免索引偏移问题
    for (let i = mermaidBlocks.length - 1; i >= 0; i--) {
      const block = mermaidBlocks[i];
      const placeholder = `<div class="mermaid-placeholder" data-mermaid-id="${block.id}">正在加载图表...</div>`;
      
      processedContent = processedContent.slice(0, block.index) + 
                        placeholder + 
                        processedContent.slice(block.index + block.original.length);
    }

    return { processedContent, mermaidBlocks };
  }
}

// 导出解析器
if (typeof window !== 'undefined') {
  window.MermaidParser = MermaidParser;
}