/**
 * Mermaid错误处理模块
 * 负责处理渲染错误、清理错误信息和降级显示
 */

class MermaidErrorHandler {
  constructor() {
    this.errorObserver = null;
    this.originalConsoleError = null;
  }

  /**
   * 设置全局错误处理
   */
  setupGlobalErrorHandling() {
    // 拦截console.error，过滤Mermaid的技术错误信息
    if (typeof window !== 'undefined' && !window._mermaidErrorHandlerSetup) {
      this.originalConsoleError = console.error;
      console.error = (...args) => {
        const message = args.join(' ');
        // 只过滤Mermaid相关的技术错误，保留其他错误输出
        if (message.includes('Syntax error in text') || 
            message.includes('mermaid version') ||
            message.includes('Parse error on line')) {
          // 静默忽略这些技术错误信息
          return;
        }
        // 其他错误正常输出
        this.originalConsoleError.apply(console, args);
      };
      
      window._mermaidErrorHandlerSetup = true;
    }
    
    // 监听DOM变化，及时清理错误元素
    if (typeof MutationObserver !== 'undefined' && !this.errorObserver) {
      this.errorObserver = new MutationObserver((mutations) => {
        let hasNewNodes = false;
        mutations.forEach((mutation) => {
          if (mutation.type === 'childList' && mutation.addedNodes.length > 0) {
            hasNewNodes = true;
          }
        });
        
        if (hasNewNodes) {
          // 延迟执行清理，确保错误元素已经添加到DOM
          setTimeout(() => this.cleanupMermaidErrors(), 50);
        }
      });
      
      this.errorObserver.observe(document.body, {
        childList: true,
        subtree: true
      });
    }
  }

  /**
   * 立即清理特定ID的错误元素
   * @param {string} chartId - 图表ID
   */
  immediateCleanupErrorElements(chartId) {
    try {
      console.log(`[立即清理] 开始清理图表ID: ${chartId}`);
      
      // 清理指定ID的元素（Mermaid错误时常创建 d+chartId 的元素）
      const specificElement = document.getElementById(`d${chartId}`);
      if (specificElement) {
        console.log(`[立即清理] 移除主要错误元素: d${chartId}`);
        specificElement.remove();
      }
      
      // 清理所有可能的相关元素
      const relatedSelectors = [
        `#${chartId}`,
        `[id*="${chartId}"]`,
        `[id^="d${chartId}"]`,
        `[id$="${chartId}"]`
      ];
      
      relatedSelectors.forEach(selector => {
        try {
          const elements = document.querySelectorAll(selector);
          elements.forEach(element => {
            if (element.id && element.id.includes(chartId)) {
              console.log(`[立即清理] 移除相关错误元素:`, element.id);
              element.remove();
            }
          });
        } catch (e) {
          // 忽略选择器错误
        }
      });
      
      // 使用更激进的清理策略 - 清理所有包含错误文本的新创建元素
      const allElements = document.querySelectorAll('*');
      allElements.forEach(element => {
        const text = element.textContent || '';
        const hasErrorText = text.includes('Syntax error in text') || 
                            text.includes('mermaid version') ||
                            text.includes('Parse error on line');
        
        // 检查是否是最近创建的元素（简单的时间戳检查）
        const isRecentlyCreated = !element.getAttribute('data-processed');
        
        if (hasErrorText && isRecentlyCreated && element.children.length === 0) {
          console.log(`[立即清理] 移除包含错误文本的元素:`, text.substring(0, 50));
          element.remove();
        } else if (element.id && element.id.includes('mermaid')) {
          // 标记已处理，避免重复检查
          element.setAttribute('data-processed', 'true');
        }
      });
      
    } catch (error) {
      console.warn('[立即清理] 清理特定错误元素时出现问题:', error);
    }
  }

  /**
   * 清理Mermaid错误信息元素
   */
  cleanupMermaidErrors() {
    try {
      // 移除所有可能显示技术错误信息的元素
      const errorSelectors = [
        '[class*="error"]',
        '[class*="mermaid-error"]', 
        'pre[style*="background-color: rgb(255, 255, 255)"]',
        '.mermaid + div',
        'div:contains("Syntax error in text")',
        'div:contains("mermaid version")'
      ];
      
      errorSelectors.forEach(selector => {
        try {
          const elements = document.querySelectorAll(selector);
          elements.forEach(element => {
            const text = element.textContent || '';
            // 检查是否包含Mermaid技术错误信息
            if (text.includes('Syntax error in text') || 
                text.includes('mermaid version') ||
                text.includes('Parse error on line') ||
                text.match(/Expecting.*got/)) {
              element.remove();
            }
          });
        } catch (e) {
          // 某些选择器可能不被支持，静默忽略
        }
      });
      
      // 使用更精确的方式查找包含错误文本的元素
      const allDivs = document.querySelectorAll('div');
      allDivs.forEach(div => {
        const text = div.textContent || '';
        if ((text.includes('Syntax error in text') || text.includes('mermaid version')) && 
            div.children.length === 0) { // 确保是文本节点而不是容器
          div.remove();
        }
      });
      
    } catch (error) {
      console.warn('[错误清理] 清理Mermaid错误信息时出现问题:', error);
    }
  }

  /**
   * 创建降级渲染结果（显示原始代码）
   * @param {Element} element - 要替换的元素
   * @param {string} code - 原始Mermaid代码
   */
  createFallbackRender(element, code) {
    const fallbackDiv = document.createElement('div');
    fallbackDiv.className = 'mermaid-fallback';
    fallbackDiv.style.cssText = `
      margin: 20px 0; 
      padding: 15px; 
      background: #f8f9fa; 
      border: 1px solid #dee2e6; 
      border-radius: 4px; 
      font-family: 'Courier New', monospace;
    `;
    
    fallbackDiv.innerHTML = `
      <div style="color: #6c757d; font-size: 12px; margin-bottom: 10px;">
        💡 Mermaid图表暂时无法渲染，显示原始代码：
      </div>
      <pre style="
        margin: 0; 
        padding: 10px; 
        background: #ffffff; 
        border: 1px solid #e9ecef; 
        border-radius: 3px; 
        font-size: 13px; 
        line-height: 1.4;
        white-space: pre-wrap;
        word-wrap: break-word;
        color: #495057;
      ">${this.escapeHtml(code)}</pre>
    `;
    
    element.replaceWith(fallbackDiv);
  }

  /**
   * 处理渲染错误（优化版）
   * @param {string} htmlContent - HTML内容
   * @param {Array} mermaidBlocks - Mermaid代码块
   * @param {Error} error - 错误对象
   * @returns {string} 处理后的HTML
   */
  handleRenderError(htmlContent, mermaidBlocks, error) {
    let result = htmlContent;
    
    console.log('[渲染错误处理] 处理渲染错误，影响图表数量:', mermaidBlocks.length);
    
    // 将占位符替换为降级显示（原始代码）
    for (const block of mermaidBlocks) {
      const placeholder = `<div class="mermaid-placeholder" data-mermaid-id="${block.id}"`;
      
      // 创建优雅的降级显示
      const fallbackHtml = `<div class="mermaid-fallback" style="
        margin: 20px 0; 
        padding: 15px; 
        background: #f8f9fa; 
        border: 1px solid #dee2e6; 
        border-radius: 4px; 
        font-family: 'Courier New', monospace;
      ">
        <div style="color: #6c757d; font-size: 12px; margin-bottom: 10px;">
          💡 Mermaid图表暂时无法渲染，显示原始代码：
        </div>
        <pre style="
          margin: 0; 
          padding: 10px; 
          background: #ffffff; 
          border: 1px solid #e9ecef; 
          border-radius: 3px; 
          font-size: 13px; 
          line-height: 1.4;
          white-space: pre-wrap;
          word-wrap: break-word;
          color: #495057;
        ">${this.escapeHtml(block.code)}</pre>
      </div>`;
      
      result = result.replace(new RegExp(placeholder + '[^>]*>.*?</div>', 'g'), fallbackHtml);
    }
    
    return result;
  }

  /**
   * HTML转义
   * @param {string} text - 要转义的文本
   * @returns {string} 转义后的文本
   */
  escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
  }

  /**
   * 清理资源
   */
  cleanup() {
    // 停止错误观察器
    if (this.errorObserver) {
      this.errorObserver.disconnect();
      this.errorObserver = null;
    }
    
    // 恢复原始console.error
    if (this.originalConsoleError && typeof window !== 'undefined') {
      console.error = this.originalConsoleError;
      window._mermaidErrorHandlerSetup = false;
    }
    
    // 清理错误信息
    this.cleanupMermaidErrors();
  }
}

// 导出错误处理器
if (typeof window !== 'undefined') {
  window.MermaidErrorHandler = MermaidErrorHandler;
}