/**
 * Mermaid核心处理器
 * 提供统一的Mermaid流程图解析、渲染和错误处理功能
 * 为popup、editor和content页面提供一致的Mermaid处理能力
 * 依赖：config.js, parser.js, syntax-fixer.js, error-handler.js, utils.js, renderer.js
 */

class MermaidProcessor {
  constructor(options = {}) {
    this.options = {
      theme: options.theme || 'default',
      securityLevel: 'loose',
      startOnLoad: false,
      ...options
    };
    
    this.isInitialized = false;
    
    // 初始化各个模块
    this.config = new MermaidConfig();
    this.parser = new MermaidParser();
    this.syntaxFixer = new MermaidSyntaxFixer();
    this.errorHandler = new MermaidErrorHandler();
    this.utils = new MermaidUtils();
    this.renderer = new MermaidRenderer(this.parser, this.syntaxFixer, this.errorHandler);
  }

  /**
   * 初始化Mermaid库
   */
  initialize(pdfMode = false) {
    if (this.isInitialized || typeof mermaid === 'undefined') {
      return;
    }

    try {
      // 生成Mermaid配置
      const mermaidConfig = this.config.generateInitConfig(this.options, pdfMode);
      
      // 初始化Mermaid
      mermaid.initialize(mermaidConfig);
      
      this.isInitialized = true;
      
      // 设置全局错误处理
      this.errorHandler.setupGlobalErrorHandling();
      
      // 包装mermaid.render方法
      this.renderer.wrapMermaidRender();
      
    } catch (error) {
      console.error('Mermaid初始化失败:', error);
    }
  }

  /**
   * 检测Markdown内容中是否包含Mermaid图表
   */
  detectMermaidContent(content) {
    const result = this.parser.detectMermaidContent(content);
    return result;
  }

  /**
   * 提取Markdown中的Mermaid代码块
   */
  extractMermaidBlocks(content) {
    const result = this.parser.extractMermaidBlocks(content);
    
    if (result && result.blocks) {
      result.blocks.forEach((block, index) => {
        console.log(`[MermaidFlow] 📝 [代码块提取] 块${index}:`, {
          id: block.id,
          contentLength: (block.content || '').length,
          preview: (block.content || '').substring(0, 100) + '...'
        });
      });
    } else {
      console.log('[MermaidFlow] ⚠️ [代码块提取] 未找到代码块或结果格式异常:', result);
    }
    
    return result;
  }

  /**
   * 预处理Markdown内容
   */
  preprocessMarkdown(content) {
    const result = this.parser.preprocessMarkdown(content);
    
    if (result) {
    } else {
      console.log('[MermaidFlow] ⚠️ [预处理] 预处理结果为空或格式异常:', result);
    }
    
    return result;
  }

  /**
   * 统一的处理入口：处理包含Mermaid的Markdown内容
   * @param {string} markdownContent - Markdown内容
   * @param {Object} markedInstance - marked实例
   * @param {Object} options - 处理选项
   * @returns {Promise<string>} 处理后的HTML
   */
  async processMarkdownWithMermaid(markdownContent, markedInstance = marked, options = {}) {
    if (!this.detectMermaidContent(markdownContent)) {
      // 没有Mermaid内容，直接返回marked处理结果
      console.log('[MermaidFlow] ⚠️ [主流程] 未检测到 mermaid 内容，使用普通 marked 处理');
      return markedInstance.parse(markdownContent);
    }

    // 检查是否为PDF模式，重新初始化Mermaid配置
    const isPdfMode = options.pdfExport === true;
    
    if (isPdfMode) {
      this.isInitialized = false; // 强制重新初始化
      this.initialize(true); // 传递PDF模式参数
    } else {
      // 普通模式或Word模式
      if (!this.isInitialized) {
        this.initialize(false);
      }
    }
    
    // 预处理：提取Mermaid代码块
    const { processedContent, mermaidBlocks } = this.preprocessMarkdown(markdownContent);
    
    // 使用marked处理预处理后的Markdown
    const htmlContent = markedInstance.parse(processedContent);
    
    // 渲染Mermaid图表，传递选项
    const finalHtml = await this.renderer.renderMermaidInHTML(htmlContent, mermaidBlocks, options);
    
    return finalHtml;
  }

  /**
   * 专门用于Word导出的处理入口：将SVG转换为图片
   * @param {string} markdownContent - Markdown内容
   * @param {Object} markedInstance - marked实例
   * @param {Object} conversionOptions - 转换选项
   * @returns {Promise<string>} 处理后的HTML（SVG已转换为图片）
   */
  async processMarkdownForWordExport(markdownContent, markedInstance = marked, conversionOptions = {}) {
    
    if (!this.detectMermaidContent(markdownContent)) {
      return markedInstance.parse(markdownContent);
    }

    // 检查SVG转换器是否可用
    if (!window.svgConverter) {
      return this.processMarkdownWithMermaid(markdownContent, markedInstance);
    }

    // 使用带SVG转换选项的处理方法
    const htmlContent = await this.processMarkdownWithMermaid(markdownContent, markedInstance, {
      convertSvg: true,
      svgOptions: {
        format: 'png',
        quality: 1.0,           // 默认最高质量
        backgroundColor: '#ffffff',
        wordExport: true,       // 标记为Word导出模式
        ...conversionOptions    // 允许调用方覆盖配置
      }
    });
    
    return htmlContent;
  }

  /**
   * 等待Mermaid渲染完成
   */
  async waitForMermaidRender(container, timeout = 5000) {
    return this.utils.waitForMermaidRender(container, timeout);
  }

  /**
   * 等待SVG元素渲染完成
   */
  async waitForSvgRender(container, timeout = 2000) {
    return this.utils.waitForSvgRender(container, timeout);
  }

  /**
   * 快速检查HTML内容是否包含SVG元素
   */
  containsSvgElements(htmlContent) {
    return this.utils.containsSvgElements(htmlContent);
  }

  /**
   * 获取支持的图表类型
   */
  getSupportedDiagramTypes() {
    return this.syntaxFixer.getSupportedDiagramTypes();
  }

  /**
   * 清理资源
   */
  cleanup() {
    // 清理临时DOM元素
    this.utils.cleanupTempElements();
    
    // 清理错误处理器
    this.errorHandler.cleanup();
  }
}

// 全局单例实例
window.MermaidProcessor = MermaidProcessor;

// 创建默认实例
if (typeof window !== 'undefined') {
  window.mermaidProcessor = new MermaidProcessor();
}