/**
 * Mermaid配置管理模块
 * 负责管理Mermaid的主题配置和初始化配置
 */

class MermaidConfig {
  constructor() {
    this.defaultOptions = {
      theme: 'default',
      securityLevel: 'loose',
      startOnLoad: false
    };
  }

  /**
   * 获取Mermaid主题配置
   * @param {string} theme - 主题名称
   * @returns {Object} 主题配置对象
   */
  getMermaidConfig(theme = 'default') {
    // 简化主题配置，让Mermaid使用其默认样式
    const themes = {
      default: {
        theme: 'default'
        // 移除所有自定义颜色配置，使用Mermaid默认主题
      },
      
      dark: {
        theme: 'dark'
        // 使用Mermaid内置的dark主题
      },
      
      neutral: {
        theme: 'neutral'
        // 使用Mermaid内置的neutral主题
      }
    };

    return themes[theme] || themes.default;
  }

  /**
   * 生成完整的Mermaid初始化配置
   * @param {Object} options - 配置选项
   * @param {boolean} pdfMode - 是否为PDF模式
   * @returns {Object} Mermaid初始化配置
   */
  generateInitConfig(options = {}, pdfMode = false) {
    const mergedOptions = { ...this.defaultOptions, ...options };
    const mermaidConfig = this.getMermaidConfig(mergedOptions.theme);
    
    // 使用Mermaid的默认配置，只设置必要的选项
    const config = {
      startOnLoad: mergedOptions.startOnLoad,
      theme: mermaidConfig.theme,
      securityLevel: mergedOptions.securityLevel,
      maxTextSize: 90000,
      suppressErrors: false,
      deterministicIds: true,
      // 添加flowchart配置，限制宽度防止超出页面
      flowchart: {
        useMaxWidth: true,   // 启用最大宽度限制
        maxWidth: 700,       // 设置最大宽度为700px（小于PDF容器的714px可用宽度）
        htmlLabels: false,   // 使用纯SVG文字，避免foreignObject导致的文字遮挡问题
        curve: 'basis',      // 使用默认的连线样式
        nodeSpacing: 50,     // 减小节点间距（默认50）
        rankSpacing: 50,     // 减小层级间距（默认50）
        diagramPadding: 8,   // 减小图表内边距
        wrap: true,          // 启用文字换行
        wrapPadding: 15,     // 增加文字换行时的内边距，为垂直居中提供更多空间
        padding: 20,         // 增加节点内边距，提供足够的垂直空间
        fontSize: 13,        // 设置字体大小为13px，减少文字被遮挡的问题
        // 文字对齐相关配置
        textPlacement: 'middle',      // 文字垂直居中
        labelPosition: 'center',      // 标签位置居中
        defaultRenderer: 'svg'        // 确保使用SVG渲染器以获得更好的文字控制
      },
      // 为其他图表类型也添加宽度限制、字体大小和禁用HTML标签
      sequence: {
        useMaxWidth: true,
        maxWidth: 700,
        fontSize: 13,
        htmlLabels: false   // 使用纯SVG文字，避免foreignObject问题
      },
      gantt: {
        useMaxWidth: true,
        maxWidth: 700,
        fontSize: 13,
        htmlLabels: false   // 使用纯SVG文字，避免foreignObject问题
      },
      pie: {
        useMaxWidth: true,
        maxWidth: 700,
        fontSize: 13,
        htmlLabels: false   // 使用纯SVG文字，避免foreignObject问题
      },
      er: {
        useMaxWidth: true,
        maxWidth: 700,
        fontSize: 13,
        htmlLabels: false   // 使用纯SVG文字，避免foreignObject问题
      },
      journey: {
        useMaxWidth: true,
        maxWidth: 700,
        fontSize: 13,
        htmlLabels: false   // 使用纯SVG文字，避免foreignObject问题
      }
    };
    
    return config;
  }
}

// 导出配置管理器
if (typeof window !== 'undefined') {
  window.MermaidConfig = MermaidConfig;
}