/**
 * Mermaid模块加载器
 * 负责检查和管理所有Mermaid相关模块的加载状态
 * 确保所有依赖模块按正确顺序加载后才初始化核心处理器
 * 注意：需要按依赖顺序加载
 */

// 检查环境并加载模块
(function() {
  'use strict';
  
  // 确保在浏览器环境中运行
  if (typeof window === 'undefined') {
    console.error('MermaidProcessor requires browser environment');
    return;
  }
  
  // 检查必要的依赖
  if (typeof mermaid === 'undefined') {
    console.error('MermaidProcessor requires mermaid.js to be loaded first');
    return;
  }
  
  // 模块加载状态
  const moduleStatus = {
    config: false,
    parser: false,
    syntaxFixer: false,
    errorHandler: false,
    utils: false,
    renderer: false,
    processor: false
  };
  
  // 检查所有模块是否已加载
  function checkAllModulesLoaded() {
    return Object.values(moduleStatus).every(status => status === true);
  }
  
  // 标记模块已加载
  function markModuleLoaded(moduleName) {
    moduleStatus[moduleName] = true;
    console.log(`[MermaidProcessor] ${moduleName} 模块已加载`);
    
    if (checkAllModulesLoaded()) {
      console.log('[MermaidProcessor] 所有模块加载完成');
      initializeProcessor();
    }
  }
  
  // 初始化处理器
  function initializeProcessor() {
    // 确保全局实例存在
    if (!window.mermaidProcessor || !(window.mermaidProcessor instanceof window.MermaidProcessor)) {
      console.warn('[MermaidProcessor] 处理器实例未正确创建');
    }
  }
  
  // 动态加载模块（用于开发环境）
  function loadModule(path, moduleName) {
    const script = document.createElement('script');
    script.src = path;
    script.onload = () => markModuleLoaded(moduleName);
    script.onerror = () => console.error(`Failed to load ${moduleName} from ${path}`);
    document.head.appendChild(script);
  }
  
  // 检查模块是否已经加载
  if (window.MermaidConfig) moduleStatus.config = true;
  if (window.MermaidParser) moduleStatus.parser = true;
  if (window.MermaidSyntaxFixer) moduleStatus.syntaxFixer = true;
  if (window.MermaidErrorHandler) moduleStatus.errorHandler = true;
  if (window.MermaidUtils) moduleStatus.utils = true;
  if (window.MermaidRenderer) moduleStatus.renderer = true;
  if (window.MermaidProcessor) moduleStatus.processor = true;
  
  // 如果所有模块都已加载，直接初始化
  if (checkAllModulesLoaded()) {
    initializeProcessor();
  } else {
    console.log('[MermaidProcessor] 等待所有模块加载...');
    // 注意：在生产环境中，模块应该通过manifest.json按顺序加载
    // 这里的动态加载仅用于调试
  }
  
})();