const MathContentDetector = require('../MathContentDetector.js');
const DollarDollarProtector = require('../DollarDollarProtector.js');

describe('Utils模块集成测试', () => {
  let detector, protector;

  beforeEach(() => {
    detector = new MathContentDetector();
    protector = new DollarDollarProtector();
    // 禁用测试期间的控制台日志
    process.env.NODE_ENV = 'test';
  });

  afterEach(() => {
    delete process.env.NODE_ENV;
  });

  describe('MathContentDetector与DollarDollarProtector协同工作', () => {
    test('应该在保护后仍能检测数学环境', () => {
      const text = `
\\begin{equation}
E = mc^2
\\end{equation}

Display math: $$F = ma$$

Inline math: $v = u + at$
      `;

      // 先保护$$公式
      const protectedText = protector.protect(text);
      
      // 然后检测数学环境（应该不受影响）
      const mathEnvResult = detector.isMathEnvironment(protectedText);
      expect(mathEnvResult.isMath).toBe(true);
      expect(mathEnvResult.environment).toBe('equation');

      // 恢复后应该包含完整内容
      const restored = protector.restore(protectedText);
      expect(restored).toContain('$$F = ma$$');
      expect(restored).toContain('$v = u + at$');
    });

    test('应该保护$$公式后检测剩余的行内数学', () => {
      const text = 'Mixed: $$display = formula$$ and $inline = math$ content.';

      const protectedText = protector.protect(text);
      expect(protectedText).not.toContain('$$');
      expect(protectedText).toContain('$inline = math$');

      // 检测剩余的行内数学
      const hasInline = detector.isInlineMath(protectedText);
      expect(hasInline.isMath).toBe(true);
      expect(hasInline.type).toBe('inline_math');

      const blocks = detector.extractMathBlocks(protectedText);
      expect(blocks.length).toBeGreaterThan(0);
      
      const inlineBlocks = blocks.filter(block => block.type === 'inline');
      expect(inlineBlocks.length).toBeGreaterThan(0);
    });

    test('应该在保护-恢复循环中保持内容完整性', () => {
      const originalText = `
Complex document with:
\\begin{align}
x &= a + b \\\\
y &= c + d
\\end{align}

And display math: $$E = mc^2$$

And inline math: $v = u + at$
      `;

      // 1. 检测原始内容
      const originalHasMath = detector.hasMathContent(originalText);
      expect(originalHasMath).toBe(true);

      // 2. 保护$$公式
      const protectedText = protector.protect(originalText);
      expect(protector.getProtectionCount()).toBe(1);

      // 3. 处理保护后的文本（模拟解析器处理）
      const envResult = detector.isMathEnvironment(protectedText);
      expect(envResult.isMath).toBe(true);
      expect(envResult.environment).toBe('align');
      
      // 4. 恢复保护的内容
      const finalText = protector.restore(protectedText);

      // 验证完整性
      expect(finalText).toBe(originalText);

      // 验证最终结果仍包含数学内容
      const finalHasMath = detector.hasMathContent(finalText);
      expect(finalHasMath).toBe(true);
    });

    test('应该正确处理数学内容分类', () => {
      const testCases = [
        {
          input: '$$E = mc^2$$',
          expectedHasDisplay: true
        },
        {
          input: '$x = y$',
          expectedHasInline: true
        },
        {
          input: '\\begin{equation}F = ma\\end{equation}',
          expectedHasEnvironment: true
        },
        {
          input: 'Regular text',
          expectedHasDisplay: false,
          expectedHasInline: false
        }
      ];

      testCases.forEach(testCase => {
        if (testCase.expectedHasDisplay) {
          const result = detector.isDisplayMath(testCase.input);
          expect(result.isMath).toBe(true);
        }
        
        if (testCase.expectedHasInline) {
          const result = detector.isInlineMath(testCase.input);
          expect(result.isMath).toBe(true);
        }
        
        if (testCase.expectedHasEnvironment) {
          const result = detector.isMathEnvironment(testCase.input);
          expect(result.isMath).toBe(true);
        }
        
        if (testCase.expectedHasDisplay === false && testCase.expectedHasInline === false) {
          const displayResult = detector.isDisplayMath(testCase.input);
          const inlineResult = detector.isInlineMath(testCase.input);
          expect(displayResult.isMath).toBe(false);
          expect(inlineResult.isMath).toBe(false);
        }
      });
    });
  });

  describe('在解析器使用模式中的集成', () => {
    test('应该模拟LayerParser的典型使用流程', () => {
      const inputText = `
Document with mixed content:

\\begin{equation}
E = mc^2  
\\end{equation}

Some text with $$display = math$$ and $inline = math$.

\\begin{matrix}
1 & 2 \\\\
3 & 4
\\end{matrix}
      `;

      // 模拟LayerParser.process()的典型流程
      
      // 1. 保护$$数学内容
      const protectedText = protector.protect(inputText);
      
      expect(protector.getProtectionCount()).toBe(1);
      expect(protectedText).not.toContain('$$display = math$$');

      // 2. 检测剩余数学环境
      const hasEquation = detector.isMathEnvironment(protectedText);
      expect(hasEquation.isMath).toBe(true);
      expect(['equation', 'matrix']).toContain(hasEquation.environment);

      // 3. 检测剩余行内数学
      const hasInlineMath = detector.isInlineMath(protectedText);
      expect(hasInlineMath.isMath).toBe(true);

      const mathBlocks = detector.extractMathBlocks(protectedText);
      const inlineBlocks = mathBlocks.filter(block => block.type === 'inline');
      expect(inlineBlocks.length).toBeGreaterThan(0);

      // 4. 模拟处理逻辑（这里只是简单标记）
      let processedText = protectedText.replace(/\\begin{(\w+)}/g, '[PROCESSED:$1]');
      processedText = processedText.replace(/\$([^$]+)\$/g, '[INLINE:$1]');

      // 5. 恢复保护的内容
      const finalText = protector.restore(processedText);

      // 验证结果
      expect(finalText).toContain('$$display = math$$');
      expect(finalText).toContain('[PROCESSED:equation]');
      expect(finalText).toContain('[PROCESSED:matrix]');
      expect(finalText).toContain('[INLINE:inline = math]');
    });

    test('应该处理多层嵌套的保护和检测', () => {
      const complexText = `
Nested content:
\\begin{equation}
\\text{Energy formula: } E = mc^2 \\text{ and force } $$F = ma$$ 
\\end{equation}

Additional: $v = u + at$ formula.
      `;

      // 第一层保护
      const firstProtectedText = protector.protect(complexText);
      
      // 检测环境（应该检测到包含保护内容的环境）
      const envResult = detector.isMathEnvironment(firstProtectedText);
      expect(envResult.isMath).toBe(true);
      expect(envResult.environment).toBe('equation');

      // 恢复第一层
      const firstRestored = protector.restore(firstProtectedText);
      
      // 验证完整性
      expect(firstRestored).toBe(complexText);
    });
  });

  describe('错误处理和边界情况的集成', () => {
    test('应该在一个工具出错时不影响另一个工具', () => {
      const problematicText = `
\\begin{equation}
E = mc^2
\\end{equation

Incomplete $$formula
      `;

      // 保护应该处理不完整的$$
      const protectedText = protector.protect(problematicText);
      expect(protector.getProtectionCount()).toBe(0); // 不保护不完整的公式

      // 检测应该处理不完整的环境
      const envResult = detector.isMathEnvironment(problematicText);
      expect(typeof envResult.isMath).toBe('boolean');
      
      // 两个工具都应该继续工作
      expect(() => {
        const inlineResult = detector.isInlineMath(protectedText);
        const hasDoubleDollar = protector.hasDoubleDollarMath(problematicText);
        
        expect(typeof inlineResult.isMath).toBe('boolean');
        expect(typeof hasDoubleDollar).toBe('boolean');
      }).not.toThrow();
    });

    test('应该处理空内容和无效输入的组合', () => {
      const emptyInputs = ['', null, undefined];

      emptyInputs.forEach(input => {
        expect(() => {
          const protectedText = protector.protect(input);
          const hasInline = detector.isInlineMath(input);
          const hasDisplay = detector.isDisplayMath(input);
          const envResult = detector.isMathEnvironment(input);
          
          if (hasInline) expect(typeof hasInline.isMath).toBe('boolean');
          if (hasDisplay) expect(typeof hasDisplay.isMath).toBe('boolean');
          if (envResult) expect(typeof envResult.isMath).toBe('boolean');
        }).not.toThrow();
      });
    });
  });

  describe('性能优化的集成测试', () => {
    test('应该在大文档中高效协作', () => {
      // 创建大文档
      const baseUnit = `
\\begin{equation}
E_{${Math.random()}} = mc^2
\\end{equation}

Text with $$F_{${Math.random()}} = ma$$ and $v = u + at$.
      `;
      const largeDocument = baseUnit.repeat(20); // 减少重复次数以避免超时

      const startTime = Date.now();

      // 完整的保护-检测-恢复流程
      const protectedText = protector.protect(largeDocument);
      const envResult = detector.isMathEnvironment(protectedText);
      const blocks = detector.extractMathBlocks(protectedText);
      const restored = protector.restore(protectedText);

      const endTime = Date.now();

      // 验证结果
      expect(protector.getProtectionCount()).toBe(20); // 20个$$公式
      expect(envResult.isMath).toBe(true); // 至少检测到equation环境
      expect(blocks.length).toBeGreaterThan(0); // 检测到行内数学
      expect(restored).toBe(largeDocument);

      // 性能要求
      expect(endTime - startTime).toBeLessThan(2000); // 2秒内完成
    });

    test('应该支持内存高效的清理', () => {
      // 创建多个保护实例
      const texts = Array.from({length: 10}, (_, i) => `Formula ${i}: $$x_{${i}} = y_{${i}}$$.`);

      texts.forEach(text => {
        protector.protect(text);
      });

      expect(protector.getProtectionCount()).toBe(10);

      // 清理应该释放内存
      protector.clear();
      expect(protector.getProtectionCount()).toBe(0);

      // 后续操作应该正常
      const newText = 'New: $$a = b$$.';
      const newResult = protector.protect(newText);
      const newRestored = protector.restore(newResult);
      expect(newRestored).toBe(newText);
    });
  });

  describe('接口稳定性的集成测试', () => {
    test('应该维护接口契约的一致性', () => {
      const testText = 'Mixed: $$E = mc^2$$ and $F = ma$ content.';

      // MathContentDetector接口契约
      const envResult = detector.isMathEnvironment(testText);
      const inlineResult = detector.isInlineMath(testText);
      const displayResult = detector.isDisplayMath(testText);
      const blocks = detector.extractMathBlocks(testText);
      const classification = detector.classifyContent(testText);

      expect(envResult).toHaveProperty('isMath');
      expect(envResult).toHaveProperty('type');
      expect(inlineResult).toHaveProperty('isMath');
      expect(inlineResult).toHaveProperty('type');
      expect(displayResult).toHaveProperty('isMath');
      expect(displayResult).toHaveProperty('type');
      expect(Array.isArray(blocks)).toBe(true);
      expect(classification).toHaveProperty('type');
      expect(classification).toHaveProperty('confidence');

      // DollarDollarProtector接口契约
      const protectedText = protector.protect(testText);
      const protectionCount = protector.getProtectionCount();
      const hasDoubleDollar = protector.hasDoubleDollarMath(testText);

      expect(typeof protectedText).toBe('string');
      expect(typeof protectionCount).toBe('number');
      expect(typeof hasDoubleDollar).toBe('boolean');

      // 恢复接口契约
      const restored = protector.restore(protectedText);
      expect(typeof restored).toBe('string');
    });

    test('应该在异常情况下保持接口契约', () => {
      const invalidInputs = [null, undefined, {}, [], 123];

      invalidInputs.forEach(input => {
        // 所有方法都应该返回预期类型，不抛出异常
        expect(() => {
          const envResult = detector.isMathEnvironment(input);
          const inlineResult = detector.isInlineMath(input);
          const blocks = detector.extractMathBlocks(input);
          const hasMath = detector.hasMathContent(input);
          
          const protectedText = protector.protect(input);
          const hasDoubleDollar = protector.hasDoubleDollarMath(input);

          if (envResult) expect(typeof envResult.isMath).toBe('boolean');
          if (inlineResult) expect(typeof inlineResult.isMath).toBe('boolean');
          if (blocks) expect(Array.isArray(blocks)).toBe(true);
          if (hasMath !== undefined) expect(typeof hasMath).toBe('boolean');
          if (hasDoubleDollar !== undefined) expect(typeof hasDoubleDollar).toBe('boolean');
        }).not.toThrow();
      });
    });
  });

  describe('实际使用场景测试', () => {
    test('应该处理典型的LaTeX文档', () => {
      const latexDocument = `
# Mathematical Document

This document contains various mathematical expressions.

## Inline Math
The famous equation $E = mc^2$ shows the relationship between mass and energy.

## Display Math
The quadratic formula is:
$$x = \\frac{-b \\pm \\sqrt{b^2 - 4ac}}{2a}$$

## Mathematical Environments
The system of equations can be written as:
\\begin{align}
2x + 3y &= 7 \\\\
x - y &= 1
\\end{align}

Matrix representation:
\\begin{pmatrix}
2 & 3 \\\\
1 & -1
\\end{pmatrix}
\\begin{pmatrix}
x \\\\
y
\\end{pmatrix}
=
\\begin{pmatrix}
7 \\\\
1
\\end{pmatrix}
      `;

      // 完整的处理流程
      const protectedText = protector.protect(latexDocument);
      
      // 验证保护效果
      expect(protector.getProtectionCount()).toBe(1); // 一个$$公式被保护
      expect(protectedText).not.toContain('$$x = \\frac');

      // 检测各种数学内容
      const hasInline = detector.isInlineMath(protectedText);
      const hasDisplay = detector.isDisplayMath(protectedText);
      const hasEnvironment = detector.isMathEnvironment(protectedText);
      const hasMathContent = detector.hasMathContent(protectedText);

      expect(hasInline.isMath).toBe(true); // $E = mc^2$
      expect(hasDisplay.isMath).toBe(false); // $$公式已被保护
      expect(hasEnvironment.isMath).toBe(true); // align或pmatrix环境
      expect(hasMathContent).toBe(true);

      // 提取数学块
      const mathBlocks = detector.extractMathBlocks(protectedText);
      expect(mathBlocks.length).toBeGreaterThan(0);

      // 恢复并验证完整性
      const restored = protector.restore(protectedText);
      expect(restored).toBe(latexDocument);
      
      // 恢复后应该能检测到所有数学内容
      const restoredHasMath = detector.hasMathContent(restored);
      expect(restoredHasMath).toBe(true);
    });

    test('应该处理边界情况的组合', () => {
      const edgeCases = [
        '', // 空文档
        'No math content', // 无数学内容
        '$single$', // 单个行内公式
        '$$single$$', // 单个显示公式
        '$a$ $$b$$ $c$', // 混合公式
        '\\begin{equation}x = y\\end{equation}', // 单个环境
        'Text $incomplete', // 不完整公式
        '$$incomplete', // 不完整显示公式
      ];

      edgeCases.forEach(testCase => {
        expect(() => {
          const protectedText = protector.protect(testCase);
          const blocks = detector.extractMathBlocks(protectedText);
          const restored = protector.restore(protectedText);
          
          expect(typeof protectedText).toBe('string');
          expect(Array.isArray(blocks)).toBe(true);
          expect(typeof restored).toBe('string');
        }).not.toThrow();
      });
    });

    test('应该支持链式操作', () => {
      const text = 'Chain test: $$E = mc^2$$ and $F = ma$.';
      
      // 链式操作：保护 -> 检测 -> 处理 -> 恢复
      const result = protector.protect(text);
      const hasMath = detector.hasMathContent(result);
      const blocks = detector.extractMathBlocks(result);
      const final = protector.restore(result);
      
      expect(hasMath).toBe(true);
      expect(blocks.length).toBeGreaterThan(0);
      expect(final).toBe(text);
      
      // 验证中间状态
      expect(result).not.toContain('$$E = mc^2$$');
      expect(result).toContain('$F = ma$');
    });
  });
});