#!/usr/bin/env node

/**
 * Utils模块测试运行器
 * 
 * 使用说明：
 * 1. 确保已安装Jest: npm install --save-dev jest
 * 2. 运行所有测试: node run-tests.js
 * 3. 运行特定测试: node run-tests.js MathContentDetector
 * 4. 显示覆盖率: node run-tests.js --coverage
 */

const { execSync } = require('child_process');
const fs = require('fs');
const path = require('path');

class TestRunner {
  constructor() {
    this.testDir = __dirname;
    this.utilsDir = path.dirname(this.testDir);
    this.testFiles = this.getTestFiles();
  }

  getTestFiles() {
    const files = fs.readdirSync(this.testDir);
    return files.filter(file => file.endsWith('.test.js') && file !== 'run-tests.js');
  }

  printHeader() {
    console.log('\n' + '='.repeat(60));
    console.log('🧪 LaTeX Parser V3 - Utils模块测试套件');
    console.log('='.repeat(60));
    console.log(`📂 测试目录: ${this.testDir}`);
    console.log(`📁 工具目录: ${this.utilsDir}`);
    console.log(`📋 测试文件: ${this.testFiles.length}个`);
    this.testFiles.forEach((file, index) => {
      console.log(`   ${index + 1}. ${file}`);
    });
    console.log('='.repeat(60) + '\n');
  }

  checkPrerequisites() {
    // 检查Jest是否可用
    try {
      execSync('npx jest --version', { stdio: 'pipe' });
      console.log('✅ Jest测试框架可用');
    } catch (error) {
      console.error('❌ Jest测试框架不可用，请先安装：npm install --save-dev jest');
      process.exit(1);
    }

    // 检查工具类文件是否存在
    const requiredFiles = [
      path.join(this.utilsDir, 'MathContentDetector.js'),
      path.join(this.utilsDir, 'DollarDollarProtector.js'),
      path.join(this.utilsDir, 'interface.md')
    ];

    let allFilesExist = true;
    requiredFiles.forEach(file => {
      if (fs.existsSync(file)) {
        console.log(`✅ ${path.basename(file)} 存在`);
      } else {
        console.error(`❌ ${path.basename(file)} 不存在`);
        allFilesExist = false;
      }
    });

    if (!allFilesExist) {
      console.error('\n❌ 缺少必要文件，无法运行测试');
      process.exit(1);
    }
  }

  runTests(args = []) {
    let jestArgs = ['jest'];
    let testPattern = '**/*.test.js';

    // 处理命令行参数
    if (args.includes('--coverage')) {
      jestArgs.push('--coverage');
      jestArgs.push('--coverageDirectory=coverage');
      jestArgs.push('--collectCoverageFrom=../*.js');
      jestArgs.push('--collectCoverageFrom=!../tests/**');
    }

    if (args.includes('--verbose')) {
      jestArgs.push('--verbose');
    }

    if (args.includes('--watch')) {
      jestArgs.push('--watch');
    }

    // 检查是否指定了特定测试文件
    const specificTest = args.find(arg => 
      !arg.startsWith('--') && 
      this.testFiles.some(file => file.includes(arg))
    );

    if (specificTest) {
      const matchingFile = this.testFiles.find(file => file.includes(specificTest));
      testPattern = matchingFile;
      console.log(`🎯 运行特定测试: ${matchingFile}`);
    }

    jestArgs.push(testPattern);

    // 设置Jest配置
    const jestConfig = {
      testEnvironment: 'node',
      testMatch: [`${this.testDir}/**/*.test.js`],
      verbose: true,
      collectCoverage: args.includes('--coverage'),
      coverageReporters: ['text', 'html', 'lcov'],
      coverageDirectory: path.join(this.testDir, 'coverage'),
      collectCoverageFrom: [
        '../*.js',
        '!../tests/**'
      ],
      testTimeout: 30000
    };

    // 创建临时Jest配置文件
    const configFile = path.join(this.testDir, 'jest.config.temp.js');
    fs.writeFileSync(configFile, `module.exports = ${JSON.stringify(jestConfig, null, 2)};`);

    try {
      console.log('🚀 开始运行测试...\n');
      
      const command = `npx jest --config ${configFile}`;
      console.log(`执行命令: ${command}\n`);
      
      execSync(command, { 
        stdio: 'inherit',
        cwd: this.testDir
      });
      
      console.log('\n✅ 所有测试完成！');
      
    } catch (error) {
      console.error('\n❌ 测试运行失败');
      console.error(error.message);
      process.exit(1);
    } finally {
      // 清理临时配置文件
      if (fs.existsSync(configFile)) {
        fs.unlinkSync(configFile);
      }
    }
  }

  showHelp() {
    console.log(`
📖 使用说明:

基本命令:
  node run-tests.js                    # 运行所有测试
  node run-tests.js --coverage         # 运行测试并生成覆盖率报告
  node run-tests.js --verbose          # 显示详细输出
  node run-tests.js --watch            # 监视模式运行

指定测试:
  node run-tests.js MathContentDetector    # 只运行MathContentDetector测试
  node run-tests.js DollarDollarProtector  # 只运行DollarDollarProtector测试
  node run-tests.js integration           # 只运行集成测试

组合选项:
  node run-tests.js MathContentDetector --coverage --verbose

可用的测试文件:
${this.testFiles.map((file, index) => `  ${index + 1}. ${file}`).join('\n')}

覆盖率报告:
  运行带 --coverage 参数的测试后，覆盖率报告将保存在:
  📊 HTML报告: ${path.join(this.testDir, 'coverage/lcov-report/index.html')}
  📋 文本报告: 直接显示在终端中
    `);
  }

  run() {
    const args = process.argv.slice(2);

    if (args.includes('--help') || args.includes('-h')) {
      this.showHelp();
      return;
    }

    this.printHeader();
    this.checkPrerequisites();
    console.log('\n');
    this.runTests(args);
  }
}

// 主程序入口
if (require.main === module) {
  const runner = new TestRunner();
  runner.run();
}

module.exports = TestRunner;