const MathContentDetector = require('../MathContentDetector.js');

describe('MathContentDetector', () => {
  let detector;

  beforeEach(() => {
    detector = new MathContentDetector();
  });

  describe('构造函数', () => {
    test('应该创建MathContentDetector实例', () => {
      expect(detector).toBeInstanceOf(MathContentDetector);
    });

    test('应该初始化必要的内部状态', () => {
      expect(detector).toBeDefined();
      expect(typeof detector.isMathEnvironment).toBe('function');
      expect(typeof detector.isInlineMath).toBe('function');
      expect(typeof detector.isDisplayMath).toBe('function');
      expect(typeof detector.extractMathBlocks).toBe('function');
      expect(typeof detector.classifyContent).toBe('function');
      expect(typeof detector.hasMathContent).toBe('function');
    });

    test('应该初始化数学环境列表', () => {
      expect(Array.isArray(detector.mathEnvironments)).toBe(true);
      expect(detector.mathEnvironments.length).toBeGreaterThan(0);
      expect(detector.mathEnvironments).toContain('equation');
      expect(detector.mathEnvironments).toContain('align');
      expect(detector.mathEnvironments).toContain('matrix');
    });

    test('应该初始化数学分隔符', () => {
      expect(Array.isArray(detector.inlineMathDelimiters)).toBe(true);
      expect(Array.isArray(detector.displayMathDelimiters)).toBe(true);
      expect(detector.inlineMathDelimiters.length).toBeGreaterThan(0);
      expect(detector.displayMathDelimiters.length).toBeGreaterThan(0);
    });
  });

  describe('isMathEnvironment(content)', () => {
    test('应该识别equation环境', () => {
      const text = `\\begin{equation}
E = mc^2
\\end{equation}`;
      
      const result = detector.isMathEnvironment(text);
      expect(result).toMatchObject({
        isMath: true,
        type: 'display_environment',
        environment: 'equation'
      });
    });

    test('应该识别align环境', () => {
      const text = `\\begin{align}
a &= b + c \\\\
d &= e + f
\\end{align}`;
      
      const result = detector.isMathEnvironment(text);
      expect(result).toMatchObject({
        isMath: true,
        type: 'display_environment', 
        environment: 'align'
      });
    });

    test('应该识别矩阵环境', () => {
      const matrixTypes = ['matrix', 'pmatrix', 'bmatrix', 'vmatrix', 'Vmatrix'];
      
      matrixTypes.forEach(matrixType => {
        const text = `\\begin{${matrixType}}
1 & 2 \\\\
3 & 4
\\end{${matrixType}}`;
        
        const result = detector.isMathEnvironment(text);
        expect(result).toMatchObject({
          isMath: true,
          type: 'display_environment',
          environment: matrixType
        });
      });
    });

    test('应该拒绝非数学环境', () => {
      const nonMathTexts = [
        '\\begin{document}content\\end{document}',
        '\\begin{section}title\\end{section}',
        '\\begin{itemize}\\item test\\end{itemize}',
        'Regular text without environments'
      ];

      nonMathTexts.forEach(text => {
        const result = detector.isMathEnvironment(text);
        expect(result).toMatchObject({
          isMath: false,
          type: 'non_math'
        });
        expect(result.environment).toBeUndefined();
      });
    });

    test('应该处理空输入', () => {
      const result = detector.isMathEnvironment('');
      expect(result).toMatchObject({
        isMath: false,
        type: 'non_math'
      });
    });

    test('应该处理无效输入', () => {
      const invalidInputs = [null, undefined, 123, {}];
      
      invalidInputs.forEach(input => {
        expect(() => {
          const result = detector.isMathEnvironment(input);
          expect(result).toMatchObject({
            isMath: false,
            type: 'non_math'
          });
        }).not.toThrow();
      });
    });
  });

  describe('isInlineMath(content)', () => {
    test('应该识别$包装的行内数学', () => {
      const texts = [
        'This is $x = y + z$ inline math.',
        'Multiple $a = b$ and $c = d$ formulas.',
        'Single $x$ variable.',
        'Complex $\\frac{a}{b} + \\sqrt{c}$ expression.'
      ];

      texts.forEach(text => {
        const result = detector.isInlineMath(text);
        expect(result).toMatchObject({
          isMath: true,
          type: 'inline_math'
        });
      });
    });

    test('应该识别\\(\\)包装的行内数学', () => {
      const text = 'This is \\(x = y + z\\) inline math.';
      const result = detector.isInlineMath(text);
      expect(result).toMatchObject({
        isMath: true,
        type: 'inline_math'
      });
    });

    test('应该拒绝非行内数学', () => {
      const nonInlineTexts = [
        'No math here.',
        '$$display math$$',
        '\\[display math\\]',
        'Price is \\$100.',
        'Regular text'
      ];

      nonInlineTexts.forEach(text => {
        const result = detector.isInlineMath(text);
        expect(result).toMatchObject({
          isMath: false,
          type: 'non_math'
        });
      });
    });

    test('应该处理转义的美元符号', () => {
      const text = 'Price is \\$100 and \\$200.';
      const result = detector.isInlineMath(text);
      expect(result).toMatchObject({
        isMath: false,
        type: 'non_math'
      });
    });

    test('应该处理空输入', () => {
      const result = detector.isInlineMath('');
      expect(result).toMatchObject({
        isMath: false,
        type: 'non_math'
      });
    });
  });

  describe('isDisplayMath(content)', () => {
    test('应该识别$$包装的显示数学', () => {
      const texts = [
        'This is $$E = mc^2$$ display math.',
        'Multiple $$a = b$$ and $$c = d$$ formulas.',
        'Complex $$\\frac{a}{b} + \\sqrt{c}$$ expression.'
      ];

      texts.forEach(text => {
        const result = detector.isDisplayMath(text);
        expect(result).toMatchObject({
          isMath: true,
          type: 'display_math'
        });
      });
    });

    test('应该识别\\[\\]包装的显示数学', () => {
      const text = 'This is \\[E = mc^2\\] display math.';
      const result = detector.isDisplayMath(text);
      expect(result).toMatchObject({
        isMath: true,
        type: 'display_math'
      });
    });

    test('应该拒绝非显示数学', () => {
      const nonDisplayTexts = [
        'No math here.',
        '$inline math$',
        '\\(inline math\\)',
        'Regular text'
      ];

      nonDisplayTexts.forEach(text => {
        const result = detector.isDisplayMath(text);
        expect(result).toMatchObject({
          isMath: false,
          type: 'non_math'
        });
      });
    });

    test('应该处理多行显示数学', () => {
      const text = `Display math:
$$
E = mc^2
F = ma
$$
End.`;
      
      const result = detector.isDisplayMath(text);
      expect(result).toMatchObject({
        isMath: true,
        type: 'display_math'
      });
    });

    test('应该处理空输入', () => {
      const result = detector.isDisplayMath('');
      expect(result).toMatchObject({
        isMath: false,
        type: 'non_math'
      });
    });
  });

  describe('extractMathBlocks(content)', () => {
    test('应该提取所有数学块', () => {
      const text = 'Mixed content with $inline$ and $$display$$ math.';
      const blocks = detector.extractMathBlocks(text);
      
      expect(Array.isArray(blocks)).toBe(true);
      expect(blocks.length).toBeGreaterThan(0);
      
      // 验证返回的块包含必要信息
      blocks.forEach(block => {
        expect(block).toHaveProperty('content');
        expect(block).toHaveProperty('type');
        expect(typeof block.content).toBe('string');
        expect(typeof block.type).toBe('string');
      });
    });

    test('应该区分行内和显示数学块', () => {
      const text = 'Text with $x = y$ inline and $$E = mc^2$$ display.';
      const blocks = detector.extractMathBlocks(text);
      
      expect(blocks.length).toBeGreaterThanOrEqual(2);
      
      const inlineBlocks = blocks.filter(block => block.type === 'inline');
      const displayBlocks = blocks.filter(block => block.type === 'display');
      
      expect(inlineBlocks.length).toBeGreaterThan(0);
      expect(displayBlocks.length).toBeGreaterThan(0);
    });

    test('应该处理无数学内容的文本', () => {
      const text = 'This text has no mathematical content.';
      const blocks = detector.extractMathBlocks(text);
      
      expect(Array.isArray(blocks)).toBe(true);
      expect(blocks.length).toBe(0);
    });

    test('应该处理空输入', () => {
      const blocks = detector.extractMathBlocks('');
      expect(Array.isArray(blocks)).toBe(true);
      expect(blocks.length).toBe(0);
    });

    test('应该处理复杂的数学表达式', () => {
      const text = 'Complex: $$\\sum_{i=1}^{n} \\frac{1}{i^2} = \\frac{\\pi^2}{6}$$ formula.';
      const blocks = detector.extractMathBlocks(text);
      
      expect(blocks.length).toBe(1);
      expect(blocks[0].type).toBe('display');
      expect(blocks[0].content).toContain('\\sum');
      expect(blocks[0].content).toContain('\\frac');
    });
  });

  describe('classifyContent(content)', () => {
    test('应该分类数学环境内容', () => {
      const text = `\\begin{equation}
E = mc^2
\\end{equation}`;
      
      const result = detector.classifyContent(text);
      expect(result).toHaveProperty('type');
      expect(result).toHaveProperty('confidence');
      expect(typeof result.confidence).toBe('number');
      expect(result.confidence).toBeGreaterThanOrEqual(0);
      expect(result.confidence).toBeLessThanOrEqual(1);
    });

    test('应该分类行内数学内容', () => {
      const text = 'This has $x = y$ math.';
      const result = detector.classifyContent(text);
      
      expect(result.type).toMatch(/math|inline/);
      expect(result.confidence).toBeGreaterThan(0);
    });

    test('应该分类显示数学内容', () => {
      const text = 'This has $$E = mc^2$$ math.';
      const result = detector.classifyContent(text);
      
      expect(result.type).toMatch(/math|display/);
      expect(result.confidence).toBeGreaterThan(0);
    });

    test('应该分类非数学内容', () => {
      const text = 'This is just regular text with no mathematics.';
      const result = detector.classifyContent(text);
      
      expect(result.type).toMatch(/non_math|text/);
      expect(result.confidence).toBeGreaterThanOrEqual(0);
    });

    test('应该提供置信度评分', () => {
      const testCases = [
        'Pure math: $$E = mc^2$$',
        'Mixed: some text with $x = y$ formula',
        'No math content here'
      ];

      testCases.forEach(text => {
        const result = detector.classifyContent(text);
        expect(result.confidence).toBeGreaterThanOrEqual(0);
        expect(result.confidence).toBeLessThanOrEqual(1);
      });
    });

    test('应该处理空输入', () => {
      const result = detector.classifyContent('');
      expect(result.confidence).toBe(0);
    });
  });

  describe('hasMathContent(content)', () => {
    test('应该检测包含数学的内容', () => {
      const mathTexts = [
        'Inline: $x = y$',
        'Display: $$E = mc^2$$',
        'Environment: \\begin{equation}F = ma\\end{equation}',
        'Multiple: $a = b$ and $$c = d$$'
      ];

      mathTexts.forEach(text => {
        const hasMath = detector.hasMathContent(text);
        expect(hasMath).toBe(true);
      });
    });

    test('应该拒绝无数学内容', () => {
      const nonMathTexts = [
        'Regular text only',
        'Price is \\$100',
        'No mathematical symbols',
        ''
      ];

      nonMathTexts.forEach(text => {
        const hasMath = detector.hasMathContent(text);
        expect(hasMath).toBe(false);
      });
    });

    test('应该处理复杂混合内容', () => {
      const text = `
Document with text and mathematics:
Regular paragraph here.

\\begin{equation}
E = mc^2
\\end{equation}

More text with $inline = math$ here.
`;
      
      const hasMath = detector.hasMathContent(text);
      expect(hasMath).toBe(true);
    });
  });

  describe('错误处理和边界条件', () => {
    test('应该处理无效输入而不抛出异常', () => {
      const invalidInputs = [null, undefined, {}, [], 123, true];

      invalidInputs.forEach(input => {
        expect(() => {
          detector.isMathEnvironment(input);
          detector.isInlineMath(input);
          detector.isDisplayMath(input);
          detector.extractMathBlocks(input);
          detector.classifyContent(input);
          detector.hasMathContent(input);
        }).not.toThrow();
      });
    });

    test('应该处理格式错误的LaTeX', () => {
      const malformedTexts = [
        '\\begin{equation}no end tag',
        '$incomplete inline',
        '$$incomplete display',
        '\\begin{unknown}content\\end{unknown}'
      ];

      malformedTexts.forEach(text => {
        expect(() => {
          const mathEnv = detector.isMathEnvironment(text);
          const inline = detector.isInlineMath(text);
          const display = detector.isDisplayMath(text);
          const blocks = detector.extractMathBlocks(text);
          
          expect(typeof mathEnv.isMath).toBe('boolean');
          expect(typeof inline.isMath).toBe('boolean');
          expect(typeof display.isMath).toBe('boolean');
          expect(Array.isArray(blocks)).toBe(true);
        }).not.toThrow();
      });
    });

    test('应该处理特殊字符和Unicode', () => {
      const specialTexts = [
        'Math with special: $α + β = γ$',
        'Unicode: $$∑_{i=1}^∞ x_i$$',
        'Accents: $\\acute{x} + \\grave{y}$'
      ];

      specialTexts.forEach(text => {
        expect(() => {
          const blocks = detector.extractMathBlocks(text);
          const hasMath = detector.hasMathContent(text);
          
          expect(Array.isArray(blocks)).toBe(true);
          expect(typeof hasMath).toBe('boolean');
        }).not.toThrow();
      });
    });
  });

  describe('性能测试', () => {
    test('应该高效处理大文档', () => {
      const baseText = 'Text with $E = mc^2$ and $$F = ma$$ formulas. ';
      const largeText = baseText.repeat(500);

      const startTime = Date.now();
      const blocks = detector.extractMathBlocks(largeText);
      const hasMath = detector.hasMathContent(largeText);
      const endTime = Date.now();

      expect(Array.isArray(blocks)).toBe(true);
      expect(hasMath).toBe(true);
      expect(endTime - startTime).toBeLessThan(1000); // 1秒内完成
    });

    test('应该处理深度嵌套结构', () => {
      const nestedText = `
\\begin{equation}
\\begin{matrix}
\\begin{pmatrix}
1 & 2 \\\\
3 & 4
\\end{pmatrix}
\\end{matrix}
\\end{equation}
`;

      expect(() => {
        const result = detector.isMathEnvironment(nestedText);
        const blocks = detector.extractMathBlocks(nestedText);
        
        expect(result.isMath).toBe(true);
        expect(Array.isArray(blocks)).toBe(true);
      }).not.toThrow();
    });
  });
});