const DollarDollarProtector = require('../DollarDollarProtector.js');

describe('DollarDollarProtector', () => {
  let protector;

  beforeEach(() => {
    protector = new DollarDollarProtector();
    // 禁用测试期间的控制台日志
    process.env.NODE_ENV = 'test';
  });

  afterEach(() => {
    // 清理环境变量
    delete process.env.NODE_ENV;
  });

  describe('构造函数', () => {
    test('应该创建DollarDollarProtector实例', () => {
      expect(protector).toBeInstanceOf(DollarDollarProtector);
    });

    test('应该初始化必要的内部状态', () => {
      expect(protector).toBeDefined();
      expect(typeof protector.protect).toBe('function');
      expect(typeof protector.restore).toBe('function');
      expect(typeof protector.hasDoubleDollarMath).toBe('function');
      expect(typeof protector.getProtectionCount).toBe('function');
      expect(typeof protector.clear).toBe('function');
    });

    test('应该初始化保护配置', () => {
      expect(protector.protectionPrefix).toBeDefined();
      expect(protector.protectionSuffix).toBeDefined();
      expect(typeof protector.protectionPrefix).toBe('string');
      expect(typeof protector.protectionSuffix).toBe('string');
    });

    test('应该初始化空的保护记录', () => {
      expect(Array.isArray(protector.protections)).toBe(true);
      expect(protector.protections).toHaveLength(0);
      expect(protector.getProtectionCount()).toBe(0);
    });
  });

  describe('protect(text)', () => {
    test('应该保护单个$$公式', () => {
      const text = 'This is $$E = mc^2$$ formula.';
      const result = protector.protect(text);

      expect(typeof result).toBe('string');
      expect(result).not.toContain('$$E = mc^2$$');
      expect(result).toContain(protector.protectionPrefix);
      expect(result).toContain(protector.protectionSuffix);
      expect(protector.getProtectionCount()).toBe(1);
    });

    test('应该保护多个$$公式', () => {
      const text = 'First $$a = b$$ and second $$c = d$$ formulas.';
      const result = protector.protect(text);

      expect(protector.getProtectionCount()).toBe(2);
      expect(result).not.toContain('$$a = b$$');
      expect(result).not.toContain('$$c = d$$');
      
      // 应该有两个不同的占位符
      const matches = result.match(new RegExp(`${protector.protectionPrefix.replace(/[.*+?^${}()|[\]\\]/g, '\\$&')}\\d+${protector.protectionSuffix.replace(/[.*+?^${}()|[\]\\]/g, '\\$&')}`, 'g'));
      expect(matches).toHaveLength(2);
    });

    test('应该生成唯一的占位符', () => {
      const text = 'Multiple formulas: $$x = y$$ and $$a = b$$ and $$p = q$$.';
      const result = protector.protect(text);

      expect(protector.getProtectionCount()).toBe(3);
      
      const placeholders = protector.protections.map(p => p.placeholder);
      const uniquePlaceholders = [...new Set(placeholders)];
      expect(placeholders.length).toBe(uniquePlaceholders.length);
    });

    test('应该正确记录保护信息', () => {
      const text = 'Formula: $$E = mc^2$$ here.';
      protector.protect(text);

      expect(protector.protections).toHaveLength(1);
      const protection = protector.protections[0];
      
      expect(protection).toHaveProperty('placeholder');
      expect(protection).toHaveProperty('content');
      expect(protection).toHaveProperty('type');
      expect(protection.content).toBe('$$E = mc^2$$');
      expect(protection.type).toBe('display_math');
      expect(protection.placeholder).toContain(protector.protectionPrefix);
    });

    test('应该处理多行公式', () => {
      const text = `Text before
$$
E = mc^2
F = ma
$$
text after.`;
      
      const result = protector.protect(text);
      expect(protector.getProtectionCount()).toBe(1);
      expect(result).not.toContain('E = mc^2');
      expect(protector.protections[0].content).toContain('E = mc^2');
      expect(protector.protections[0].content).toContain('F = ma');
    });

    test('应该处理复杂数学表达式', () => {
      const text = 'Complex: $$\\sum_{i=1}^{n} \\frac{1}{i^2} = \\frac{\\pi^2}{6}$$ formula.';
      const result = protector.protect(text);

      expect(protector.getProtectionCount()).toBe(1);
      expect(result).not.toContain('\\sum');
      expect(protector.protections[0].content).toContain('\\sum');
      expect(protector.protections[0].content).toContain('\\frac');
    });

    test('应该处理空输入', () => {
      const result = protector.protect('');
      expect(result).toBe('');
      expect(protector.getProtectionCount()).toBe(0);
    });

    test('应该处理无$$公式的文本', () => {
      const text = 'This text has no dollar-dollar formulas.';
      const result = protector.protect(text);

      expect(result).toBe(text);
      expect(protector.getProtectionCount()).toBe(0);
    });

    test('应该处理null和undefined输入', () => {
      expect(protector.protect(null)).toBe(null);
      expect(protector.protect(undefined)).toBe(undefined);
      expect(protector.getProtectionCount()).toBe(0);
    });

    test('应该重置保护记录在每次调用时', () => {
      protector.protect('First: $$a = b$$.');
      expect(protector.getProtectionCount()).toBe(1);
      
      protector.protect('Second: $$c = d$$.');
      expect(protector.getProtectionCount()).toBe(1); // 应该重置
      expect(protector.protections[0].content).toBe('$$c = d$$');
    });

    test('应该处理连续的$$公式', () => {
      const text = 'Formulas: $$a = b$$ $$c = d$$ together.';
      const result = protector.protect(text);

      expect(protector.getProtectionCount()).toBe(2);
      expect(result).not.toContain('$$a = b$$');
      expect(result).not.toContain('$$c = d$$');
    });
  });

  describe('restore(text)', () => {
    test('应该恢复受保护的$$公式', () => {
      const originalText = 'This is $$E = mc^2$$ formula.';
      const protectedText = protector.protect(originalText);
      const restoredText = protector.restore(protectedText);

      expect(restoredText).toBe(originalText);
    });

    test('应该恢复多个受保护的公式', () => {
      const originalText = 'First $$a = b$$ and second $$c = d$$ formulas.';
      const protectedText = protector.protect(originalText);
      const restoredText = protector.restore(protectedText);

      expect(restoredText).toBe(originalText);
    });

    test('应该按正确顺序恢复公式', () => {
      const originalText = 'Formula 1: $$x = 1$$ and formula 2: $$y = 2$$ and formula 3: $$z = 3$$.';
      const protectedText = protector.protect(originalText);
      const restoredText = protector.restore(protectedText);

      expect(restoredText).toBe(originalText);
      expect(restoredText).toContain('$$x = 1$$');
      expect(restoredText).toContain('$$y = 2$$');
      expect(restoredText).toContain('$$z = 3$$');
    });

    test('应该处理无占位符的文本', () => {
      const text = 'This text has no protected formulas.';
      const restoredText = protector.restore(text);

      expect(restoredText).toBe(text);
    });

    test('应该处理空保护记录', () => {
      const text = 'Some text with fake placeholder.';
      protector.clear(); // 确保没有保护记录
      const restoredText = protector.restore(text);

      expect(restoredText).toBe(text);
    });

    test('应该处理空输入', () => {
      expect(protector.restore('')).toBe('');
      expect(protector.restore(null)).toBe(null);
      expect(protector.restore(undefined)).toBe(undefined);
    });

    test('应该处理复杂数学公式的恢复', () => {
      const originalText = 'Complex: $$\\sum_{i=1}^{n} \\frac{1}{i^2} = \\frac{\\pi^2}{6}$$ formula.';
      const protectedText = protector.protect(originalText);
      const restoredText = protector.restore(protectedText);

      expect(restoredText).toBe(originalText);
      expect(restoredText).toContain('\\sum_{i=1}^{n}');
      expect(restoredText).toContain('\\frac{\\pi^2}{6}');
    });

    test('应该正确处理JavaScript特殊字符', () => {
      const originalText = 'Formula with $1 and $& symbols: $$x = $y + $z$$ end.';
      const protectedText = protector.protect(originalText);
      const restoredText = protector.restore(protectedText);

      expect(restoredText).toBe(originalText);
      expect(restoredText).toContain('$1');
      expect(restoredText).toContain('$&');
    });
  });

  describe('hasDoubleDollarMath(text)', () => {
    test('应该检测包含$$数学的文本', () => {
      const mathTexts = [
        'Formula: $$E = mc^2$$ here.',
        'Multiple $$a = b$$ and $$c = d$$.',
        'Multiline: $$\nE = mc^2\n$$'
      ];

      mathTexts.forEach(text => {
        expect(protector.hasDoubleDollarMath(text)).toBe(true);
      });
    });

    test('应该拒绝无$$数学的文本', () => {
      const nonMathTexts = [
        'No math here.',
        'Inline math: $x = y$',
        'Escaped: \\$\\$not math\\$\\$',
        ''
      ];

      nonMathTexts.forEach(text => {
        expect(protector.hasDoubleDollarMath(text)).toBe(false);
      });
    });

    test('应该处理null和undefined', () => {
      expect(protector.hasDoubleDollarMath(null)).toBe(false);
      expect(protector.hasDoubleDollarMath(undefined)).toBe(false);
      expect(protector.hasDoubleDollarMath('')).toBe(false);
    });

    test('应该检测复杂的$$公式', () => {
      const complexTexts = [
        'Complex: $$\\frac{a}{b} + \\sqrt{c}$$',
        'Matrix: $$\\begin{pmatrix} 1 & 2 \\\\ 3 & 4 \\end{pmatrix}$$',
        'Long formula: $$\\sum_{i=1}^{n} \\int_0^1 f(x) dx$$'
      ];

      complexTexts.forEach(text => {
        expect(protector.hasDoubleDollarMath(text)).toBe(true);
      });
    });
  });

  describe('getProtectionCount()', () => {
    test('应该返回正确的保护数量', () => {
      expect(protector.getProtectionCount()).toBe(0);
      
      protector.protect('Formula: $$E = mc^2$$.');
      expect(protector.getProtectionCount()).toBe(1);
      
      protector.protect('Two: $$a = b$$ and $$c = d$$.');
      expect(protector.getProtectionCount()).toBe(2);
    });

    test('应该在clear后返回0', () => {
      protector.protect('Formula: $$E = mc^2$$.');
      expect(protector.getProtectionCount()).toBe(1);
      
      protector.clear();
      expect(protector.getProtectionCount()).toBe(0);
    });

    test('应该在无数学内容时返回0', () => {
      protector.protect('No math content here.');
      expect(protector.getProtectionCount()).toBe(0);
    });
  });

  describe('clear()', () => {
    test('应该清除所有保护记录', () => {
      protector.protect('Formula: $$E = mc^2$$.');
      expect(protector.getProtectionCount()).toBe(1);

      protector.clear();
      expect(protector.getProtectionCount()).toBe(0);
      expect(protector.protections).toHaveLength(0);
    });

    test('应该重置计数器', () => {
      protector.protect('First: $$a = b$$.');
      const firstPlaceholder = protector.protections[0].placeholder;
      
      protector.clear();
      protector.protect('After clear: $$x = y$$.');
      const secondPlaceholder = protector.protections[0].placeholder;
      
      // 计数器应该重置，所以占位符应该相同
      expect(secondPlaceholder).toBe(firstPlaceholder);
    });

    test('应该处理空保护记录的清除', () => {
      expect(() => {
        protector.clear();
      }).not.toThrow();

      expect(protector.getProtectionCount()).toBe(0);
    });

    test('应该允许清除后正常使用', () => {
      protector.protect('Before: $$a = b$$.');
      protector.clear();
      
      const text = 'After: $$c = d$$.';
      const result = protector.protect(text);
      const restored = protector.restore(result);
      
      expect(restored).toBe(text);
      expect(protector.getProtectionCount()).toBe(1);
    });
  });

  describe('往返转换一致性', () => {
    test('应该保持简单公式的完整性', () => {
      const texts = [
        'Simple: $$x = y$$',
        'Multiple: $$a = b$$ and $$c = d$$',
        'Complex: $$\\frac{d}{dx}\\int_0^x f(t) dt = f(x)$$'
      ];

      texts.forEach(originalText => {
        const protectedText = protector.protect(originalText);
        const restoredText = protector.restore(protectedText);
        expect(restoredText).toBe(originalText);
      });
    });

    test('应该处理包含特殊字符的公式', () => {
      const originalText = 'Special chars: $$α + β = γ$$ and $$∑_{i=1}^∞ x_i$$';
      const protectedText = protector.protect(originalText);
      const restoredText = protector.restore(protectedText);
      
      expect(restoredText).toBe(originalText);
    });

    test('应该处理嵌套LaTeX结构', () => {
      const originalText = `Nested:
$$
\\begin{align}
x &= \\frac{a + b}{c} \\\\
y &= \\sqrt{x^2 + z^2}
\\end{align}
$$`;
      
      const protectedText = protector.protect(originalText);
      const restoredText = protector.restore(protectedText);
      
      expect(restoredText).toBe(originalText);
    });

    test('应该处理JavaScript特殊替换字符', () => {
      const texts = [
        'Dollar signs: $$cost = \\$100 + \\$200$$',
        'Replacement patterns: $$value = $1 + $& + $`$$',
        'Mixed: $$x = f($0) + g($1)$$'
      ];

      texts.forEach(originalText => {
        const protectedText = protector.protect(originalText);
        const restoredText = protector.restore(protectedText);
        expect(restoredText).toBe(originalText);
      });
    });
  });

  describe('错误处理和边界条件', () => {
    test('应该处理无效输入而不抛出异常', () => {
      const invalidInputs = [null, undefined, {}, [], 123, true];

      invalidInputs.forEach(input => {
        expect(() => {
          protector.protect(input);
          protector.restore(input);
          protector.hasDoubleDollarMath(input);
        }).not.toThrow();
      });
    });

    test('应该处理不匹配的$$符号', () => {
      const problematicTexts = [
        'Single $$ only',
        'Triple $$$ symbols',
        'Incomplete $$formula without end',
        'End without start formula$$'
      ];

      problematicTexts.forEach(text => {
        expect(() => {
          const result = protector.protect(text);
          expect(typeof result).toBe('string');
        }).not.toThrow();
      });
    });

    test('应该处理极长的公式', () => {
      const longFormula = '$$' + 'x + y + z + '.repeat(1000) + 'w$$';
      
      expect(() => {
        const protectedText = protector.protect(longFormula);
        const restoredText = protector.restore(protectedText);
        expect(restoredText).toBe(longFormula);
      }).not.toThrow();
    });

    test('应该处理空格和换行的各种组合', () => {
      const texts = [
        '$$\n\nE = mc^2\n\n$$',
        '$$   a = b   $$',
        '$$\t\tx = y\t\t$$',
        '$$\r\nformula\r\n$$'
      ];

      texts.forEach(originalText => {
        const protectedText = protector.protect(originalText);
        const restoredText = protector.restore(protectedText);
        expect(restoredText).toBe(originalText);
      });
    });
  });

  describe('性能和内存管理', () => {
    test('应该高效处理大量公式', () => {
      const baseText = 'Formula ';
      const formulas = Array.from({length: 100}, (_, i) => `$$x_{${i}} = y_{${i}}$$`);
      const largeText = formulas.join(baseText);

      const startTime = Date.now();
      const protectedText = protector.protect(largeText);
      const restoredText = protector.restore(protectedText);
      const endTime = Date.now();

      expect(restoredText).toBe(largeText);
      expect(protector.getProtectionCount()).toBe(100);
      expect(endTime - startTime).toBeLessThan(1000); // 1秒内完成
    });

    test('应该正确管理内存使用', () => {
      // 创建大量保护记录
      for (let i = 0; i < 50; i++) {
        protector.protect(`Formula ${i}: $$x_${i} = y_${i}$$.`);
      }

      expect(protector.getProtectionCount()).toBe(50);

      // 清理应该释放内存
      protector.clear();
      expect(protector.getProtectionCount()).toBe(0);
      expect(protector.protections).toHaveLength(0);

      // 后续操作应该正常
      const newText = 'New: $$a = b$$.';
      const result = protector.protect(newText);
      const restored = protector.restore(result);
      expect(restored).toBe(newText);
    });

    test('应该处理重复保护和恢复', () => {
      const text = 'Formula: $$E = mc^2$$';
      
      // 多次保护-恢复循环
      for (let i = 0; i < 10; i++) {
        const protected1 = protector.protect(text);
        const restored1 = protector.restore(protected1);
        expect(restored1).toBe(text);
      }
    });
  });

  describe('占位符格式一致性', () => {
    test('应该生成正确格式的占位符', () => {
      protector.protect('Formula: $$E = mc^2$$.');
      const placeholder = protector.protections[0].placeholder;
      
      expect(placeholder).toContain(protector.protectionPrefix);
      expect(placeholder).toContain(protector.protectionSuffix);
      expect(placeholder).toMatch(/\d+/); // 应该包含数字
    });

    test('应该确保占位符唯一性', () => {
      const text = 'Multiple: $$a$$ $$b$$ $$c$$ formulas.';
      protector.protect(text);

      const placeholders = protector.protections.map(p => p.placeholder);
      const uniquePlaceholders = [...new Set(placeholders)];
      
      expect(placeholders.length).toBe(uniquePlaceholders.length);
      expect(placeholders).toHaveLength(3);
    });

    test('应该使用自动计数器', () => {
      protector.clear(); // 确保计数器从0开始
      
      protector.protect('First: $$a$$.');
      protector.protect('Second: $$b$$.');
      protector.protect('Third: $$c$$.');

      const placeholders = protector.protections.map(p => p.placeholder);
      expect(placeholders[0]).toContain('0');
    });
  });
});