# Utils 模块接口文档

## 模块概述

Utils 模块提供LaTeX Parser V3的核心工具类，包括数学内容检测、占位符保护等关键功能。这些工具被各层解析器广泛使用，确保解析过程的一致性和准确性。

## 公共接口

### MathContentDetector

**位置**: `utils/MathContentDetector.js`

**职责**: 统一的数学内容识别和分类机制，为五层解析器提供一致的数学公式检测功能

**构造函数**:
```javascript
new MathContentDetector()
```

**核心方法**:
```javascript
detectMathEnvironments(text: String): Array<MathEnvironment>
extractMathBlocks(text: String): Array<MathBlock>  
isMathEnvironment(envName: String): Boolean
hasInlineMath(text: String): Boolean
hasDisplayMath(text: String): Boolean
classifyMathContent(content: String): MathClassification
```

**数学环境支持**:
- 显示数学环境: `equation`, `equation*`, `align`, `align*`, `gather`, `gather*`, `multline`, `multline*`
- 矩阵环境: `matrix`, `pmatrix`, `bmatrix`, `vmatrix`, `Vmatrix`
- 数组环境: `array`, `eqnarray`, `eqnarray*`
- 特殊环境: `displaymath`, `split`

**输出格式**:
```javascript
// MathEnvironment
{
  type: String,           // 环境类型 ('equation', 'align', etc.)
  content: String,        // 数学内容
  startPos: Number,       // 开始位置
  endPos: Number,        // 结束位置
  isDisplay: Boolean     // 是否为显示数学
}

// MathBlock  
{
  originalContent: String, // 原始内容
  mathContent: String,    // 纯数学内容
  type: String,          // 'inline' 或 'display'
  startIndex: Number,    // 开始索引
  endIndex: Number      // 结束索引
}

// MathClassification
{
  type: String,          // 'inline', 'display', 'environment', 'none'
  confidence: Number,    // 置信度 (0-1)
  details: Object       // 详细信息
}
```

### DollarDollarProtector

**位置**: `utils/DollarDollarProtector.js`

**职责**: 保护$$包装的数学公式，防止在多层解析过程中被意外修改

**构造函数**:
```javascript
new DollarDollarProtector()
```

**核心方法**:
```javascript
protect(text: String): ProtectionResult
restore(text: String): String
isProtected(text: String): Boolean
getProtections(): Array<Protection>
clearProtections(): void
```

**保护机制**:
- 占位符格式: `__DD_PROTECTED_{id}__`
- 自动计数器: 确保占位符唯一性
- JavaScript $符特殊字符处理: 使用函数返回值避免$被解释

**输出格式**:
```javascript
// ProtectionResult
{
  text: String,           // 替换后的文本
  protections: Array,     // 保护记录数组
  count: Number          // 保护数量
}

// Protection
{
  id: String,            // 保护ID
  placeholder: String,   // 占位符
  originalContent: String, // 原始内容
  type: String          // 保护类型
}
```

## 详细接口规范

### MathContentDetector 方法详情

#### detectMathEnvironments(text)

检测文本中的数学环境。

**参数**:
- `text` {String} - 待检测的文本

**返回值**: {Array<MathEnvironment>} - 数学环境数组

**示例**:
```javascript
const detector = new MathContentDetector();
const environments = detector.detectMathEnvironments(`
\\begin{equation}
E = mc^2
\\end{equation}
`);
// 返回: [{ type: 'equation', content: 'E = mc^2', ... }]
```

#### extractMathBlocks(text)

提取所有数学块（行内和显示）。

**参数**:
- `text` {String} - 待处理文本

**返回值**: {Array<MathBlock>} - 数学块数组

#### isMathEnvironment(envName)

检查给定名称是否为数学环境。

**参数**:
- `envName` {String} - 环境名称

**返回值**: {Boolean} - 是否为数学环境

#### classifyMathContent(content)

对数学内容进行分类和评估。

**参数**:
- `content` {String} - 数学内容

**返回值**: {MathClassification} - 分类结果

### DollarDollarProtector 方法详情

#### protect(text)

保护文本中的$$数学公式。

**参数**:
- `text` {String} - 包含$$公式的文本

**返回值**: {ProtectionResult} - 保护结果

**示例**:
```javascript
const protector = new DollarDollarProtector();
const result = protector.protect('This is $$E = mc^2$$ formula');
// result.text: 'This is __DD_PROTECTED_0__ formula'
```

#### restore(text)

恢复被保护的$$公式。

**参数**:
- `text` {String} - 包含占位符的文本

**返回值**: {String} - 恢复后的文本

## 使用模式

### 在解析器中的使用

各层解析器应该按以下模式使用工具类：

```javascript
class LayerParser {
  constructor() {
    this.mathDetector = new MathContentDetector();
    this.protector = new DollarDollarProtector();
  }
  
  process(text, existingPlaceholders) {
    // 1. 保护数学内容
    const protectedResult = this.protector.protect(text);
    let processedText = protectedResult.text;
    
    // 2. 检测剩余数学内容
    const mathEnvs = this.mathDetector.detectMathEnvironments(processedText);
    
    // 3. 处理非数学内容
    // ... 解析逻辑 ...
    
    // 4. 恢复保护的内容
    const finalText = this.protector.restore(processedText);
    
    return { text: finalText, ... };
  }
}
```

### 数学内容检测模式

```javascript
const detector = new MathContentDetector();

// 检测环境类型
if (detector.isMathEnvironment('equation')) {
  // 按数学环境处理
}

// 分类内容
const classification = detector.classifyMathContent(content);
switch (classification.type) {
  case 'display':
    // 显示数学处理
    break;
  case 'inline':  
    // 行内数学处理
    break;
  case 'none':
    // 非数学内容处理
    break;
}
```

## 依赖关系

### 输入依赖
- 无外部依赖
- 提供基础工具功能

### 输出依赖
- `core/DocumentParser.js` - 使用MathContentDetector
- `core/BlockElementParser.js` - 使用DollarDollarProtector  
- `core/InlineElementParser.js` - 使用MathContentDetector
- `core/MathParser.js` - 使用两个工具类
- `core/SpecialParser.js` - 使用保护机制

## 错误处理约定

### MathContentDetector
- 无效输入时返回空数组
- 解析错误时在warnings中记录详情
- 继续处理，不中断解析流程

### DollarDollarProtector
- 保护失败时记录警告，返回原文本
- 恢复时找不到对应保护记录则跳过
- 保证输入输出的文本长度一致性

## 性能优化

### 缓存机制
- MathContentDetector支持结果缓存
- DollarDollarProtector重用占位符模式
- 正则表达式预编译优化

### 内存管理
- 及时清理保护记录
- 避免循环引用
- 支持分批处理大文档

## 扩展性设计

- 支持添加新的数学环境类型
- 允许自定义保护策略
- 提供插件接口扩展检测功能
- 支持异步处理模式