/**
 * 数学内容识别工具类
 * 为五层解析器架构提供统一的数学公式识别和分类机制
 * 
 * TDD要求：每层解析器都应该使用此工具来区分数学vs非数学内容
 * 核心原则：数学内容保持LaTeX格式输出$$包装，非数学内容转换为HTML
 */

class MathContentDetector {
  constructor() {
    // 数学环境列表（显示数学）- 排除矩阵环境
    this.mathEnvironments = [
      'equation', 'equation*',
      'align', 'align*',
      'alignat', 'alignat*',
      'gather', 'gather*',
      'multline', 'multline*',
      'displaymath',
      'eqnarray', 'eqnarray*',
      'split', 'array'
      // 移除矩阵环境：'matrix', 'pmatrix', 'bmatrix', 'vmatrix', 'Bmatrix', 'Vmatrix'
      // 这些环境由KaTeX的宏定义处理，不应在此处检测
    ];
    
    // 行内数学模式分隔符
    this.inlineMathDelimiters = [
      { start: '$', end: '$' },
      { start: '\\(', end: '\\)' }
    ];
    
    // 显示数学模式分隔符
    this.displayMathDelimiters = [
      { start: '$$', end: '$$' },
      { start: '\\[', end: '\\]' }
    ];
    
    // 数学命令（不是环境，但是数学相关）
    this.mathCommands = [
      'frac', 'sqrt', 'sum', 'int', 'prod', 'lim',
      'sin', 'cos', 'tan', 'log', 'exp', 'ln',
      'alpha', 'beta', 'gamma', 'delta', 'epsilon', 'theta',
      'lambda', 'mu', 'pi', 'sigma', 'tau', 'phi',
      'infty', 'partial', 'nabla', 'times', 'cdot'
    ];
  }

  /**
   * 判断给定内容是否为数学环境
   * @param {string} content - 要检查的内容
   * @returns {Object} { isMath: boolean, type: string, environment?: string }
   */
  isMathEnvironment(content) {
    const envPattern = /\\begin\{([^}]+)\}/;
    const match = content.match(envPattern);
    
    if (match) {
      const envName = match[1];
      if (this.mathEnvironments.includes(envName)) {
        return {
          isMath: true,
          type: 'display_environment',
          environment: envName
        };
      }
    }
    
    return { isMath: false, type: 'non_math' };
  }

  /**
   * 判断给定内容是否为行内数学公式
   * @param {string} content - 要检查的内容
   * @returns {Object} { isMath: boolean, type: string }
   */
  isInlineMath(content) {
    for (const delimiter of this.inlineMathDelimiters) {
      const pattern = new RegExp(`\\${delimiter.start}[^$]+\\${delimiter.end}`);
      if (pattern.test(content)) {
        return {
          isMath: true,
          type: 'inline_math'
        };
      }
    }
    
    return { isMath: false, type: 'non_math' };
  }

  /**
   * 判断给定内容是否为显示数学公式
   * @param {string} content - 要检查的内容
   * @returns {Object} { isMath: boolean, type: string }
   */
  isDisplayMath(content) {
    for (const delimiter of this.displayMathDelimiters) {
      const start = delimiter.start.replace(/\$/g, '\\$');
      const end = delimiter.end.replace(/\$/g, '\\$');
      const pattern = new RegExp(`${start}[\\s\\S]+?${end}`);
      if (pattern.test(content)) {
        return {
          isMath: true,
          type: 'display_math'
        };
      }
    }
    
    return { isMath: false, type: 'non_math' };
  }

  /**
   * 判断LaTeX命令是否为数学相关
   * @param {string} command - LaTeX命令名（不含反斜杠）
   * @returns {Object} { isMath: boolean, type: string }
   */
  isMathCommand(command) {
    if (this.mathCommands.includes(command)) {
      return {
        isMath: true,
        type: 'math_command'
      };
    }
    
    return { isMath: false, type: 'non_math' };
  }

  /**
   * 综合判断内容类型
   * @param {string} content - 要检查的内容
   * @returns {Object} { isMath: boolean, type: string, environment?: string, needsWrapper?: string }
   */
  classifyContent(content) {
    // 1. 检查是否是数学环境
    const envCheck = this.isMathEnvironment(content);
    if (envCheck.isMath) {
      return {
        ...envCheck,
        needsWrapper: '$$'  // 数学环境需要$$包装
      };
    }
    
    // 2. 检查是否是显示数学
    const displayCheck = this.isDisplayMath(content);
    if (displayCheck.isMath) {
      return {
        ...displayCheck,
        needsWrapper: 'none'  // 已经有分隔符了
      };
    }
    
    // 3. 检查是否是行内数学
    const inlineCheck = this.isInlineMath(content);
    if (inlineCheck.isMath) {
      return {
        ...inlineCheck,
        needsWrapper: 'none'  // 已经有分隔符了
      };
    }
    
    // 4. 默认为非数学内容
    return {
      isMath: false,
      type: 'non_math',
      needsWrapper: 'html'  // 需要转换为HTML
    };
  }

  /**
   * 为数学内容添加适当的包装
   * @param {string} content - 原始内容
   * @param {string} wrapperType - 包装类型 ('$$', '$', 'none', 'html')
   * @returns {string} 包装后的内容
   */
  wrapMathContent(content, wrapperType) {
    switch (wrapperType) {
      case '$$':
        return `$$${content}$$`;
      case '$':
        return `$${content}$`;
      case 'none':
        return content;
      case 'html':
      default:
        return content;  // HTML转换由各层解析器负责
    }
  }

  /**
   * 提取内容中的所有数学块
   * @param {string} content - 输入内容
   * @returns {Array} 数学块列表 [{ type, content, startIndex, endIndex }]
   */
  extractMathBlocks(content) {
    const mathBlocks = [];
    
    // 1. 提取数学环境 - 修复：使用更可靠的方法逐个环境检测
    this.mathEnvironments.forEach(envName => {
      // 为每个环境创建专门的正则表达式
      // 需要转义环境名中的特殊字符（如 * ）
      const escapedEnv = envName.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
      const envPattern = new RegExp(
        `\\\\begin\\{${escapedEnv}\\}([\\s\\S]*?)\\\\end\\{${escapedEnv}\\}`,
        'g'
      );
      
      let match;
      while ((match = envPattern.exec(content)) !== null) {
        mathBlocks.push({
          type: 'display_environment',
          environment: envName,
          content: match[0],
          startIndex: match.index,
          endIndex: match.index + match[0].length
        });
      }
    });
    
    // 2. 提取行内数学（避免与环境重叠）
    const inlinePattern = /\$([^$]+)\$/g;
    let match;
    while ((match = inlinePattern.exec(content)) !== null) {
      // 检查是否与现有数学环境重叠
      const overlaps = mathBlocks.some(block => 
        match.index >= block.startIndex && match.index < block.endIndex
      );
      
      if (!overlaps) {
        mathBlocks.push({
          type: 'inline_math',
          content: match[0],
          startIndex: match.index,
          endIndex: match.index + match[0].length
        });
      }
    }
    
    // 按起始位置排序
    return mathBlocks.sort((a, b) => a.startIndex - b.startIndex);
  }

  /**
   * 检查文本是否包含数学内容
   * @param {string} content - 要检查的内容
   * @returns {boolean}
   */
  hasMathContent(content) {
    return this.extractMathBlocks(content).length > 0;
  }
}

// 浏览器和Node.js兼容导出
if (typeof module !== 'undefined' && module.exports) {
  module.exports = MathContentDetector;
}
if (typeof window !== 'undefined') {
  window.MathContentDetector = MathContentDetector;
}