/**
 * $$数学公式保护工具类
 * 
 * TDD原则：每一层解析器都应该保护$$包装的数学公式
 * 这些公式已经是KaTeX可以直接识别的格式，不需要任何修改
 * 
 * 核心问题修复：JavaScript的replace函数中$是特殊字符
 * - $$ 会被解释为单个$
 * - 必须使用函数返回值方式避免$被解释
 */

class DollarDollarProtector {
  constructor() {
    this.protections = [];
    this.protectionPrefix = '__DD_PROTECTED_';
    this.protectionSuffix = '__';
    this.protectionCounter = 0;
  }

  /**
   * 保护文本中的$$...$$内容
   * @param {string} text - 输入文本
   * @returns {string} 保护后的文本
   */
  protect(text) {
    if (!text) return text;
    
    // 重置保护列表
    this.protections = [];
    this.protectionCounter = 0;
    
    // 保护$$...$$格式的显示数学公式
    // 使用[\s\S]匹配包括换行在内的所有字符
    // 使用+?进行非贪婪匹配
    const protectedText = text.replace(/\$\$[\s\S]+?\$\$/g, (match) => {
      const placeholder = `${this.protectionPrefix}${this.protectionCounter}${this.protectionSuffix}`;
      this.protectionCounter++;
      
      this.protections.push({
        placeholder,
        content: match,
        type: 'display_math'
      });
      
      if (process.env.NODE_ENV !== 'test') {
        console.log(`[DollarDollarProtector] 保护$$公式: ${match.substring(0, 30)}...`);
      }
      
      return placeholder;
    });
    
    return protectedText;
  }

  /**
   * 恢复被保护的$$...$$内容
   * @param {string} text - 保护后的文本
   * @returns {string} 恢复后的文本
   */
  restore(text) {
    if (!text || this.protections.length === 0) return text;
    
    let restoredText = text;
    
    // 恢复所有保护的内容
    this.protections.forEach((protection, index) => {
      // 关键修复：使用函数返回值避免$被解释为特殊字符
      restoredText = restoredText.replace(
        protection.placeholder,
        () => protection.content  // 使用函数返回值，不是直接替换
      );
      
      if (process.env.NODE_ENV !== 'test') {
        console.log(`[DollarDollarProtector] 恢复$$公式 ${index}: ${protection.content.substring(0, 30)}...`);
      }
    });
    
    return restoredText;
  }

  /**
   * 检查文本是否包含$$格式的数学公式
   * @param {string} text - 要检查的文本
   * @returns {boolean}
   */
  hasDoubleDollarMath(text) {
    if (!text) return false;
    return /\$\$[\s\S]+?\$\$/g.test(text);
  }

  /**
   * 获取当前保护的数学公式数量
   * @returns {number}
   */
  getProtectionCount() {
    return this.protections.length;
  }

  /**
   * 清空保护列表
   */
  clear() {
    this.protections = [];
    this.protectionCounter = 0;
  }
}

// 浏览器和Node.js兼容导出
if (typeof module !== 'undefined' && module.exports) {
  module.exports = DollarDollarProtector;
}
if (typeof window !== 'undefined') {
  window.DollarDollarProtector = DollarDollarProtector;
}