/**
 * LaTeX Parser V3 接口测试运行器
 * 用于在 Node.js 环境中运行所有接口测试
 */

const path = require('path');
const fs = require('fs');

// 检查 Jest 是否可用
let useJest = false;
try {
  require('jest');
  // 即使能require jest，也要检查全局函数是否可用
  if (typeof describe !== 'undefined' && typeof test !== 'undefined' && typeof expect !== 'undefined') {
    useJest = true;
  } else {
    console.warn('Jest 模块存在但全局函数未加载，将使用简单的测试运行器');
  }
} catch (error) {
  console.warn('Jest 不可用，将使用简单的测试运行器');
}

// 简单的测试框架实现
if (!useJest) {
  global.describe = (name, fn) => {
    console.log(`\n📝 ${name}`);
    try {
      fn();
    } catch (error) {
      console.error(`❌ 测试套件失败: ${name}`, error.message);
    }
  };

  global.test = global.it = (name, fn) => {
    try {
      fn();
      console.log(`  ✅ ${name}`);
    } catch (error) {
      console.log(`  ❌ ${name}`);
      console.error(`     错误: ${error.message}`);
    }
  };

  global.beforeAll = (fn) => {
    try {
      fn();
    } catch (error) {
      console.error('beforeAll 失败:', error.message);
    }
  };

  global.beforeEach = (fn) => {
    // 简单实现，在每个测试前执行
    global._beforeEachFn = fn;
  };

  global.afterAll = (fn) => {
    global._afterAllFn = fn;
  };

  // 设置expect函数（移到单独的函数中）
}

// 设置expect函数
function setupExpectFunction() {
  global.expect = (actual) => {
    return {
      toBe: (expected) => {
        if (actual !== expected) {
          throw new Error(`期望 ${JSON.stringify(actual)} 严格等于 ${JSON.stringify(expected)}`);
        }
      },
      toEqual: (expected) => {
        if (JSON.stringify(actual) !== JSON.stringify(expected)) {
          throw new Error(`期望 ${JSON.stringify(actual)} 等于 ${JSON.stringify(expected)}`);
        }
      },
      toContain: (expected) => {
        if (Array.isArray(actual)) {
          if (!actual.includes(expected)) {
            throw new Error(`期望数组 [${actual.join(', ')}] 包含 "${expected}"`);
          }
        } else if (typeof actual === 'string') {
          if (!actual.includes(expected)) {
            throw new Error(`期望字符串 "${actual}" 包含 "${expected}"`);
          }
        } else {
          throw new Error(`toContain 期望数组或字符串，但收到 ${typeof actual}`);
        }
      },
      toMatch: (regex) => {
        if (!regex.test(actual)) {
          throw new Error(`期望 "${actual}" 匹配 ${regex}`);
        }
      },
      toBeDefined: () => {
        if (actual === undefined) {
          throw new Error('期望值已定义，但收到 undefined');
        }
      },
      toBeUndefined: () => {
        if (actual !== undefined) {
          throw new Error(`期望值为 undefined，但收到 ${JSON.stringify(actual)}`);
        }
      },
      toBeNull: () => {
        if (actual !== null) {
          throw new Error(`期望值为 null，但收到 ${JSON.stringify(actual)}`);
        }
      },
      toBeTruthy: () => {
        if (!actual) {
          throw new Error(`期望值为真值，但收到 ${JSON.stringify(actual)}`);
        }
      },
      toBeFalsy: () => {
        if (actual) {
          throw new Error(`期望值为假值，但收到 ${JSON.stringify(actual)}`);
        }
      },
      toBeGreaterThan: (expected) => {
        if (!(actual > expected)) {
          throw new Error(`期望 ${actual} 大于 ${expected}`);
        }
      },
      toBeLessThan: (expected) => {
        if (!(actual < expected)) {
          throw new Error(`期望 ${actual} 小于 ${expected}`);
        }
      },
      toBeGreaterThanOrEqual: (expected) => {
        if (!(actual >= expected)) {
          throw new Error(`期望 ${actual} 大于或等于 ${expected}`);
        }
      },
      toBeLessThanOrEqual: (expected) => {
        if (!(actual <= expected)) {
          throw new Error(`期望 ${actual} 小于或等于 ${expected}`);
        }
      },
      toHaveProperty: (prop, value) => {
        if (!(prop in actual)) {
          throw new Error(`期望对象有属性 "${prop}"`);
        }
        if (value !== undefined && actual[prop] !== value) {
          throw new Error(`期望对象属性 "${prop}" 等于 ${JSON.stringify(value)}，但收到 ${JSON.stringify(actual[prop])}`);
        }
      },
      toHaveLength: (length) => {
        if (!actual || typeof actual.length !== 'number') {
          throw new Error(`期望值有 length 属性，但收到 ${typeof actual}`);
        }
        if (actual.length !== length) {
          throw new Error(`期望长度为 ${length}，但收到 ${actual.length}`);
        }
      },
      toThrow: (expectedError) => {
        if (typeof actual !== 'function') {
          throw new Error('toThrow 期望函数');
        }
        try {
          actual();
          throw new Error('期望函数抛出错误，但没有抛出');
        } catch (error) {
          if (expectedError) {
            if (typeof expectedError === 'string') {
              if (!error.message.includes(expectedError)) {
                throw new Error(`期望错误消息包含 "${expectedError}"，但收到 "${error.message}"`);
              }
            } else if (expectedError instanceof RegExp) {
              if (!expectedError.test(error.message)) {
                throw new Error(`期望错误消息匹配 ${expectedError}，但收到 "${error.message}"`);
              }
            }
          }
        }
      },
      not: {
        toBe: (expected) => {
          if (actual === expected) {
            throw new Error(`期望 ${JSON.stringify(actual)} 不严格等于 ${JSON.stringify(expected)}`);
          }
        },
        toEqual: (expected) => {
          if (JSON.stringify(actual) === JSON.stringify(expected)) {
            throw new Error(`期望 ${JSON.stringify(actual)} 不等于 ${JSON.stringify(expected)}`);
          }
        },
        toContain: (expected) => {
          if (Array.isArray(actual) && actual.includes(expected)) {
            throw new Error(`期望数组 [${actual.join(', ')}] 不包含 "${expected}"`);
          } else if (typeof actual === 'string' && actual.includes(expected)) {
            throw new Error(`期望字符串 "${actual}" 不包含 "${expected}"`);
          }
        },
        toBeDefined: () => {
          if (actual !== undefined) {
            throw new Error(`期望值为 undefined，但收到 ${JSON.stringify(actual)}`);
          }
        },
        toThrow: () => {
          if (typeof actual !== 'function') {
            throw new Error('not.toThrow 期望函数');
          }
          try {
            actual();
          } catch (error) {
            throw new Error(`期望函数不抛出错误，但抛出了: ${error.message}`);
          }
        }
      }
    };
  };
}

// 测试文件列表
const testFiles = [
  'interface-basic.spec.js',
  'interface-math.spec.js', 
  'interface-document.spec.js',
  'interface-browser.spec.js',
  'interface-error.spec.js',
  'interface-performance.spec.js'
];

// 运行测试
async function runTests() {
  console.log('🚀 开始运行 LaTeX Parser V3 接口测试...\n');
  
  const startTime = Date.now();
  let totalTestFiles = 0;
  let passedTestFiles = 0;
  let failedTestFiles = 0;
  let totalIndividualTests = 0;
  let passedIndividualTests = 0;
  let failedIndividualTests = 0;

  // 如果没有 Jest，先设置测试框架
  if (!useJest) {
    console.log('📦 使用内置测试框架...\n');
    // 确保 expect 函数在全局范围内可用
    setupExpectFunction();
  }

  for (const testFile of testFiles) {
    const testPath = path.resolve(__dirname, testFile);
    totalTestFiles++;
    
    if (!fs.existsSync(testPath)) {
      console.log(`⚠️  测试文件不存在: ${testFile}`);
      failedTestFiles++;
      continue;
    }
    
    console.log(`\n📁 运行测试文件: ${testFile}`);
    console.log('=' .repeat(60));
    
    try {
      // 重置测试计数器和收集器
      let currentFileTests = [];
      let currentDescribe = '';
      let beforeAllFunctions = [];
      let beforeEachFunctions = [];
      let afterAllFunctions = [];
      let currentFileTestCount = 0;
      let currentFilePassedCount = 0;
      let currentFileFailedCount = 0;
      
      // 重新定义测试框架函数来收集和执行测试
      const originalDescribe = global.describe;
      const originalTest = global.test;
      const originalBeforeAll = global.beforeAll;
      const originalBeforeEach = global.beforeEach;
      const originalAfterAll = global.afterAll;
      
      // 收集阶段 - 重新定义所有测试框架函数
      global.describe = (name, fn) => {
        const previousDescribe = currentDescribe;
        currentDescribe = name;
        console.log(`\n📝 ${name}`);
        try {
          fn(); // 执行 describe 回调来收集测试
        } catch (error) {
          console.error(`❌ 测试套件失败: ${name}`, error.message);
        }
        currentDescribe = previousDescribe;
      };

      global.test = global.it = (name, fn) => {
        currentFileTestCount++;
        currentFileTests.push({
          name: `${currentDescribe ? currentDescribe + ' - ' : ''}${name}`,
          fn: fn
        });
      };
      
      // 收集钩子函数
      global.beforeAll = (fn) => {
        beforeAllFunctions.push(fn);
      };
      
      global.beforeEach = (fn) => {
        beforeEachFunctions.push(fn);
      };
      
      global.afterAll = (fn) => {
        afterAllFunctions.push(fn);
      };
      
      // 加载测试文件（收集阶段）
      delete require.cache[testPath];
      require(testPath);
      
      // 执行 beforeAll 钩子
      for (const beforeAllFn of beforeAllFunctions) {
        try {
          beforeAllFn();
        } catch (error) {
          console.error(`❌ beforeAll 失败: ${error.message}`);
          throw error; // 如果 beforeAll 失败，停止执行此文件的测试
        }
      }
      
      // 执行阶段 - 运行收集到的测试
      for (const testCase of currentFileTests) {
        try {
          // 执行所有 beforeEach 钩子
          for (const beforeEachFn of beforeEachFunctions) {
            beforeEachFn();
          }
          
          // 确保 expect 函数在测试执行时可用
          if (typeof expect === 'undefined' && typeof global.expect === 'undefined') {
            setupExpectFunction();
          }
          
          testCase.fn();
          console.log(`  ✅ ${testCase.name}`);
          currentFilePassedCount++;
        } catch (error) {
          console.log(`  ❌ ${testCase.name}`);
          console.error(`     错误: ${error.message}`);
          currentFileFailedCount++;
        }
      }
      
      // 执行 afterAll 钩子
      for (const afterAllFn of afterAllFunctions) {
        try {
          afterAllFn();
        } catch (error) {
          console.error(`❌ afterAll 失败: ${error.message}`);
        }
      }
      
      // 更新总计数器
      totalIndividualTests += currentFileTestCount;
      passedIndividualTests += currentFilePassedCount;
      failedIndividualTests += currentFileFailedCount;
      
      console.log(`📋 ${testFile}: ${currentFileTestCount} 个测试, ${currentFilePassedCount} 通过, ${currentFileFailedCount} 失败`);
      
      // 如果这个文件有测试失败，则文件失败
      if (currentFileFailedCount > 0) {
        failedTestFiles++;
      } else {
        passedTestFiles++;
      }
      
      // 恢复原始函数
      global.describe = originalDescribe;
      global.test = originalTest;
      global.beforeAll = originalBeforeAll;
      global.beforeEach = originalBeforeEach;
      global.afterAll = originalAfterAll;
      
    } catch (error) {
      console.error(`❌ ${testFile} 测试文件加载失败:`, error.message);
      failedTestFiles++;
      failedIndividualTests++; // 至少算一个失败的测试
    }
  }
  
  const endTime = Date.now();
  const duration = endTime - startTime;
  
  console.log('\n' + '='.repeat(60));
  console.log('📊 测试总结');
  console.log('=' .repeat(60));
  console.log(`⏱️  总用时: ${duration}ms`);
  console.log(`📁 测试文件: ${totalTestFiles} 个 (通过: ${passedTestFiles}, 失败: ${failedTestFiles})`);
  console.log(`📝 个别测试: ${totalIndividualTests} 个 (通过: ${passedIndividualTests}, 失败: ${failedIndividualTests})`);
  
  if (failedTestFiles === 0 && failedIndividualTests === 0) {
    console.log('\n🎉 所有测试通过！');
  } else {
    if (failedTestFiles > 0) {
      console.log(`\n⚠️  有 ${failedTestFiles} 个测试文件失败`);
    }
    if (failedIndividualTests > 0) {
      console.log(`\n⚠️  有 ${failedIndividualTests} 个具体测试失败`);
    }
  }
  
  // 执行清理
  if (global._afterAllFn) {
    try {
      global._afterAllFn();
    } catch (error) {
      console.error('afterAll 失败:', error.message);
    }
  }
}

// 检查环境和依赖
function checkEnvironment() {
  console.log('🔍 检查测试环境...');
  
  const checks = [
    {
      name: 'Node.js 版本',
      check: () => process.version,
      expected: 'v12+'
    },
    {
      name: 'LaTeX Parser 核心文件',
      check: () => {
        const corePath = path.resolve(__dirname, '../core/LaTeXParser.js');
        return fs.existsSync(corePath) ? '✅ 存在' : '❌ 不存在';
      },
      expected: '存在'
    },
    {
      name: '测试文件',
      check: () => {
        const existingFiles = testFiles.filter(file => 
          fs.existsSync(path.resolve(__dirname, file))
        );
        return `${existingFiles.length}/${testFiles.length} 个文件存在`;
      },
      expected: `${testFiles.length} 个文件`
    }
  ];
  
  checks.forEach(check => {
    console.log(`  ${check.name}: ${check.check()}`);
  });
  
  console.log('');
}

// 主函数
function main() {
  console.log('🧪 LaTeX Parser V3 接口测试运行器');
  console.log('='.repeat(60));
  
  checkEnvironment();
  
  if (process.argv.includes('--help') || process.argv.includes('-h')) {
    console.log(`
使用方法:
  node test-runner.js [选项]

选项:
  --help, -h     显示帮助信息
  --version, -v  显示版本信息
  --list, -l     列出所有测试文件
  --file <name>  只运行指定的测试文件

示例:
  node test-runner.js                           # 运行所有测试
  node test-runner.js --file interface-basic   # 只运行基础接口测试
  node test-runner.js --list                   # 列出所有测试文件
`);
    return;
  }
  
  if (process.argv.includes('--version') || process.argv.includes('-v')) {
    console.log('LaTeX Parser V3 Test Runner v1.0.0');
    return;
  }
  
  if (process.argv.includes('--list') || process.argv.includes('-l')) {
    console.log('📝 可用的测试文件:');
    testFiles.forEach((file, index) => {
      const exists = fs.existsSync(path.resolve(__dirname, file));
      console.log(`  ${index + 1}. ${file} ${exists ? '✅' : '❌'}`);
    });
    return;
  }
  
  // 检查是否指定了特定文件
  const fileIndex = process.argv.indexOf('--file');
  if (fileIndex !== -1 && process.argv[fileIndex + 1]) {
    const fileName = process.argv[fileIndex + 1];
    const matchingFiles = testFiles.filter(file => 
      file.includes(fileName) || file === fileName + '.spec.js'
    );
    
    if (matchingFiles.length === 0) {
      console.log(`❌ 未找到匹配的测试文件: ${fileName}`);
      console.log('可用文件:', testFiles.join(', '));
      return;
    }
    
    // 只运行匹配的文件
    testFiles.length = 0;
    testFiles.push(...matchingFiles);
    console.log(`🎯 只运行匹配的测试文件: ${matchingFiles.join(', ')}`);
  }
  
  // 运行测试
  runTests().catch(error => {
    console.error('运行测试时发生错误:', error);
    process.exit(1);
  });
}

// 如果直接运行此文件
if (require.main === module) {
  main();
}

module.exports = {
  runTests,
  checkEnvironment,
  testFiles
};