/**
 * 简单测试文件 - 验证LaTeX Parser V3接口功能
 */

const path = require('path');

// 简单的断言函数
function assert(condition, message) {
  if (!condition) {
    throw new Error(message || '断言失败');
  }
}

function assertEqual(actual, expected, message) {
  if (actual !== expected) {
    throw new Error(message || `期望 ${expected}, 得到 ${actual}`);
  }
}

function assertContains(str, substr, message) {
  if (!str.includes(substr)) {
    throw new Error(message || `期望 "${str}" 包含 "${substr}"`);
  }
}

console.log('🧪 开始简单接口测试...\n');

let testCount = 0;
let passCount = 0;
let failCount = 0;

function runTest(name, testFn) {
  testCount++;
  try {
    testFn();
    console.log(`✅ ${name}`);
    passCount++;
  } catch (error) {
    console.log(`❌ ${name}`);
    console.log(`   错误: ${error.message}`);
    failCount++;
  }
}

try {
  // 加载 LaTeX Parser
  const LaTeXParser = require('../core/LaTeXParser.js');
  console.log('📦 LaTeX Parser 加载成功\n');

  // 测试1: 基础构造函数
  runTest('构造函数应该正常工作', () => {
    const parser = new LaTeXParser();
    assert(parser !== null, '解析器不应该为null');
    assert(typeof parser.parseComplete === 'function', 'parseComplete应该是函数');
  });

  // 测试2: 默认配置
  runTest('默认配置应该正确', () => {
    const parser = new LaTeXParser();
    assert(parser.config, '应该有config对象');
    assertEqual(parser.config.mode, 'tolerant', '默认模式应该是tolerant');
    assert(Array.isArray(parser.config.enabledLayers), 'enabledLayers应该是数组');
    assertEqual(parser.config.maxRecursionDepth, 100, '默认递归深度应该是100');
  });

  // 测试3: 自定义配置
  runTest('自定义配置应该被应用', () => {
    const config = {
      mode: 'strict',
      enabledLayers: ['document', 'math'],
      maxRecursionDepth: 50
    };
    const parser = new LaTeXParser(config);
    assertEqual(parser.config.mode, 'strict', '自定义模式应该被应用');
    assertEqual(parser.config.enabledLayers.length, 2, '自定义层数应该被应用');
    assertEqual(parser.config.maxRecursionDepth, 50, '自定义递归深度应该被应用');
  });

  // 测试4: 空字符串处理
  runTest('应该处理空字符串', () => {
    const parser = new LaTeXParser();
    const result = parser.parseComplete('');
    assert(result, '应该返回结果对象');
    assertEqual(result.success, true, '空字符串处理应该成功');
    assertEqual(result.processedText, '', '空输入应该返回空输出');
  });

  // 测试5: 简单文本处理
  runTest('应该处理纯文本', () => {
    const parser = new LaTeXParser();
    const result = parser.parseComplete('Hello World');
    assertEqual(result.success, true, '纯文本处理应该成功');
    assertContains(result.processedText, 'Hello World', '输出应该包含输入文本');
  });

  // 测试6: 基础LaTeX命令
  runTest('应该处理基础LaTeX命令', () => {
    const parser = new LaTeXParser();
    const result = parser.parseComplete('\\section{测试章节}');
    assertEqual(result.success, true, '基础命令处理应该成功');
    assertContains(result.processedText, '测试章节', '输出应该包含章节内容');
  });

  // 测试7: 数学公式
  runTest('应该处理数学公式', () => {
    const parser = new LaTeXParser();
    const result = parser.parseComplete('$E = mc^2$');
    assertEqual(result.success, true, '数学公式处理应该成功');
    assertContains(result.processedText, 'E = mc^2', '输出应该包含数学内容');
  });

  // 测试8: 返回结果结构
  runTest('返回结果应该有正确的结构', () => {
    const parser = new LaTeXParser();
    const result = parser.parseComplete('测试');
    
    assert(result.hasOwnProperty('success'), '应该有success属性');
    assert(result.hasOwnProperty('processedText'), '应该有processedText属性');
    assert(result.hasOwnProperty('blocks'), '应该有blocks属性');
    assert(result.hasOwnProperty('metadata'), '应该有metadata属性');
    assert(result.hasOwnProperty('warnings'), '应该有warnings属性');
    assert(result.hasOwnProperty('errors'), '应该有errors属性');
    
    assertEqual(typeof result.success, 'boolean', 'success应该是布尔值');
    assertEqual(typeof result.processedText, 'string', 'processedText应该是字符串');
    assert(Array.isArray(result.blocks), 'blocks应该是数组');
    assertEqual(typeof result.metadata, 'object', 'metadata应该是对象');
    assert(Array.isArray(result.warnings), 'warnings应该是数组');
    assert(Array.isArray(result.errors), 'errors应该是数组');
  });

  // 测试9: 性能测试
  runTest('处理应该在合理时间内完成', () => {
    const parser = new LaTeXParser();
    const startTime = Date.now();
    const result = parser.parseComplete('\\section{性能测试} 内容 $x^2$');
    const endTime = Date.now();
    
    const processingTime = endTime - startTime;
    assertEqual(result.success, true, '性能测试应该成功');
    assert(processingTime < 1000, `处理时间应该小于1000ms，实际：${processingTime}ms`);
  });

  // 测试10: 错误处理
  runTest('应该处理未知命令并生成警告', () => {
    const parser = new LaTeXParser();
    const result = parser.parseComplete('\\unknowncommand{test}');
    assertEqual(result.success, true, '错误处理应该允许继续处理');
    assert(result.warnings.length > 0, '应该生成警告');
  });

} catch (error) {
  console.error('❌ 加载LaTeX Parser失败:', error.message);
  process.exit(1);
}

// 输出测试总结
console.log('\n' + '='.repeat(50));
console.log('📊 测试总结');
console.log('='.repeat(50));
console.log(`总测试数: ${testCount}`);
console.log(`✅ 通过: ${passCount}`);
console.log(`❌ 失败: ${failCount}`);
console.log(`成功率: ${((passCount / testCount) * 100).toFixed(1)}%`);

if (failCount === 0) {
  console.log('\n🎉 所有测试通过！');
  process.exit(0);
} else {
  console.log(`\n⚠️  有 ${failCount} 个测试失败`);
  process.exit(1);
}