/**
 * Jest 测试设置文件
 * 在所有测试文件运行前执行的设置代码
 */

const path = require('path');
const fs = require('fs');

// 扩展 Jest 的期望匹配器
expect.extend({
  toBeWithinRange(received, floor, ceiling) {
    const pass = received >= floor && received <= ceiling;
    if (pass) {
      return {
        message: () => `expected ${received} not to be within range ${floor} - ${ceiling}`,
        pass: true,
      };
    } else {
      return {
        message: () => `expected ${received} to be within range ${floor} - ${ceiling}`,
        pass: false,
      };
    }
  },

  toHaveProcessingTime(received, maxTime = 1000) {
    const hasMetadata = received && received.metadata;
    const hasProcessingTime = hasMetadata && typeof received.metadata.processingTime === 'number';
    const isWithinTime = hasProcessingTime && received.metadata.processingTime <= maxTime;
    
    if (hasProcessingTime && isWithinTime) {
      return {
        message: () => `expected processing time ${received.metadata.processingTime}ms not to be <= ${maxTime}ms`,
        pass: true,
      };
    } else {
      return {
        message: () => `expected processing time <= ${maxTime}ms, got ${hasProcessingTime ? received.metadata.processingTime : 'undefined'}ms`,
        pass: false,
      };
    }
  },

  toHaveValidLatexResult(received) {
    const isObject = received && typeof received === 'object';
    const hasSuccess = isObject && typeof received.success === 'boolean';
    const hasProcessedText = isObject && typeof received.processedText === 'string';
    const hasBlocks = isObject && Array.isArray(received.blocks);
    const hasMetadata = isObject && typeof received.metadata === 'object';
    const hasWarnings = isObject && Array.isArray(received.warnings);
    const hasErrors = isObject && Array.isArray(received.errors);
    
    const isValid = hasSuccess && hasProcessedText && hasBlocks && 
                   hasMetadata && hasWarnings && hasErrors;
    
    if (isValid) {
      return {
        message: () => `expected result not to have valid LaTeX result structure`,
        pass: true,
      };
    } else {
      const missing = [];
      if (!hasSuccess) missing.push('success (boolean)');
      if (!hasProcessedText) missing.push('processedText (string)');
      if (!hasBlocks) missing.push('blocks (array)');
      if (!hasMetadata) missing.push('metadata (object)');
      if (!hasWarnings) missing.push('warnings (array)');
      if (!hasErrors) missing.push('errors (array)');
      
      return {
        message: () => `expected valid LaTeX result structure, missing: ${missing.join(', ')}`,
        pass: false,
      };
    }
  }
});

// 全局测试配置
global.testConfig = {
  maxProcessingTime: 1000,
  maxLargeDocProcessingTime: 5000,
  performanceTestIterations: 100
};

// 测试工具函数
global.createTestDocument = (size = 'small') => {
  const templates = {
    small: '\\section{测试} 内容 $x^2$',
    medium: `
      \\section{中型文档测试}
      \\subsection{数学内容}
      $\\sum_{i=1}^{100} i = 5050$
      
      \\subsection{列表}
      \\begin{itemize}
      \\item 项目1
      \\item 项目2
      \\end{itemize}
      
      \\subsection{表格}
      \\begin{tabular}{|c|c|}
      \\hline
      A & B \\\\
      \\hline
      \\end{tabular}
    `,
    large: ''
  };
  
  if (size === 'large') {
    const sections = [];
    for (let i = 1; i <= 100; i++) {
      sections.push(`
        \\section{第${i}章}
        这是第${i}章的内容 $f_${i}(x) = x^${i}$
        
        \\begin{itemize}
        \\item 项目${i}.1
        \\item 项目${i}.2
        \\end{itemize}
      `);
    }
    templates.large = sections.join('\n');
  }
  
  return templates[size] || templates.small;
};

global.createMathDocument = () => {
  return `
    \\section{数学公式测试}
    
    行内公式: $E = mc^2$
    
    显示公式:
    $$\\int_0^\\infty e^{-x} dx = 1$$
    
    矩阵:
    $$\\begin{matrix}
    1 & 2 \\\\
    3 & 4
    \\end{matrix}$$
    
    对齐:
    \\begin{align}
    a &= b + c \\\\
    d &= e + f
    \\end{align}
  `;
};

global.createErrorDocument = () => {
  return `
    \\section{正常章节}
    \\unknowncommand{测试}
    \\begin{itemize}
    \\item 项目1
    \\end{enumerate}
    继续内容
  `;
};

// 性能测试工具
global.measurePerformance = (fn, iterations = 1) => {
  const startTime = Date.now();
  for (let i = 0; i < iterations; i++) {
    fn();
  }
  const endTime = Date.now();
  return {
    totalTime: endTime - startTime,
    averageTime: (endTime - startTime) / iterations,
    iterations
  };
};

// 检查核心文件是否存在
global.checkCoreFiles = () => {
  const coreDir = path.resolve(__dirname, '../core');
  const coreFiles = [
    'LaTeXParser.js',
    'DocumentParser.js',
    'BlockElementParser.js',
    'InlineElementParser.js',
    'MathParser.js',
    'SpecialParser.js'
  ];
  
  const missing = [];
  const existing = [];
  
  coreFiles.forEach(file => {
    const filePath = path.join(coreDir, file);
    if (fs.existsSync(filePath)) {
      existing.push(file);
    } else {
      missing.push(file);
    }
  });
  
  return { existing, missing };
};

// 在所有测试开始前运行
beforeAll(() => {
  console.log('🚀 设置 LaTeX Parser V3 测试环境...');
  
  // 检查核心文件
  const { existing, missing } = checkCoreFiles();
  console.log(`✅ 核心文件存在: ${existing.length}/${existing.length + missing.length}`);
  
  if (missing.length > 0) {
    console.warn(`⚠️  缺失文件: ${missing.join(', ')}`);
  }
  
  // 设置测试超时
  jest.setTimeout(30000);
});

// 在所有测试结束后运行
afterAll(() => {
  console.log('🏁 LaTeX Parser V3 测试环境清理完成');
});

// 错误处理
process.on('unhandledRejection', (reason, promise) => {
  console.error('Unhandled Rejection at:', promise, 'reason:', reason);
});

// 导出测试工具
module.exports = {
  createTestDocument: global.createTestDocument,
  createMathDocument: global.createMathDocument,
  createErrorDocument: global.createErrorDocument,
  measurePerformance: global.measurePerformance,
  checkCoreFiles: global.checkCoreFiles
};