/**
 * LaTeX Parser V3 性能和兼容性接口测试
 * 测试处理时间验证、大文档处理、内存使用合理性等
 */

describe('LaTeX Parser V3 性能和兼容性接口测试', () => {
  let LaTeXParser;

  beforeAll(() => {
    // 检查是否在 Node.js 环境中
    if (typeof require !== 'undefined') {
      // Node.js 环境
      const path = require('path');
      const fs = require('fs');
      
      // 检查核心文件是否存在
      const coreFiles = [
        '../core/LaTeXParser.js'
      ];
      
      for (const file of coreFiles) {
        const filePath = path.resolve(__dirname, file);
        if (!fs.existsSync(filePath)) {
          throw new Error(`核心文件不存在: ${file}`);
        }
      }
      
      // 加载 LaTeXParser
      LaTeXParser = require('../core/LaTeXParser.js');
    } else {
      // 浏览器环境
      if (typeof window !== 'undefined' && window.LaTeXParser) {
        LaTeXParser = window.LaTeXParser;
      } else {
        throw new Error('浏览器环境中 LaTeXParser 未定义');
      }
    }
  });

  describe('基础性能指标测试', () => {
    let parser;

    beforeEach(() => {
      parser = new LaTeXParser();
    });

    test('小型文档处理时间应该 < 100ms', () => {
      const smallDoc = `
        \\section{测试}
        这是测试文档 $E = mc^2$
        
        \\begin{itemize}
        \\item 项目1
        \\item 项目2
        \\end{itemize}
      `;
      
      const startTime = Date.now();
      const result = parser.parseComplete(smallDoc);
      const endTime = Date.now();
      
      const processingTime = endTime - startTime;
      
      expect(result.success).toBe(true);
      expect(processingTime).toBeLessThan(100);
      expect(result.metadata.processingTime).toBeDefined();
      expect(typeof result.metadata.processingTime).toBe('number');
    });

    test('中型文档（1000行）处理时间应该 < 1000ms', () => {
      // 生成1000行的文档
      const lines = [];
      for (let i = 1; i <= 1000; i++) {
        if (i % 50 === 0) {
          lines.push(`\\section{章节 ${Math.floor(i/50)}}`);
        } else if (i % 10 === 0) {
          lines.push(`\\subsection{子章节 ${i}}`);
        } else if (i % 5 === 0) {
          lines.push(`数学公式: $x_{${i}} = ${i}^2 + \\sqrt{${i}}$`);
        } else {
          lines.push(`第${i}行文本内容，包含 \\textbf{粗体} 和 \\textit{斜体}。`);
        }
      }
      
      const mediumDoc = lines.join('\n');
      
      const startTime = Date.now();
      const result = parser.parseComplete(mediumDoc);
      const endTime = Date.now();
      
      const processingTime = endTime - startTime;
      
      expect(result.success).toBe(true);
      expect(processingTime).toBeLessThan(1000);
      expect(result.processedText.length).toBeGreaterThan(1000);
    });

    test('空文档处理应该非常快速（< 10ms）', () => {
      const startTime = Date.now();
      const result = parser.parseComplete('');
      const endTime = Date.now();
      
      const processingTime = endTime - startTime;
      
      expect(result.success).toBe(true);
      expect(processingTime).toBeLessThan(10);
    });

    test('纯文本处理应该很快（< 50ms）', () => {
      const plainText = Array(1000).fill('这是普通文本。').join(' ');
      
      const startTime = Date.now();
      const result = parser.parseComplete(plainText);
      const endTime = Date.now();
      
      const processingTime = endTime - startTime;
      
      expect(result.success).toBe(true);
      expect(processingTime).toBeLessThan(50);
    });
  });

  describe('大文档处理测试', () => {
    let parser;

    beforeEach(() => {
      parser = new LaTeXParser();
    });

    test('应该处理包含大量数学公式的文档', () => {
      const mathFormulas = [];
      for (let i = 1; i <= 500; i++) {
        mathFormulas.push(`$f_{${i}}(x) = x^{${i}} + \\sum_{k=1}^{${i}} \\frac{1}{k}$`);
      }
      
      const mathDoc = mathFormulas.join(' ');
      
      const startTime = Date.now();
      const result = parser.parseComplete(mathDoc);
      const endTime = Date.now();
      
      const processingTime = endTime - startTime;
      
      expect(result.success).toBe(true);
      expect(processingTime).toBeLessThan(5000); // 5秒内
      expect(result.processedText).toContain('f_1');
      expect(result.processedText).toContain('f_500');
    });

    test('应该处理包含大量列表的文档', () => {
      let largeListDoc = '';
      for (let i = 1; i <= 50; i++) {
        largeListDoc += `\\begin{itemize}\n`;
        for (let j = 1; j <= 20; j++) {
          largeListDoc += `\\item 列表${i}项目${j}\n`;
        }
        largeListDoc += `\\end{itemize}\n`;
      }
      
      const startTime = Date.now();
      const result = parser.parseComplete(largeListDoc);
      const endTime = Date.now();
      
      const processingTime = endTime - startTime;
      
      expect(result.success).toBe(true);
      expect(processingTime).toBeLessThan(3000); // 3秒内
    });

    test('应该处理包含大量表格的文档', () => {
      let tableDoc = '';
      for (let i = 1; i <= 20; i++) {
        tableDoc += `
          \\begin{tabular}{|c|c|c|c|}
          \\hline
        `;
        for (let row = 1; row <= 10; row++) {
          tableDoc += `R${row}C1 & R${row}C2 & R${row}C3 & R${row}C4 \\\\\n`;
          tableDoc += `\\hline\n`;
        }
        tableDoc += `\\end{tabular}\n`;
      }
      
      const startTime = Date.now();
      const result = parser.parseComplete(tableDoc);
      const endTime = Date.now();
      
      const processingTime = endTime - startTime;
      
      expect(result.success).toBe(true);
      expect(processingTime).toBeLessThan(4000); // 4秒内
    });

    test('应该处理深度嵌套的结构', () => {
      let nestedDoc = '';
      const depth = 50;
      
      // 创建深度嵌套的结构
      for (let i = 0; i < depth; i++) {
        nestedDoc += '\\textbf{';
      }
      nestedDoc += '深度嵌套的内容';
      for (let i = 0; i < depth; i++) {
        nestedDoc += '}';
      }
      
      const startTime = Date.now();
      const result = parser.parseComplete(nestedDoc);
      const endTime = Date.now();
      
      const processingTime = endTime - startTime;
      
      expect(result.success).toBe(true);
      expect(processingTime).toBeLessThan(1000);
      expect(result.processedText).toContain('深度嵌套的内容');
    });

    test('应该处理包含特殊字符的大文档', () => {
      const specialChars = '§±°µ©®™€£¥¢¤¦¨ªº¹²³¼½¾×÷αβγδεζηθικλμνξοπρστυφχψω';
      const largeSpecialDoc = Array(1000).fill(`特殊字符测试: ${specialChars} 和数学: $\\alpha + \\beta$`).join('\n');
      
      const startTime = Date.now();
      const result = parser.parseComplete(largeSpecialDoc);
      const endTime = Date.now();
      
      const processingTime = endTime - startTime;
      
      expect(result.success).toBe(true);
      expect(processingTime).toBeLessThan(2000);
    });
  });

  describe('内存使用测试', () => {
    let parser;

    beforeEach(() => {
      parser = new LaTeXParser();
    });

    test('多次处理同一文档不应该造成内存泄漏', () => {
      const testDoc = `
        \\section{测试}
        \\begin{itemize}
        \\item 项目1 $x^2$
        \\item 项目2 $y^2$
        \\end{itemize}
      `;
      
      // 多次处理
      const iterations = 100;
      const startTime = Date.now();
      
      for (let i = 0; i < iterations; i++) {
        const result = parser.parseComplete(testDoc);
        expect(result.success).toBe(true);
      }
      
      const endTime = Date.now();
      const averageTime = (endTime - startTime) / iterations;
      
      expect(averageTime).toBeLessThan(100); // 平均每次处理 < 100ms
    });

    test('处理大文档后内存应该能够被释放', () => {
      // 创建一个非常大的文档
      const hugeSections = [];
      for (let i = 1; i <= 100; i++) {
        hugeSections.push(`
          \\section{第${i}章}
          ${Array(100).fill(`第${i}章的内容，包含数学公式 $f_${i}(x) = x^${i}$`).join('\n')}
          
          \\begin{itemize}
          ${Array(50).fill(`\\item 第${i}章项目`).join('\n')}
          \\end{itemize}
        `);
      }
      
      const hugeDoc = hugeSections.join('\n');
      
      const startTime = Date.now();
      const result = parser.parseComplete(hugeDoc);
      const endTime = Date.now();
      
      const processingTime = endTime - startTime;
      
      expect(result.success).toBe(true);
      expect(processingTime).toBeLessThan(10000); // 10秒内
      
      // 验证结果包含期望内容
      expect(result.processedText).toContain('第1章');
      expect(result.processedText).toContain('第100章');
    });

    test('创建多个解析器实例应该是高效的', () => {
      const instanceCount = 50;
      const parsers = [];
      
      const startTime = Date.now();
      
      for (let i = 0; i < instanceCount; i++) {
        parsers.push(new LaTeXParser({
          mode: i % 2 === 0 ? 'tolerant' : 'strict',
          enabledLayers: i % 3 === 0 ? ['math'] : ['document', 'inline']
        }));
      }
      
      const creationTime = Date.now() - startTime;
      expect(creationTime).toBeLessThan(1000); // 创建50个实例 < 1秒
      
      // 每个实例都应该能正常工作
      const testPromises = parsers.map((parser, index) => {
        return new Promise((resolve) => {
          const result = parser.parseComplete(`\\section{测试${index}} $x_${index}$`);
          expect(result.success).toBe(true);
          resolve();
        });
      });
      
      return Promise.all(testPromises);
    });
  });

  describe('兼容性测试', () => {
    test('应该在不同配置下保持一致的性能', () => {
      const testDoc = `
        \\section{性能测试}
        \\subsection{数学内容}
        $\\sum_{i=1}^{100} i = 5050$
        
        \\subsection{列表内容}
        \\begin{itemize}
        \\item 项目1
        \\item 项目2
        \\end{itemize}
      `;
      
      const configs = [
        { mode: 'tolerant', enabledLayers: ['document', 'block', 'inline', 'math', 'special'] },
        { mode: 'strict', enabledLayers: ['document', 'block', 'inline', 'math', 'special'] },
        { mode: 'tolerant', enabledLayers: ['document', 'math'] },
        { mode: 'tolerant', enabledLayers: ['math'] },
        { mode: 'tolerant', maxRecursionDepth: 50 }
      ];
      
      const results = configs.map(config => {
        const parser = new LaTeXParser(config);
        const startTime = Date.now();
        const result = parser.parseComplete(testDoc);
        const endTime = Date.now();
        
        return {
          config,
          processingTime: endTime - startTime,
          success: result.success,
          outputLength: result.processedText.length
        };
      });
      
      // 所有配置都应该成功
      results.forEach(result => {
        expect(result.success).toBe(true);
        expect(result.processingTime).toBeLessThan(500);
      });
      
      // 性能差异不应该太大
      const times = results.map(r => r.processingTime);
      const maxTime = Math.max(...times);
      const minTime = Math.min(...times);
      expect(maxTime - minTime).toBeLessThan(1000); // 差异 < 1秒
    });

    test('应该在不同输入编码下正常工作', () => {
      const encodings = [
        '简体中文内容',
        '繁體中文內容',
        'English content with math $x^2$',
        'Contenu français avec des maths $\\alpha + \\beta$',
        'Contenido español con matemáticas $\\sum x_i$',
        'Deutsche Inhalte mit Mathematik $\\int_0^1 f(x)dx$',
        'Русский контент с математикой $\\partial f/\\partial x$',
        '日本語の内容と数学 $\\lim_{x \\to \\infty} f(x)$'
      ];
      
      const parser = new LaTeXParser();
      
      encodings.forEach((content, index) => {
        const input = `\\section{测试 ${index + 1}} ${content}`;
        const startTime = Date.now();
        const result = parser.parseComplete(input);
        const endTime = Date.now();
        
        expect(result.success).toBe(true);
        expect(endTime - startTime).toBeLessThan(200);
        expect(result.processedText).toContain(`测试 ${index + 1}`);
      });
    });

    test('应该处理极端大小的输入', () => {
      const parser = new LaTeXParser();
      
      // 极小输入
      const tinyResult = parser.parseComplete('$x$');
      expect(tinyResult.success).toBe(true);
      
      // 极大输入（但在合理范围内）
      const largeInput = Array(10000).fill('x').join('') + '$y^2$';
      const startTime = Date.now();
      const largeResult = parser.parseComplete(largeInput);
      const endTime = Date.now();
      
      expect(largeResult.success).toBe(true);
      expect(endTime - startTime).toBeLessThan(3000);
    });

    test('应该在递归深度限制下正常工作', () => {
      const shallowParser = new LaTeXParser({ maxRecursionDepth: 10 });
      const deepParser = new LaTeXParser({ maxRecursionDepth: 1000 });
      
      // 创建深度嵌套结构
      const deepNesting = 'a'.repeat(5) + 
                         Array(20).fill().map((_, i) => `\\textbf{`).join('') +
                         '深层内容' +
                         Array(20).fill().map((_, i) => `}`).join('');
      
      const shallowResult = shallowParser.parseComplete(deepNesting);
      const deepResult = deepParser.parseComplete(deepNesting);
      
      expect(shallowResult).toBeDefined();
      expect(deepResult).toBeDefined();
      expect(deepResult.success).toBe(true);
    });
  });

  describe('并发和异步处理测试', () => {
    test('多个解析器实例应该能够并发工作', () => {
      const concurrentTasks = [];
      
      for (let i = 0; i < 10; i++) {
        concurrentTasks.push(new Promise((resolve) => {
          const parser = new LaTeXParser();
          const doc = `\\section{并发测试 ${i}} 内容 $x_{${i}}^2$`;
          
          const startTime = Date.now();
          const result = parser.parseComplete(doc);
          const endTime = Date.now();
          
          resolve({
            index: i,
            success: result.success,
            processingTime: endTime - startTime,
            hasContent: result.processedText.includes(`并发测试 ${i}`)
          });
        }));
      }
      
      return Promise.all(concurrentTasks).then(results => {
        results.forEach(result => {
          expect(result.success).toBe(true);
          expect(result.processingTime).toBeLessThan(500);
          expect(result.hasContent).toBe(true);
        });
        
        // 所有任务的总时间不应该是单个任务时间的简单倍数
        const totalTime = results.reduce((sum, r) => sum + r.processingTime, 0);
        const averageTime = totalTime / results.length;
        expect(averageTime).toBeLessThan(300);
      });
    });

    test('应该能够处理异步式的多次调用', async () => {
      const parser = new LaTeXParser();
      const results = [];
      
      // 模拟异步调用
      for (let i = 0; i < 5; i++) {
        await new Promise(resolve => setTimeout(resolve, 10)); // 小延迟
        
        const startTime = Date.now();
        const result = parser.parseComplete(`\\section{异步测试 ${i}} $f_${i}(x)$`);
        const endTime = Date.now();
        
        results.push({
          success: result.success,
          processingTime: endTime - startTime
        });
      }
      
      results.forEach(result => {
        expect(result.success).toBe(true);
        expect(result.processingTime).toBeLessThan(200);
      });
    });
  });

  describe('资源清理和垃圾回收测试', () => {
    test('大量处理后应该能够正常释放资源', () => {
      const parser = new LaTeXParser();
      
      // 处理大量不同的文档
      for (let i = 0; i < 200; i++) {
        const doc = `
          \\section{文档 ${i}}
          内容 ${i} 包含数学 $x_{${i}}^{${i % 10}}$
          
          \\begin{itemize}
          \\item 项目 ${i}.1
          \\item 项目 ${i}.2
          \\end{itemize}
        `;
        
        const result = parser.parseComplete(doc);
        expect(result.success).toBe(true);
        
        // 每隔一定次数检查一次性能
        if (i % 50 === 0) {
          const startTime = Date.now();
          const testResult = parser.parseComplete('\\section{性能检查} $test$');
          const endTime = Date.now();
          
          expect(testResult.success).toBe(true);
          expect(endTime - startTime).toBeLessThan(100);
        }
      }
    });

    test('错误处理不应该导致资源泄漏', () => {
      const parser = new LaTeXParser({ mode: 'tolerant' });
      
      // 生成大量错误
      for (let i = 0; i < 100; i++) {
        const errorDoc = `
          \\unknowncommand${i}{test}
          \\begin{unknownenv${i}}
          错误内容
          \\end{differentenv${i}}
        `;
        
        const result = parser.parseComplete(errorDoc);
        expect(result.success).toBe(true);
        expect(result.warnings.length).toBeGreaterThan(0);
      }
      
      // 处理正常文档仍应该高效
      const normalDoc = '\\section{正常} 内容 $x^2$';
      const startTime = Date.now();
      const normalResult = parser.parseComplete(normalDoc);
      const endTime = Date.now();
      
      expect(normalResult.success).toBe(true);
      expect(endTime - startTime).toBeLessThan(50);
    });
  });
});