/**
 * LaTeX Parser V3 数学公式接口测试
 * 测试行内公式、显示公式、数学环境等
 */

describe('LaTeX Parser V3 数学公式接口测试', () => {
  let LaTeXParser;

  beforeAll(() => {
    // 检查是否在 Node.js 环境中
    if (typeof require !== 'undefined') {
      // Node.js 环境
      const path = require('path');
      const fs = require('fs');
      
      // 检查核心文件是否存在
      const coreFiles = [
        '../core/MathParser.js',
        '../core/LaTeXParser.js'
      ];
      
      for (const file of coreFiles) {
        const filePath = path.resolve(__dirname, file);
        if (!fs.existsSync(filePath)) {
          throw new Error(`核心文件不存在: ${file}`);
        }
      }
      
      // 加载 LaTeXParser
      LaTeXParser = require('../core/LaTeXParser.js');
    } else {
      // 浏览器环境
      if (typeof window !== 'undefined' && window.LaTeXParser) {
        LaTeXParser = window.LaTeXParser;
      } else {
        throw new Error('浏览器环境中 LaTeXParser 未定义');
      }
    }
  });

  describe('行内数学公式测试', () => {
    let parser;

    beforeEach(() => {
      parser = new LaTeXParser();
    });

    test('应该正确处理简单的行内公式 $...$', () => {
      const input = '$E = mc^2$';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/katex|math-inline/i);
      expect(result.processedText).toContain('E = mc^2');
    });

    test('应该正确处理 \\(...\\) 格式的行内公式', () => {
      const input = '\\(a^2 + b^2 = c^2\\)';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/katex|math/i);
    });

    test('应该处理文本中嵌入的行内公式', () => {
      const input = '这是一个包含数学公式 $x + y = z$ 的句子。';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('这是一个包含数学公式');
      expect(result.processedText).toContain('的句子');
      expect(result.processedText).toMatch(/x \+ y = z/);
    });

    test('应该处理复杂的行内数学表达式', () => {
      const input = '$\\sum_{i=1}^{n} x_i^2 = \\int_0^\\infty e^{-x} dx$';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/sum.*int/i);
    });

    test('应该处理包含希腊字母的行内公式', () => {
      const input = '$\\alpha + \\beta = \\gamma$';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/alpha.*beta.*gamma/i);
    });

    test('应该处理分数表达式', () => {
      const input = '$\\frac{a}{b} = \\frac{c}{d}$';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/frac/i);
    });

    test('应该处理上下标', () => {
      const input = '$x^{2n+1}_i = y_{j}^{k}$';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('x');
      expect(result.processedText).toContain('y');
    });
  });

  describe('显示数学公式测试', () => {
    let parser;

    beforeEach(() => {
      parser = new LaTeXParser();
    });

    test('应该正确处理简单的显示公式 $$...$$', () => {
      const input = '$$E = mc^2$$';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/katex|math-display/i);
      expect(result.processedText).toContain('E = mc^2');
    });

    test('应该正确处理 \\[...\\] 格式的显示公式', () => {
      const input = '\\[a^2 + b^2 = c^2\\]';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/katex|math/i);
    });

    test('应该处理多行显示公式', () => {
      const input = `$$
        \\int_{0}^{\\infty} e^{-x} dx = 1
      $$`;
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/int.*infty/i);
    });

    test('应该处理复杂的积分表达式', () => {
      const input = '$$\\int_{-\\infty}^{+\\infty} e^{-x^2} dx = \\sqrt{\\pi}$$';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/int.*sqrt.*pi/i);
    });

    test('应该处理求和表达式', () => {
      const input = '$$\\sum_{k=1}^{\\infty} \\frac{1}{k^2} = \\frac{\\pi^2}{6}$$';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/sum.*frac/i);
    });

    test('应该处理极限表达式', () => {
      const input = '$$\\lim_{x \\to \\infty} \\left(1 + \\frac{1}{x}\\right)^x = e$$';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/lim.*to.*infty/i);
    });
  });

  describe('数学环境测试', () => {
    let parser;

    beforeEach(() => {
      parser = new LaTeXParser();
    });

    test('应该处理 equation 环境', () => {
      const input = `
        \\begin{equation}
        E = mc^2
        \\end{equation}
      `;
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/equation/i);
      expect(result.processedText).toContain('E = mc^2');
    });

    test('应该处理 align 环境', () => {
      const input = `
        \\begin{align}
        a &= b + c \\\\
        d &= e + f
        \\end{align}
      `;
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/align/i);
      expect(result.processedText).toContain('b + c');
      expect(result.processedText).toContain('e + f');
    });

    test('应该处理 gather 环境', () => {
      const input = `
        \\begin{gather}
        a = b + c \\\\
        d = e + f
        \\end{gather}
      `;
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/gather/i);
    });

    test('应该处理 matrix 环境', () => {
      const input = `
        \\begin{matrix}
        1 & 2 & 3 \\\\
        4 & 5 & 6 \\\\
        7 & 8 & 9
        \\end{matrix}
      `;
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/matrix/i);
      expect(result.processedText).toContain('1');
      expect(result.processedText).toContain('9');
    });

    test('应该处理 pmatrix 环境（带括号的矩阵）', () => {
      const input = `
        \\begin{pmatrix}
        a & b \\\\
        c & d
        \\end{pmatrix}
      `;
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/pmatrix/i);
    });

    test('应该处理 bmatrix 环境（带方括号的矩阵）', () => {
      const input = `
        \\begin{bmatrix}
        x & y \\\\
        z & w
        \\end{bmatrix}
      `;
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/bmatrix/i);
    });

    test('应该处理 cases 环境', () => {
      const input = `
        f(x) = \\begin{cases}
        x^2 & \\text{if } x > 0 \\\\
        0 & \\text{otherwise}
        \\end{cases}
      `;
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/cases/i);
      expect(result.processedText).toContain('x^2');
    });

    test('应该处理 split 环境', () => {
      const input = `
        \\begin{equation}
        \\begin{split}
        (a+b)^2 &= a^2 + 2ab + b^2 \\\\
                &= a^2 + b^2 + 2ab
        \\end{split}
        \\end{equation}
      `;
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/split/i);
    });
  });

  describe('数学命令和符号测试', () => {
    let parser;

    beforeEach(() => {
      parser = new LaTeXParser();
    });

    test('应该处理基本数学操作符', () => {
      const input = '$a + b - c \\times d \\div e = f$';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/times.*div/i);
    });

    test('应该处理比较运算符', () => {
      const input = '$a < b \\leq c = d \\geq e > f \\neq g$';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/leq.*geq.*neq/i);
    });

    test('应该处理集合符号', () => {
      const input = '$A \\cup B \\cap C \\subset D \\supset E \\in F \\notin G$';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/cup.*cap.*subset/i);
    });

    test('应该处理逻辑符号', () => {
      const input = '$p \\land q \\lor r \\neg s \\Rightarrow t \\Leftrightarrow u$';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/land.*lor.*neg/i);
    });

    test('应该处理微积分符号', () => {
      const input = '$\\nabla f \\partial g \\infty h$';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/nabla.*partial.*infty/i);
    });

    test('应该处理三角函数', () => {
      const input = '$\\sin x + \\cos y = \\tan z$';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/sin.*cos.*tan/i);
    });

    test('应该处理对数和指数函数', () => {
      const input = '$\\log a + \\ln b = \\exp c$';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/log.*ln.*exp/i);
    });
  });

  describe('复杂数学表达式测试', () => {
    let parser;

    beforeEach(() => {
      parser = new LaTeXParser();
    });

    test('应该处理嵌套的数学结构', () => {
      const input = '$\\sqrt{\\frac{a^2 + b^2}{\\sum_{i=1}^n x_i^2}}$';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/sqrt.*frac.*sum/i);
    });

    test('应该处理多重上下标', () => {
      const input = '$x^{n^{2}}_{{i_j}^k}$';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('x');
    });

    test('应该处理长分式', () => {
      const input = '$\\frac{\\frac{a}{b} + \\frac{c}{d}}{\\frac{e}{f} - \\frac{g}{h}}$';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/frac/i);
    });

    test('应该处理大型积分表达式', () => {
      const input = `$$
        \\oint_{\\partial D} \\mathbf{F} \\cdot d\\mathbf{r} = \\iint_D \\left(\\frac{\\partial Q}{\\partial x} - \\frac{\\partial P}{\\partial y}\\right) dx dy
      $$`;
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/oint.*iint/i);
    });
  });

  describe('数学公式错误处理测试', () => {
    let parser;

    beforeEach(() => {
      parser = new LaTeXParser();
    });

    test('应该处理不匹配的数学分隔符', () => {
      const input = '$x + y = z';  // 缺少结束的 $
      const result = parser.parseComplete(input);
      
      // tolerant 模式应该尝试恢复
      expect(result).toBeDefined();
      if (!result.success) {
        expect(result.errors.length).toBeGreaterThan(0);
      }
    });

    test('应该处理无效的数学命令', () => {
      const input = '$\\invalidcommand{x}$';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.warnings.length).toBeGreaterThan(0);
    });

    test('应该处理嵌套数学环境错误', () => {
      const input = `
        \\begin{equation}
        \\begin{align}
        x = y
        \\end{equation}
        \\end{align}
      `;
      const result = parser.parseComplete(input);
      
      expect(result).toBeDefined();
      // 应该有错误或警告
      expect(result.warnings.length + result.errors.length).toBeGreaterThan(0);
    });

    test('应该处理空的数学环境', () => {
      const input = '\\begin{equation}\\end{equation}';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
    });
  });

  describe('数学公式性能测试', () => {
    let parser;

    beforeEach(() => {
      parser = new LaTeXParser();
    });

    test('复杂数学表达式应该在合理时间内处理', () => {
      const complexMath = `
        $$\\sum_{n=1}^{\\infty} \\frac{1}{n^2} = \\frac{\\pi^2}{6}$$
        
        $$\\int_{-\\infty}^{+\\infty} e^{-\\frac{x^2}{2}} dx = \\sqrt{2\\pi}$$
        
        $$\\prod_{p \\text{ prime}} \\frac{1}{1-p^{-s}} = \\sum_{n=1}^{\\infty} \\frac{1}{n^s}$$
        
        $$\\lim_{n \\to \\infty} \\left(1 + \\frac{1}{n}\\right)^n = e$$
      `;
      
      const startTime = Date.now();
      const result = parser.parseComplete(complexMath);
      const endTime = Date.now();
      
      const processingTime = endTime - startTime;
      
      expect(result.success).toBe(true);
      expect(processingTime).toBeLessThan(2000); // 应该在2秒内完成
    });

    test('大量数学公式应该能够处理', () => {
      const manyFormulas = Array(100).fill('$x^2 + y^2 = z^2$').join(' ');
      
      const startTime = Date.now();
      const result = parser.parseComplete(manyFormulas);
      const endTime = Date.now();
      
      const processingTime = endTime - startTime;
      
      expect(result.success).toBe(true);
      expect(processingTime).toBeLessThan(3000); // 应该在3秒内完成
    });
  });

  describe('数学公式与文本混合测试', () => {
    let parser;

    beforeEach(() => {
      parser = new LaTeXParser();
    });

    test('应该正确处理文本和公式的混合', () => {
      const input = `
        勾股定理表述为 $a^2 + b^2 = c^2$，其中 $c$ 是斜边。
        
        更一般地，我们有：
        $$\\sum_{i=1}^{n} a_i^2 = c^2$$
        
        这个公式在欧几里得几何中成立。
      `;
      
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('勾股定理');
      expect(result.processedText).toContain('欧几里得几何');
      expect(result.processedText).toMatch(/a\^2 \+ b\^2 = c\^2/);
    });

    test('应该处理段落中的多个行内公式', () => {
      const input = '函数 $f(x) = ax^2 + bx + c$ 的导数是 $f\'(x) = 2ax + b$，二阶导数是 $f\'\'(x) = 2a$。';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('函数');
      expect(result.processedText).toContain('导数');
    });
  });
});