/**
 * LaTeX Parser V3 错误处理接口测试
 * 测试 tolerant/strict 模式错误处理、警告信息生成、未知命令处理等
 */

describe('LaTeX Parser V3 错误处理接口测试', () => {
  let LaTeXParser;

  beforeAll(() => {
    // 检查是否在 Node.js 环境中
    if (typeof require !== 'undefined') {
      // Node.js 环境
      const path = require('path');
      const fs = require('fs');
      
      // 检查核心文件是否存在
      const coreFiles = [
        '../core/LaTeXParser.js'
      ];
      
      for (const file of coreFiles) {
        const filePath = path.resolve(__dirname, file);
        if (!fs.existsSync(filePath)) {
          throw new Error(`核心文件不存在: ${file}`);
        }
      }
      
      // 加载 LaTeXParser
      LaTeXParser = require('../core/LaTeXParser.js');
    } else {
      // 浏览器环境
      if (typeof window !== 'undefined' && window.LaTeXParser) {
        LaTeXParser = window.LaTeXParser;
      } else {
        throw new Error('浏览器环境中 LaTeXParser 未定义');
      }
    }
  });

  describe('tolerant 模式错误恢复测试', () => {
    let parser;

    beforeEach(() => {
      parser = new LaTeXParser({ mode: 'tolerant' });
    });

    test('应该在遇到未知命令时继续处理', () => {
      const input = '正常文本 \\unknowncommand{参数} 继续文本';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('正常文本');
      expect(result.processedText).toContain('继续文本');
      expect(result.warnings.length).toBeGreaterThan(0);
      
      // 检查警告信息
      const hasUnknownCommandWarning = result.warnings.some(warning =>
        warning.message && warning.message.toLowerCase().includes('unknown')
      );
      expect(hasUnknownCommandWarning).toBe(true);
    });

    test('应该处理不匹配的环境标签', () => {
      const input = `
        \\begin{itemize}
        \\item 项目1
        \\item 项目2
        \\end{enumerate}
        后续内容
      `;
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('项目1');
      expect(result.processedText).toContain('后续内容');
      expect(result.warnings.length + result.errors.length).toBeGreaterThan(0);
    });

    test('应该处理未关闭的环境', () => {
      const input = `
        \\begin{itemize}
        \\item 项目1
        \\item 项目2
        后续内容
      `;
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('项目1');
      expect(result.processedText).toContain('后续内容');
      expect(result.warnings.length + result.errors.length).toBeGreaterThan(0);
    });

    test('应该处理不完整的命令', () => {
      const input = '文本 \\textbf{ 不完整的粗体';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('文本');
      expect(result.warnings.length + result.errors.length).toBeGreaterThan(0);
    });

    test('应该处理嵌套错误', () => {
      const input = '\\textbf{\\textit{\\unknowncommand{嵌套错误}}}';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.warnings.length).toBeGreaterThan(0);
    });

    test('应该处理数学公式中的错误', () => {
      const input = '$\\frac{a}{\\unknownmath{b}}$ 后续内容';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('后续内容');
      expect(result.warnings.length).toBeGreaterThan(0);
    });

    test('应该处理不匹配的数学分隔符', () => {
      const input = '$开始数学 但是没有结束 继续文本';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('继续文本');
      expect(result.warnings.length + result.errors.length).toBeGreaterThan(0);
    });

    test('应该处理表格中的结构错误', () => {
      const input = `
        \\begin{tabular}{|c|c|}
        \\hline
        A & B & C & D \\\\ % 列数不匹配
        \\hline
        1 & 2 \\\\
        \\end{tabular}
        后续内容
      `;
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('后续内容');
    });

    test('应该在严重错误时尝试恢复', () => {
      const input = `
        \\section{正常章节}
        \\begin{\\invalid环境名}
        \\unknowncommand
        \\textbf{未关闭
        正常文本继续
      `;
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('正常章节');
      expect(result.processedText).toContain('正常文本继续');
      expect(result.warnings.length + result.errors.length).toBeGreaterThan(0);
    });
  });

  describe('strict 模式错误处理测试', () => {
    let parser;

    beforeEach(() => {
      parser = new LaTeXParser({ mode: 'strict' });
    });

    test('应该对严重错误给出明确的错误信息', () => {
      const input = '\\unknowncommand{参数}';
      const result = parser.parseComplete(input);
      
      // strict 模式的行为可能因实现而异
      // 至少应该有明确的错误报告
      expect(result).toBeDefined();
      if (!result.success) {
        expect(result.errors.length).toBeGreaterThan(0);
        expect(result.errors[0]).toHaveProperty('message');
        expect(result.errors[0].message.length).toBeGreaterThan(0);
      }
    });

    test('应该提供详细的错误位置信息', () => {
      const input = `第一行
      第二行 \\unknowncommand{参数}
      第三行`;
      const result = parser.parseComplete(input);
      
      if (!result.success) {
        expect(result.errors.length).toBeGreaterThan(0);
        const error = result.errors[0];
        expect(error).toHaveProperty('message');
        // 可能包含行号或位置信息
        if (error.line !== undefined) {
          expect(typeof error.line).toBe('number');
        }
        if (error.column !== undefined) {
          expect(typeof error.column).toBe('number');
        }
      }
    });

    test('应该在遇到不可恢复的错误时停止处理', () => {
      const input = `
        \\section{正常章节}
        \\begin{invalidenvironment}
        这里有问题
        \\end{differentenvironment}
        \\section{这个章节可能不会被处理}
      `;
      const result = parser.parseComplete(input);
      
      // 检查错误处理的一致性
      expect(result).toBeDefined();
      if (!result.success) {
        expect(result.errors.length).toBeGreaterThan(0);
      }
    });

    test('应该区分错误的严重程度', () => {
      const minorError = '\\unknowncommand{test}';
      const majorError = '\\begin{itemize}\\end{enumerate}';
      
      const minorResult = parser.parseComplete(minorError);
      const majorResult = parser.parseComplete(majorError);
      
      expect(minorResult).toBeDefined();
      expect(majorResult).toBeDefined();
      
      // 在 strict 模式下，错误应该被适当分类
      if (!minorResult.success || !majorResult.success) {
        // 至少应该有错误信息
        const hasErrors = (minorResult.errors && minorResult.errors.length > 0) ||
                         (majorResult.errors && majorResult.errors.length > 0);
        expect(hasErrors).toBe(true);
      }
    });
  });

  describe('警告信息生成测试', () => {
    let parser;

    beforeEach(() => {
      parser = new LaTeXParser({ mode: 'tolerant' });
    });

    test('警告信息应该包含必要的字段', () => {
      const input = '\\unknowncommand{测试}';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.warnings.length).toBeGreaterThan(0);
      
      const warning = result.warnings[0];
      expect(warning).toHaveProperty('message');
      expect(typeof warning.message).toBe('string');
      expect(warning.message.length).toBeGreaterThan(0);
      
      // 可选字段
      if (warning.type !== undefined) {
        expect(typeof warning.type).toBe('string');
      }
      if (warning.line !== undefined) {
        expect(typeof warning.line).toBe('number');
      }
    });

    test('应该为不同类型的问题生成适当的警告', () => {
      const testCases = [
        { input: '\\unknowncommand{test}', expectedType: 'unknown-command' },
        { input: '\\begin{itemize}\\end{enumerate}', expectedType: 'environment-mismatch' },
        { input: '$\\invalidmath{x}$', expectedType: 'math-error' },
        { input: '\\textbf{未关闭', expectedType: 'unclosed-command' }
      ];
      
      testCases.forEach(testCase => {
        const result = parser.parseComplete(testCase.input);
        expect(result.success).toBe(true);
        expect(result.warnings.length).toBeGreaterThan(0);
        
        // 检查警告消息是否相关
        const hasRelevantWarning = result.warnings.some(warning =>
          warning.message.toLowerCase().includes('unknown') ||
          warning.message.toLowerCase().includes('mismatch') ||
          warning.message.toLowerCase().includes('error') ||
          warning.message.toLowerCase().includes('unclosed')
        );
        expect(hasRelevantWarning).toBe(true);
      });
    });

    test('应该提供有用的警告消息', () => {
      const input = '\\section{标题} \\unknowncommand{参数} \\textbf{正常}';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.warnings.length).toBeGreaterThan(0);
      
      const warning = result.warnings[0];
      expect(warning.message).toMatch(/unknown|invalid|unsupported|unrecognized/i);
      
      // 警告消息应该是描述性的
      expect(warning.message.length).toBeGreaterThan(10);
    });

    test('应该累积多个警告', () => {
      const input = `
        \\unknowncommand1{test}
        \\section{正常}
        \\unknowncommand2{test}
        \\unknowncommand3{test}
      `;
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.warnings.length).toBeGreaterThanOrEqual(3);
    });

    test('警告不应该影响成功的处理结果', () => {
      const input = '正常文本 \\unknowncommand{test} 更多正常文本';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.warnings.length).toBeGreaterThan(0);
      expect(result.processedText).toContain('正常文本');
      expect(result.processedText).toContain('更多正常文本');
    });
  });

  describe('未知命令处理测试', () => {
    let parser;

    beforeEach(() => {
      parser = new LaTeXParser({ mode: 'tolerant' });
    });

    test('应该识别并报告未知的文档命令', () => {
      const input = '\\newsection{新类型章节}';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.warnings.length).toBeGreaterThan(0);
      
      const hasUnknownCommandWarning = result.warnings.some(warning =>
        warning.message.toLowerCase().includes('unknown') &&
        warning.message.includes('newsection')
      );
      expect(hasUnknownCommandWarning).toBe(true);
    });

    test('应该识别并报告未知的格式化命令', () => {
      const input = '\\customformat{特殊格式}';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.warnings.length).toBeGreaterThan(0);
    });

    test('应该识别并报告未知的数学命令', () => {
      const input = '$\\custommath{x} + y$';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.warnings.length).toBeGreaterThan(0);
    });

    test('应该识别并报告未知的环境', () => {
      const input = `
        \\begin{customenvironment}
        内容
        \\end{customenvironment}
      `;
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.warnings.length).toBeGreaterThan(0);
    });

    test('应该保留未知命令的参数内容', () => {
      const input = '\\unknowncommand{重要内容}';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('重要内容');
      expect(result.warnings.length).toBeGreaterThan(0);
    });

    test('应该处理带有多个参数的未知命令', () => {
      const input = '\\unknowncommand{参数1}{参数2}[可选参数]';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.warnings.length).toBeGreaterThan(0);
      
      // 应该保留参数内容
      expect(result.processedText).toContain('参数1');
      expect(result.processedText).toContain('参数2');
    });

    test('应该处理嵌套中的未知命令', () => {
      const input = '\\textbf{粗体 \\unknowncommand{嵌套} 继续粗体}';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('粗体');
      expect(result.processedText).toContain('嵌套');
      expect(result.processedText).toContain('继续粗体');
      expect(result.warnings.length).toBeGreaterThan(0);
    });

    test('应该提供建议或替代方案（如果有）', () => {
      // 测试常见的拼写错误
      const typos = [
        { input: '\\setcion{章节}', suggestion: 'section' },
        { input: '\\textb{粗体}', suggestion: 'textbf' },
        { input: '\\beign{itemize}', suggestion: 'begin' }
      ];
      
      typos.forEach(typo => {
        const result = parser.parseComplete(typo.input);
        expect(result.success).toBe(true);
        expect(result.warnings.length).toBeGreaterThan(0);
        
        // 检查是否有建议信息
        const hasSuggestion = result.warnings.some(warning =>
          warning.message.toLowerCase().includes('suggest') ||
          warning.message.toLowerCase().includes('mean') ||
          warning.message.includes(typo.suggestion)
        );
        
        if (hasSuggestion) {
          expect(hasSuggestion).toBe(true);
        }
      });
    });
  });

  describe('错误信息质量测试', () => {
    test('错误信息应该是有用和描述性的', () => {
      const parser = new LaTeXParser({ mode: 'strict' });
      const input = '\\begin{itemize}\\item 项目\\end{enumerate}';
      const result = parser.parseComplete(input);
      
      const allMessages = [...(result.errors || []), ...(result.warnings || [])];
      expect(allMessages.length).toBeGreaterThan(0);
      
      const message = allMessages[0];
      expect(message).toHaveProperty('message');
      expect(message.message.length).toBeGreaterThan(20); // 应该是描述性的
      expect(message.message).not.toMatch(/^Error$/i); // 不应该是通用错误
    });

    test('错误信息应该包含上下文信息', () => {
      const parser = new LaTeXParser({ mode: 'tolerant' });
      const input = `
        \\section{第一章}
        这里是正常内容
        \\unknowncommand{这里有问题}
        继续的内容
      `;
      const result = parser.parseComplete(input);
      
      expect(result.warnings.length).toBeGreaterThan(0);
      const warning = result.warnings[0];
      
      // 应该包含命令名称
      expect(warning.message).toMatch(/unknowncommand/i);
    });

    test('应该提供错误恢复的策略信息', () => {
      const parser = new LaTeXParser({ mode: 'tolerant' });
      const input = '\\begin{itemize}\\item 项目\\end{enumerate}';
      const result = parser.parseComplete(input);
      
      const allMessages = [...(result.errors || []), ...(result.warnings || [])];
      expect(allMessages.length).toBeGreaterThan(0);
      
      // 检查是否有恢复策略的提示
      const hasRecoveryInfo = allMessages.some(msg =>
        msg.message.toLowerCase().includes('recover') ||
        msg.message.toLowerCase().includes('attempt') ||
        msg.message.toLowerCase().includes('assume')
      );
      
      // 这是可选的，但是好的实现应该有
      if (hasRecoveryInfo) {
        expect(hasRecoveryInfo).toBe(true);
      }
    });
  });

  describe('错误处理性能测试', () => {
    test('错误处理不应该显著影响性能', () => {
      const parser = new LaTeXParser({ mode: 'tolerant' });
      
      // 创建包含多个错误的大型文档
      const errorContent = Array(100).fill('\\unknowncommand{test}').join(' ');
      const normalContent = Array(100).fill('正常文本').join(' ');
      
      // 测试正常内容
      const normalStart = Date.now();
      const normalResult = parser.parseComplete(normalContent);
      const normalTime = Date.now() - normalStart;
      
      // 测试错误内容
      const errorStart = Date.now();
      const errorResult = parser.parseComplete(errorContent);
      const errorTime = Date.now() - errorStart;
      
      expect(normalResult.success).toBe(true);
      expect(errorResult.success).toBe(true);
      expect(errorResult.warnings.length).toBeGreaterThan(0);
      
      // 错误处理时间不应该超过正常处理时间的10倍
      expect(errorTime).toBeLessThan(normalTime * 10 + 1000);
    });

    test('大量错误不应该导致内存泄漏', () => {
      const parser = new LaTeXParser({ mode: 'tolerant' });
      
      // 生成大量不同的错误
      const largeErrorInput = [];
      for (let i = 0; i < 1000; i++) {
        largeErrorInput.push(`\\unknowncommand${i}{test}`);
      }
      
      const input = largeErrorInput.join(' ');
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.warnings.length).toBeGreaterThan(0);
      
      // 警告数量应该是合理的（可能会合并相似的警告）
      expect(result.warnings.length).toBeLessThan(10000);
    });
  });

  describe('错误恢复策略测试', () => {
    let parser;

    beforeEach(() => {
      parser = new LaTeXParser({ mode: 'tolerant' });
    });

    test('应该能够从命令错误中恢复', () => {
      const input = `
        \\section{正常章节}
        \\unknowncommand{错误}
        \\subsection{应该继续处理}
        正常文本
      `;
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('正常章节');
      expect(result.processedText).toContain('应该继续处理');
      expect(result.processedText).toContain('正常文本');
    });

    test('应该能够从环境错误中恢复', () => {
      const input = `
        \\begin{itemize}
        \\item 项目1
        \\begin{unknownenv}
        错误环境
        \\end{unknownenv}
        \\item 项目2
        \\end{itemize}
        后续内容
      `;
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('项目1');
      expect(result.processedText).toContain('项目2');
      expect(result.processedText).toContain('后续内容');
    });

    test('应该能够从数学错误中恢复', () => {
      const input = `
        数学公式: $x + y = z$
        错误公式: $\\invalidmath{test}$
        继续: $a^2 + b^2 = c^2$
      `;
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('x + y = z');
      expect(result.processedText).toContain('a^2 + b^2 = c^2');
    });
  });
});