/**
 * LaTeX Parser V3 文档结构接口测试
 * 测试章节、文本格式化、列表环境、表格环境等
 */

describe('LaTeX Parser V3 文档结构接口测试', () => {
  let LaTeXParser;

  beforeAll(() => {
    // 检查是否在 Node.js 环境中
    if (typeof require !== 'undefined') {
      // Node.js 环境
      const path = require('path');
      const fs = require('fs');
      
      // 检查核心文件是否存在
      const coreFiles = [
        '../core/DocumentParser.js',
        '../core/BlockElementParser.js',
        '../core/InlineElementParser.js',
        '../core/LaTeXParser.js'
      ];
      
      for (const file of coreFiles) {
        const filePath = path.resolve(__dirname, file);
        if (!fs.existsSync(filePath)) {
          throw new Error(`核心文件不存在: ${file}`);
        }
      }
      
      // 加载 LaTeXParser
      LaTeXParser = require('../core/LaTeXParser.js');
    } else {
      // 浏览器环境
      if (typeof window !== 'undefined' && window.LaTeXParser) {
        LaTeXParser = window.LaTeXParser;
      } else {
        throw new Error('浏览器环境中 LaTeXParser 未定义');
      }
    }
  });

  describe('章节结构测试', () => {
    let parser;

    beforeEach(() => {
      parser = new LaTeXParser();
    });

    test('应该处理 \\section 命令', () => {
      const input = '\\section{介绍}';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('介绍');
      expect(result.processedText).toMatch(/<h[1-6]|section/i);
    });

    test('应该处理 \\subsection 命令', () => {
      const input = '\\subsection{子章节}';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('子章节');
      expect(result.processedText).toMatch(/<h[1-6]|subsection/i);
    });

    test('应该处理 \\subsubsection 命令', () => {
      const input = '\\subsubsection{子子章节}';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('子子章节');
      expect(result.processedText).toMatch(/<h[1-6]|subsubsection/i);
    });

    test('应该处理 \\chapter 命令', () => {
      const input = '\\chapter{第一章}';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('第一章');
      expect(result.processedText).toMatch(/<h[1-6]|chapter/i);
    });

    test('应该处理 \\paragraph 命令', () => {
      const input = '\\paragraph{段落标题}';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('段落标题');
    });

    test('应该处理嵌套的章节结构', () => {
      const input = `
        \\section{主章节}
        这是主章节内容。
        
        \\subsection{子章节}
        这是子章节内容。
        
        \\subsubsection{子子章节}
        这是子子章节内容。
      `;
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('主章节');
      expect(result.processedText).toContain('子章节');
      expect(result.processedText).toContain('子子章节');
    });

    test('应该处理包含特殊字符的章节标题', () => {
      const input = '\\section{特殊字符: &<>"\'\\n\\t测试}';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('特殊字符');
    });

    test('应该处理空章节标题', () => {
      const input = '\\section{}';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
    });
  });

  describe('文本格式化测试', () => {
    let parser;

    beforeEach(() => {
      parser = new LaTeXParser();
    });

    test('应该处理 \\textbf 粗体命令', () => {
      const input = '\\textbf{这是粗体文本}';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/<strong>.*这是粗体文本.*<\/strong>/);
    });

    test('应该处理 \\textit 斜体命令', () => {
      const input = '\\textit{这是斜体文本}';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/<em>.*这是斜体文本.*<\/em>/i);
    });

    test('应该处理 \\emph 强调命令', () => {
      const input = '\\emph{这是强调文本}';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/<em>.*这是强调文本.*<\/em>/i);
    });

    test('应该处理 \\underline 下划线命令', () => {
      const input = '\\underline{这是下划线文本}';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('这是下划线文本');
      expect(result.processedText).toMatch(/<span.*underline|text-decoration.*underline/i);
    });

    test('应该处理 \\texttt 等宽字体命令', () => {
      const input = '\\texttt{这是等宽字体}';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('这是等宽字体');
      expect(result.processedText).toMatch(/<code|monospace|courier/i);
    });

    test('应该处理嵌套的格式化命令', () => {
      const input = '\\textbf{粗体 \\textit{加斜体} 文本}';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('粗体');
      expect(result.processedText).toContain('加斜体');
      expect(result.processedText).toMatch(/<strong>.*<em>.*<\/em>.*<\/strong>/i);
    });

    test('应该处理多种格式化的组合', () => {
      const input = '普通文本 \\textbf{粗体} 和 \\textit{斜体} 以及 \\texttt{等宽字体}。';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('普通文本');
      expect(result.processedText).toContain('粗体');
      expect(result.processedText).toContain('斜体');
      expect(result.processedText).toContain('等宽字体');
    });

    test('应该处理字体大小命令', () => {
      const commands = [
        '\\tiny{微小文字}',
        '\\small{小号文字}',
        '\\large{大号文字}',
        '\\Large{更大文字}',
        '\\huge{巨大文字}'
      ];
      
      commands.forEach(command => {
        const result = parser.parseComplete(command);
        expect(result.success).toBe(true);
        expect(result.processedText.length).toBeGreaterThan(0);
      });
    });
  });

  describe('列表环境测试', () => {
    let parser;

    beforeEach(() => {
      parser = new LaTeXParser();
    });

    test('应该处理 itemize 无序列表', () => {
      const input = `
        \\begin{itemize}
        \\item 第一项
        \\item 第二项
        \\item 第三项
        \\end{itemize}
      `;
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/<ul.*>.*<li.*>.*第一项.*<\/li>.*<\/ul>/s);
      expect(result.processedText).toContain('第二项');
      expect(result.processedText).toContain('第三项');
    });

    test('应该处理 enumerate 有序列表', () => {
      const input = `
        \\begin{enumerate}
        \\item 步骤一
        \\item 步骤二
        \\item 步骤三
        \\end{enumerate}
      `;
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/<ol.*>.*<li.*>.*步骤一.*<\/li>.*<\/ol>/s);
      expect(result.processedText).toContain('步骤二');
      expect(result.processedText).toContain('步骤三');
    });

    test('应该处理 description 描述列表', () => {
      const input = `
        \\begin{description}
        \\item[术语1] 这是术语1的描述
        \\item[术语2] 这是术语2的描述
        \\end{description}
      `;
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('术语1');
      expect(result.processedText).toContain('术语1的描述');
      expect(result.processedText).toContain('术语2');
    });

    test('应该处理嵌套列表', () => {
      const input = `
        \\begin{itemize}
        \\item 主项目1
        \\begin{itemize}
        \\item 子项目1.1
        \\item 子项目1.2
        \\end{itemize}
        \\item 主项目2
        \\end{itemize}
      `;
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('主项目1');
      expect(result.processedText).toContain('子项目1.1');
      expect(result.processedText).toContain('主项目2');
    });

    test('应该处理列表项中的格式化', () => {
      const input = `
        \\begin{itemize}
        \\item \\textbf{粗体项目}
        \\item \\textit{斜体项目}
        \\item 包含数学 $x^2 = 4$
        \\end{itemize}
      `;
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('粗体项目');
      expect(result.processedText).toContain('斜体项目');
      expect(result.processedText).toContain('包含数学');
    });

    test('应该处理空列表', () => {
      const input = '\\begin{itemize}\\end{itemize}';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/<ul.*>.*<\/ul>/s);
    });

    test('应该处理复杂的混合列表', () => {
      const input = `
        \\begin{enumerate}
        \\item 有序项目
        \\begin{itemize}
        \\item 无序子项
        \\begin{description}
        \\item[术语] 描述内容
        \\end{description}
        \\end{itemize}
        \\end{enumerate}
      `;
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('有序项目');
      expect(result.processedText).toContain('无序子项');
      expect(result.processedText).toContain('术语');
    });
  });

  describe('表格环境测试', () => {
    let parser;

    beforeEach(() => {
      parser = new LaTeXParser();
    });

    test('应该处理简单的 tabular 表格', () => {
      const input = `
        \\begin{tabular}{|c|c|c|}
        \\hline
        A & B & C \\\\
        \\hline
        1 & 2 & 3 \\\\
        4 & 5 & 6 \\\\
        \\hline
        \\end{tabular}
      `;
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/<table.*>.*<\/table>/s);
      expect(result.processedText).toContain('A');
      expect(result.processedText).toContain('B');
      expect(result.processedText).toContain('C');
      expect(result.processedText).toContain('1');
      expect(result.processedText).toContain('6');
    });

    test('应该处理不同对齐方式的表格', () => {
      const alignments = ['l', 'c', 'r'];
      alignments.forEach(align => {
        const input = `
          \\begin{tabular}{${align}${align}${align}}
          左 & 中 & 右 \\\\
          \\end{tabular}
        `;
        const result = parser.parseComplete(input);
        
        expect(result.success).toBe(true);
        expect(result.processedText).toContain('左');
        expect(result.processedText).toContain('中');
        expect(result.processedText).toContain('右');
      });
    });

    test('应该处理 array 数学表格', () => {
      const input = `
        $\\begin{array}{ccc}
        a & b & c \\\\
        d & e & f
        \\end{array}$
      `;
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/array/i);
      expect(result.processedText).toContain('a');
      expect(result.processedText).toContain('f');
    });

    test('应该处理表格中的格式化内容', () => {
      const input = `
        \\begin{tabular}{|c|c|}
        \\hline
        \\textbf{标题1} & \\textbf{标题2} \\\\
        \\hline
        \\textit{斜体} & $x^2$ \\\\
        \\end{tabular}
      `;
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('标题1');
      expect(result.processedText).toContain('标题2');
      expect(result.processedText).toContain('斜体');
    });

    test('应该处理多行表格内容', () => {
      const input = `
        \\begin{tabular}{|p{3cm}|p{3cm}|}
        \\hline
        这是一个很长的\\\\多行内容 & 另一个\\\\多行单元格 \\\\
        \\hline
        \\end{tabular}
      `;
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('很长的');
      expect(result.processedText).toContain('多行内容');
    });

    test('应该处理表格边框和线条', () => {
      const input = `
        \\begin{tabular}{|c||c|c|}
        \\hline
        \\hline
        A & B & C \\\\
        \\hline
        1 & 2 & 3 \\\\
        \\cline{1-2}
        4 & 5 & 6 \\\\
        \\hline
        \\hline
        \\end{tabular}
      `;
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/<table.*>.*<\/table>/s);
    });
  });

  describe('document 环境测试', () => {
    let parser;

    beforeEach(() => {
      parser = new LaTeXParser();
    });

    test('应该处理 document 环境', () => {
      const input = `
        \\begin{document}
        这是文档内容
        \\section{章节}
        更多内容
        \\end{document}
      `;
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('这是文档内容');
      expect(result.processedText).toContain('章节');
      expect(result.processedText).toContain('更多内容');
    });

    test('应该处理完整的文档结构', () => {
      const input = `
        \\documentclass{article}
        \\usepackage{amsmath}
        
        \\begin{document}
        \\title{文档标题}
        \\author{作者姓名}
        \\date{\\today}
        \\maketitle
        
        \\section{介绍}
        这是介绍部分。
        
        \\section{内容}
        这是主要内容。
        
        \\end{document}
      `;
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('介绍');
      expect(result.processedText).toContain('主要内容');
    });
  });

  describe('特殊环境和命令测试', () => {
    let parser;

    beforeEach(() => {
      parser = new LaTeXParser();
    });

    test('应该处理 center 环境', () => {
      const input = `
        \\begin{center}
        居中的内容
        \\end{center}
      `;
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('居中的内容');
    });

    test('应该处理 quote 和 quotation 环境', () => {
      const quoteInput = `
        \\begin{quote}
        这是一个引用
        \\end{quote}
      `;
      
      const quotationInput = `
        \\begin{quotation}
        这是一个长引用
        \\end{quotation}
      `;
      
      const quoteResult = parser.parseComplete(quoteInput);
      const quotationResult = parser.parseComplete(quotationInput);
      
      expect(quoteResult.success).toBe(true);
      expect(quotationResult.success).toBe(true);
      expect(quoteResult.processedText).toContain('这是一个引用');
      expect(quotationResult.processedText).toContain('这是一个长引用');
    });

    test('应该处理 verbatim 环境', () => {
      const input = `
        \\begin{verbatim}
        这是原样文本
        包含 \\LaTeX 命令
        \\end{verbatim}
      `;
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('这是原样文本');
    });

    test('应该处理换行和换页命令', () => {
      const input = '第一行\\\\第二行\\newline第三行\\newpage第四行';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('第一行');
      expect(result.processedText).toContain('第四行');
    });

    test('应该处理空格命令', () => {
      const input = '单词1\\quad单词2\\qquad单词3\\ 单词4';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('单词1');
      expect(result.processedText).toContain('单词4');
    });
  });

  describe('复杂文档结构测试', () => {
    let parser;

    beforeEach(() => {
      parser = new LaTeXParser();
    });

    test('应该处理包含所有元素的复杂文档', () => {
      const input = `
        \\section{综合测试}
        
        这是一个包含多种元素的测试文档。
        
        \\subsection{格式化文本}
        普通文本 \\textbf{粗体} \\textit{斜体} \\texttt{等宽}
        
        \\subsection{数学公式}
        行内公式 $E = mc^2$ 和显示公式：
        $$\\sum_{i=1}^n x_i = \\int_0^1 f(x)dx$$
        
        \\subsection{列表}
        \\begin{itemize}
        \\item 项目1
        \\item 项目2 包含数学 $\\alpha + \\beta$
        \\end{itemize}
        
        \\subsection{表格}
        \\begin{tabular}{|c|c|}
        \\hline
        A & B \\\\
        \\hline
        1 & 2 \\\\
        \\hline
        \\end{tabular}
      `;
      
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('综合测试');
      expect(result.processedText).toContain('格式化文本');
      expect(result.processedText).toContain('数学公式');
      expect(result.processedText).toContain('列表');
      expect(result.processedText).toContain('表格');
    });

    test('应该处理性能要求高的大型文档', () => {
      // 生成一个包含多种结构的大型文档
      const sections = [];
      for (let i = 1; i <= 10; i++) {
        sections.push(`
          \\section{第${i}章}
          \\subsection{章节${i}.1}
          这是第${i}章的内容，包含数学公式 $x_${i} = ${i}^2$
          
          \\begin{itemize}
          \\item 项目${i}.1
          \\item 项目${i}.2
          \\end{itemize}
          
          显示数学：$$\\sum_{k=1}^{${i}} k = \\frac{${i}(${i}+1)}{2}$$
        `);
      }
      
      const largeInput = sections.join('\n');
      
      const startTime = Date.now();
      const result = parser.parseComplete(largeInput);
      const endTime = Date.now();
      
      const processingTime = endTime - startTime;
      
      expect(result.success).toBe(true);
      expect(processingTime).toBeLessThan(5000); // 5秒内完成
      expect(result.processedText).toContain('第1章');
      expect(result.processedText).toContain('第10章');
    });
  });

  describe('错误恢复和警告测试', () => {
    let parser;

    beforeEach(() => {
      parser = new LaTeXParser();
    });

    test('应该处理不匹配的环境', () => {
      const input = `
        \\begin{itemize}
        \\item 项目1
        \\end{enumerate}
      `;
      const result = parser.parseComplete(input);
      
      expect(result).toBeDefined();
      expect(result.warnings.length + result.errors.length).toBeGreaterThan(0);
    });

    test('应该处理未知命令', () => {
      const input = '\\unknowncommand{参数} 普通文本';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.warnings.length).toBeGreaterThan(0);
      expect(result.processedText).toContain('普通文本');
    });

    test('应该处理嵌套错误', () => {
      const input = `
        \\section{\\textbf{\\unknowncommand{嵌套}错误}}
        内容继续
      `;
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('内容继续');
    });

    test('应该处理空参数', () => {
      const input = '\\section{} \\textbf{} \\textit{内容}';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('内容');
    });
  });
});