/**
 * LaTeX Parser V3 浏览器环境接口测试
 * 测试 browser.js 提供的全局函数、parseLatex 便捷函数、createLaTeXParser 工厂函数
 */

describe('LaTeX Parser V3 浏览器环境接口测试', () => {
  let originalWindow;
  let mockBrowser;

  beforeAll(() => {
    // 保存原始环境
    if (typeof window !== 'undefined') {
      originalWindow = window;
    }

    // 检查是否在 Node.js 环境中
    if (typeof require !== 'undefined') {
      // Node.js 环境 - 模拟浏览器环境
      const path = require('path');
      const fs = require('fs');
      
      // 检查 browser.js 文件是否存在
      const browserFile = path.resolve(__dirname, '../browser.js');
      if (!fs.existsSync(browserFile)) {
        throw new Error('browser.js 文件不存在');
      }

      // 模拟浏览器全局对象
      global.window = global.window || {};
      
      // 加载核心文件到全局作用域（模拟浏览器环境）
      const coreFiles = [
        '../core/DocumentParser.js',
        '../core/BlockElementParser.js',
        '../core/InlineElementParser.js', 
        '../core/MathParser.js',
        '../core/SpecialParser.js',
        '../core/LaTeXParser.js'
      ];

      // 模拟加载核心类
      try {
        global.LaTeXParser = require('../core/LaTeXParser.js');
        global.window.LaTeXParser = global.LaTeXParser;
      } catch (error) {
        console.warn('无法加载核心 LaTeX Parser，某些测试可能会跳过');
      }

      // 加载 browser.js
      try {
        const browserCode = fs.readFileSync(browserFile, 'utf8');
        // 在全局作用域中执行 browser.js 的代码
        eval(browserCode);
        
        // 将浏览器函数添加到 global 和 window
        if (typeof parseLatex !== 'undefined') {
          global.parseLatex = parseLatex;
          global.window.parseLatex = parseLatex;
        }
        if (typeof createLaTeXParser !== 'undefined') {
          global.createLaTeXParser = createLaTeXParser;
          global.window.createLaTeXParser = createLaTeXParser;
        }
        if (typeof LaTeXBrowser !== 'undefined') {
          global.LaTeXBrowser = LaTeXBrowser;
          global.window.LaTeXBrowser = LaTeXBrowser;
        }
      } catch (error) {
        console.warn('无法加载 browser.js:', error.message);
      }
    }
  });

  afterAll(() => {
    // 恢复原始环境
    if (originalWindow) {
      global.window = originalWindow;
    } else {
      delete global.window;
    }
  });

  describe('浏览器环境全局对象检测', () => {
    test('LaTeXParser 应该在全局作用域中可用', () => {
      // 在浏览器环境中，或者模拟的 Node.js 环境中
      const hasLaTeXParser = (typeof window !== 'undefined' && window.LaTeXParser) ||
                            (typeof global !== 'undefined' && global.LaTeXParser) ||
                            (typeof LaTeXParser !== 'undefined');
      
      if (!hasLaTeXParser) {
        console.warn('LaTeXParser 未在全局作用域中找到，跳过相关测试');
        return;
      }

      expect(hasLaTeXParser).toBe(true);
    });

    test('应该在浏览器环境中提供便捷函数', () => {
      const hasParseLatex = (typeof window !== 'undefined' && window.parseLatex) ||
                           (typeof global !== 'undefined' && global.parseLatex) ||
                           (typeof parseLatex !== 'undefined');

      const hasCreateParser = (typeof window !== 'undefined' && window.createLaTeXParser) ||
                             (typeof global !== 'undefined' && global.createLaTeXParser) ||
                             (typeof createLaTeXParser !== 'undefined');

      if (!hasParseLatex || !hasCreateParser) {
        console.warn('浏览器便捷函数未找到，跳过相关测试');
        return;
      }

      expect(hasParseLatex).toBe(true);
      expect(hasCreateParser).toBe(true);
    });
  });

  describe('parseLatex 便捷函数测试', () => {
    let parseLatexFunc;

    beforeEach(() => {
      // 获取 parseLatex 函数
      parseLatexFunc = (typeof window !== 'undefined' && window.parseLatex) ||
                      (typeof global !== 'undefined' && global.parseLatex) ||
                      (typeof parseLatex !== 'undefined' && parseLatex);

      if (!parseLatexFunc) {
        console.warn('parseLatex 函数不可用，跳过测试');
      }
    });

    test('应该能够直接解析 LaTeX 内容', () => {
      if (!parseLatexFunc) return;

      const result = parseLatexFunc('\\section{测试}');
      
      expect(result).toBeDefined();
      expect(result.success).toBeDefined();
      expect(result.processedText).toBeDefined();
    });

    test('应该能够使用配置选项', () => {
      if (!parseLatexFunc) return;

      const result = parseLatexFunc('\\section{测试}', {
        mode: 'tolerant',
        enabledLayers: ['document']
      });
      
      expect(result).toBeDefined();
      expect(result.success).toBeDefined();
    });

    test('应该处理数学公式', () => {
      if (!parseLatexFunc) return;

      const result = parseLatexFunc('$E = mc^2$');
      
      expect(result).toBeDefined();
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('E = mc^2');
    });

    test('应该处理复杂的 LaTeX 内容', () => {
      if (!parseLatexFunc) return;

      const input = `
        \\section{标题}
        文本 \\textbf{粗体} 和数学 $x^2$
        
        \\begin{itemize}
        \\item 项目1
        \\item 项目2
        \\end{itemize}
      `;
      
      const result = parseLatexFunc(input);
      
      expect(result).toBeDefined();
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('标题');
    });

    test('应该处理错误输入', () => {
      if (!parseLatexFunc) return;

      const result = parseLatexFunc('\\unknowncommand{test}');
      
      expect(result).toBeDefined();
      expect(result.success).toBe(true);
      expect(result.warnings.length).toBeGreaterThan(0);
    });

    test('应该处理空输入', () => {
      if (!parseLatexFunc) return;

      const result = parseLatexFunc('');
      
      expect(result).toBeDefined();
      expect(result.success).toBe(true);
      expect(result.processedText).toBe('');
    });

    test('应该处理 null 和 undefined 输入', () => {
      if (!parseLatexFunc) return;

      expect(() => parseLatexFunc(null)).not.toThrow();
      expect(() => parseLatexFunc(undefined)).not.toThrow();
    });
  });

  describe('createLaTeXParser 工厂函数测试', () => {
    let createParserFunc;

    beforeEach(() => {
      // 获取 createLaTeXParser 函数
      createParserFunc = (typeof window !== 'undefined' && window.createLaTeXParser) ||
                        (typeof global !== 'undefined' && global.createLaTeXParser) ||
                        (typeof createLaTeXParser !== 'undefined' && createLaTeXParser);

      if (!createParserFunc) {
        console.warn('createLaTeXParser 函数不可用，跳过测试');
      }
    });

    test('应该能够创建默认配置的解析器', () => {
      if (!createParserFunc) return;

      const parser = createParserFunc();
      
      expect(parser).toBeDefined();
      expect(typeof parser.parseComplete).toBe('function');
    });

    test('应该能够使用自定义配置创建解析器', () => {
      if (!createParserFunc) return;

      const config = {
        mode: 'strict',
        enabledLayers: ['document', 'math'],
        maxRecursionDepth: 50
      };
      
      const parser = createParserFunc(config);
      
      expect(parser).toBeDefined();
      expect(parser.config).toBeDefined();
      expect(parser.config.mode).toBe('strict');
    });

    test('创建的解析器应该能够正常工作', () => {
      if (!createParserFunc) return;

      const parser = createParserFunc();
      const result = parser.parseComplete('\\section{测试章节}');
      
      expect(result).toBeDefined();
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('测试章节');
    });

    test('应该能够创建不同配置的多个解析器', () => {
      if (!createParserFunc) return;

      const parser1 = createParserFunc({ enabledLayers: ['math'] });
      const parser2 = createParserFunc({ enabledLayers: ['document'] });
      
      expect(parser1).toBeDefined();
      expect(parser2).toBeDefined();
      expect(parser1.config.enabledLayers).toEqual(['math']);
      expect(parser2.config.enabledLayers).toEqual(['document']);
    });

    test('创建的解析器应该是独立的实例', () => {
      if (!createParserFunc) return;

      const parser1 = createParserFunc({ mode: 'strict' });
      const parser2 = createParserFunc({ mode: 'tolerant' });
      
      expect(parser1.config.mode).toBe('strict');
      expect(parser2.config.mode).toBe('tolerant');
      expect(parser1).not.toBe(parser2);
    });
  });

  describe('浏览器环境集成测试', () => {
    test('应该能够模拟完整的浏览器使用场景', () => {
      // 检查必要的函数是否可用
      const parseLatexFunc = (typeof window !== 'undefined' && window.parseLatex) ||
                            (typeof global !== 'undefined' && global.parseLatex) ||
                            (typeof parseLatex !== 'undefined' && parseLatex);

      const createParserFunc = (typeof window !== 'undefined' && window.createLaTeXParser) ||
                              (typeof global !== 'undefined' && global.createLaTeXParser) ||
                              (typeof createLaTeXParser !== 'undefined' && createLaTeXParser);

      if (!parseLatexFunc || !createParserFunc) {
        console.warn('浏览器函数不可用，跳过集成测试');
        return;
      }

      // 模拟用户场景1：直接使用便捷函数
      const quickResult = parseLatexFunc('快速测试 $x^2$');
      expect(quickResult.success).toBe(true);

      // 模拟用户场景2：创建专用解析器
      const mathParser = createParserFunc({ enabledLayers: ['math'] });
      const mathResult = mathParser.parseComplete('$\\sum_{i=1}^n i = \\frac{n(n+1)}{2}$');
      expect(mathResult.success).toBe(true);

      // 模拟用户场景3：处理复杂文档
      const docParser = createParserFunc({ mode: 'tolerant' });
      const complexDoc = `
        \\section{数学公式集合}
        
        这里有一些数学公式：
        
        行内公式：$E = mc^2$
        
        显示公式：
        $$\\int_0^\\infty e^{-x}dx = 1$$
        
        \\begin{itemize}
        \\item 第一个公式：$\\alpha + \\beta = \\gamma$
        \\item 第二个公式：$\\sum_{k=1}^\\infty \\frac{1}{k^2} = \\frac{\\pi^2}{6}$
        \\end{itemize}
      `;
      
      const docResult = docParser.parseComplete(complexDoc);
      expect(docResult.success).toBe(true);
      expect(docResult.processedText).toContain('数学公式集合');
    });

    test('应该处理浏览器环境中的性能要求', () => {
      const parseLatexFunc = (typeof window !== 'undefined' && window.parseLatex) ||
                            (typeof global !== 'undefined' && global.parseLatex) ||
                            (typeof parseLatex !== 'undefined' && parseLatex);

      if (!parseLatexFunc) {
        console.warn('parseLatex 函数不可用，跳过性能测试');
        return;
      }

      // 测试快速响应
      const startTime = Date.now();
      const result = parseLatexFunc('简单测试 $x + y = z$');
      const endTime = Date.now();
      
      const processingTime = endTime - startTime;
      
      expect(result.success).toBe(true);
      expect(processingTime).toBeLessThan(1000); // 浏览器环境应该快速响应
    });

    test('应该能够处理浏览器中的错误情况', () => {
      const parseLatexFunc = (typeof window !== 'undefined' && window.parseLatex) ||
                            (typeof global !== 'undefined' && global.parseLatex) ||
                            (typeof parseLatex !== 'undefined' && parseLatex);

      if (!parseLatexFunc) return;

      // 测试各种错误情况
      const errorInputs = [
        '\\begin{itemize}\\item 项目\\end{enumerate}', // 不匹配的环境
        '\\unknowncommand{参数}',                        // 未知命令
        '$\\frac{1}{0}$',                               // 数学表达式
        '',                                             // 空输入
        null,                                           // null 输入
        undefined                                       // undefined 输入
      ];

      errorInputs.forEach((input, index) => {
        expect(() => {
          const result = parseLatexFunc(input);
          expect(result).toBeDefined();
        }).not.toThrow(`错误输入 ${index}: ${input}`);
      });
    });
  });

  describe('浏览器兼容性测试', () => {
    test('应该在不同的全局作用域配置下工作', () => {
      // 测试不同的全局对象配置
      const scenarios = [
        { name: 'window 对象', globalObj: typeof window !== 'undefined' ? window : null },
        { name: 'global 对象', globalObj: typeof global !== 'undefined' ? global : null }
      ];

      scenarios.forEach(scenario => {
        if (!scenario.globalObj) return;

        const hasLaTeXParser = scenario.globalObj.LaTeXParser;
        const hasParseLatex = scenario.globalObj.parseLatex;
        const hasCreateParser = scenario.globalObj.createLaTeXParser;

        if (hasLaTeXParser || hasParseLatex || hasCreateParser) {
          console.log(`${scenario.name} 环境下可用的函数:`, {
            LaTeXParser: !!hasLaTeXParser,
            parseLatex: !!hasParseLatex,
            createLaTeXParser: !!hasCreateParser
          });
        }
      });
    });

    test('应该提供适当的错误信息当核心类不可用时', () => {
      // 临时移除核心类来测试错误处理
      const originalLaTeXParser = global.LaTeXParser;
      delete global.LaTeXParser;
      
      try {
        const parseLatexFunc = (typeof parseLatex !== 'undefined' && parseLatex);
        if (parseLatexFunc) {
          expect(() => parseLatexFunc('test')).toThrow();
        }
      } catch (error) {
        expect(error.message).toMatch(/LaTeXParser.*not.*defined|unavailable/i);
      } finally {
        // 恢复核心类
        if (originalLaTeXParser) {
          global.LaTeXParser = originalLaTeXParser;
        }
      }
    });
  });

  describe('浏览器环境文档加载顺序测试', () => {
    test('应该能够检测正确的加载顺序', () => {
      const expectedOrder = [
        'DocumentParser',
        'BlockElementParser', 
        'InlineElementParser',
        'MathParser',
        'SpecialParser',
        'LaTeXParser'
      ];

      const loadedClasses = [];
      expectedOrder.forEach(className => {
        const isLoaded = (typeof window !== 'undefined' && window[className]) ||
                        (typeof global !== 'undefined' && global[className]) ||
                        (typeof eval(`typeof ${className}`) !== 'undefined');
        
        if (isLoaded) {
          loadedClasses.push(className);
        }
      });

      // 至少应该有 LaTeXParser 可用
      const hasMainClass = loadedClasses.includes('LaTeXParser');
      if (hasMainClass) {
        expect(hasMainClass).toBe(true);
        console.log('已加载的类:', loadedClasses);
      } else {
        console.warn('未检测到核心 LaTeX 类，可能需要检查加载顺序');
      }
    });

    test('应该提供加载状态检查', () => {
      const checkLoadStatus = () => {
        const status = {
          coreLoaded: false,
          browserHelpersLoaded: false,
          readyForUse: false
        };

        // 检查核心类
        status.coreLoaded = !!(
          (typeof LaTeXParser !== 'undefined') ||
          (typeof window !== 'undefined' && window.LaTeXParser) ||
          (typeof global !== 'undefined' && global.LaTeXParser)
        );

        // 检查浏览器助手函数
        status.browserHelpersLoaded = !!(
          (typeof parseLatex !== 'undefined') ||
          (typeof createLaTeXParser !== 'undefined') ||
          (typeof window !== 'undefined' && (window.parseLatex || window.createLaTeXParser)) ||
          (typeof global !== 'undefined' && (global.parseLatex || global.createLaTeXParser))
        );

        status.readyForUse = status.coreLoaded && status.browserHelpersLoaded;

        return status;
      };

      const status = checkLoadStatus();
      console.log('LaTeX Parser V3 加载状态:', status);
      
      // 至少应该有某些组件可用
      expect(status.coreLoaded || status.browserHelpersLoaded).toBe(true);
    });
  });
});