/**
 * LaTeX Parser V3 基础接口测试
 * 测试 LaTeXParser 构造函数、parseComplete 方法基础功能、返回结果结构、配置参数等
 */

describe('LaTeX Parser V3 基础接口测试', () => {
  let LaTeXParser;

  beforeAll(() => {
    // 检查是否在 Node.js 环境中
    if (typeof require !== 'undefined') {
      // Node.js 环境
      const path = require('path');
      const fs = require('fs');
      
      // 检查核心文件是否存在
      const coreFiles = [
        '../core/DocumentParser.js',
        '../core/BlockElementParser.js', 
        '../core/InlineElementParser.js',
        '../core/MathParser.js',
        '../core/SpecialParser.js',
        '../core/LaTeXParser.js'
      ];
      
      for (const file of coreFiles) {
        const filePath = path.resolve(__dirname, file);
        if (!fs.existsSync(filePath)) {
          throw new Error(`核心文件不存在: ${file}`);
        }
      }
      
      // 加载 LaTeXParser
      LaTeXParser = require('../core/LaTeXParser.js');
    } else {
      // 浏览器环境 - 假设已通过 script 标签加载
      if (typeof window !== 'undefined' && window.LaTeXParser) {
        LaTeXParser = window.LaTeXParser;
      } else {
        throw new Error('浏览器环境中 LaTeXParser 未定义，请确保已加载所有必需的脚本文件');
      }
    }
  });

  describe('构造函数测试', () => {
    test('应该能够无参数创建解析器实例', () => {
      const parser = new LaTeXParser();
      expect(parser).toBeDefined();
      expect(typeof parser.parseComplete).toBe('function');
    });

    test('应该能够使用默认配置创建解析器', () => {
      const parser = new LaTeXParser();
      expect(parser.config).toBeDefined();
      expect(parser.config.mode).toBe('tolerant');
      expect(parser.config.enabledLayers).toEqual(['document', 'block', 'inline', 'math', 'special']);
      expect(parser.config.maxRecursionDepth).toBe(100);
    });

    test('应该能够使用自定义配置创建解析器', () => {
      const config = {
        mode: 'strict',
        enabledLayers: ['document', 'math'],
        maxRecursionDepth: 50
      };
      const parser = new LaTeXParser(config);
      
      expect(parser.config.mode).toBe('strict');
      expect(parser.config.enabledLayers).toEqual(['document', 'math']);
      expect(parser.config.maxRecursionDepth).toBe(50);
    });

    test('应该在传入无效配置时使用默认值', () => {
      const parser = new LaTeXParser({
        mode: 'invalid',
        enabledLayers: 'not-array',
        maxRecursionDepth: 'not-number'
      });
      
      expect(parser.config.mode).toBe('tolerant');
      expect(Array.isArray(parser.config.enabledLayers)).toBe(true);
      expect(typeof parser.config.maxRecursionDepth).toBe('number');
    });
  });

  describe('parseComplete 方法基础功能测试', () => {
    let parser;

    beforeEach(() => {
      parser = new LaTeXParser();
    });

    test('应该能够处理空字符串', () => {
      const result = parser.parseComplete('');
      
      expect(result).toBeDefined();
      expect(result.success).toBe(true);
      expect(typeof result.processedText).toBe('string');
      expect(result.processedText).toBe('');
    });

    test('应该能够处理纯文本', () => {
      const input = '这是普通文本';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('这是普通文本');
    });

    test('应该能够处理简单的行内数学公式', () => {
      const input = '$E = mc^2$';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/katex|math/i);
    });

    test('应该能够处理显示数学公式', () => {
      const input = '$$\\int_{0}^{\\infty} e^{-x} dx = 1$$';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toMatch(/katex|math/i);
    });

    test('应该能够处理章节命令', () => {
      const input = '\\section{测试章节}';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('测试章节');
    });

    test('应该能够处理文本格式化命令', () => {
      const input = '\\textbf{粗体文本} 和 \\textit{斜体文本}';
      const result = parser.parseComplete(input);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('<strong>粗体文本</strong>');
      expect(result.processedText).toContain('<em>斜体文本</em>');
    });
  });

  describe('返回结果结构验证', () => {
    let parser;

    beforeEach(() => {
      parser = new LaTeXParser();
    });

    test('应该返回正确的结果结构', () => {
      const result = parser.parseComplete('测试内容');
      
      expect(result).toHaveProperty('success');
      expect(result).toHaveProperty('processedText');
      expect(result).toHaveProperty('blocks');
      expect(result).toHaveProperty('metadata');
      expect(result).toHaveProperty('warnings');
      expect(result).toHaveProperty('errors');
      
      expect(typeof result.success).toBe('boolean');
      expect(typeof result.processedText).toBe('string');
      expect(Array.isArray(result.blocks)).toBe(true);
      expect(typeof result.metadata).toBe('object');
      expect(Array.isArray(result.warnings)).toBe(true);
      expect(Array.isArray(result.errors)).toBe(true);
    });

    test('成功解析时 success 应该为 true', () => {
      const result = parser.parseComplete('\\section{测试}');
      expect(result.success).toBe(true);
      expect(result.errors.length).toBe(0);
    });

    test('metadata 应该包含处理信息', () => {
      const result = parser.parseComplete('测试内容');
      
      expect(result.metadata).toHaveProperty('processingTime');
      expect(result.metadata).toHaveProperty('layerStats');
      expect(typeof result.metadata.processingTime).toBe('number');
      expect(typeof result.metadata.layerStats).toBe('object');
    });

    test('blocks 数组应该包含结构化数据', () => {
      const result = parser.parseComplete('\\section{章节} 内容 $math$');
      
      expect(Array.isArray(result.blocks)).toBe(true);
      // 应该有解析出的块级元素
      if (result.blocks.length > 0) {
        const block = result.blocks[0];
        expect(block).toHaveProperty('type');
        expect(typeof block.type).toBe('string');
      }
    });
  });

  describe('配置参数测试', () => {
    test('mode 配置应该影响错误处理', () => {
      const tolerantParser = new LaTeXParser({ mode: 'tolerant' });
      const strictParser = new LaTeXParser({ mode: 'strict' });
      
      const invalidInput = '\\unknowncommand{test}';
      
      const tolerantResult = tolerantParser.parseComplete(invalidInput);
      const strictResult = strictParser.parseComplete(invalidInput);
      
      // tolerant 模式应该继续处理并生成警告
      expect(tolerantResult.success).toBe(true);
      expect(tolerantResult.warnings.length).toBeGreaterThan(0);
      
      // strict 模式的行为取决于具体实现，但应该有不同的处理方式
      expect(strictResult).toBeDefined();
    });

    test('enabledLayers 配置应该控制解析层', () => {
      const mathOnlyParser = new LaTeXParser({ 
        enabledLayers: ['math'] 
      });
      
      const input = '\\section{标题} 文本 $math$ \\textbf{粗体}';
      const result = mathOnlyParser.parseComplete(input);
      
      expect(result.success).toBe(true);
      // 只启用 math 层，其他命令应该不被处理
      expect(result.processedText).toContain('$math$');
    });

    test('maxRecursionDepth 配置应该限制递归深度', () => {
      const shallowParser = new LaTeXParser({ 
        maxRecursionDepth: 5 
      });
      
      // 创建一个深度嵌套的结构
      const deepInput = '\\textbf{\\textit{\\underline{\\emph{\\texttt{deep}}}}}';
      const result = shallowParser.parseComplete(deepInput);
      
      // 应该有递归深度限制的保护
      expect(result).toBeDefined();
      expect(result.success).toBeDefined();
    });

    test('部分启用的层应该正常工作', () => {
      const docMathParser = new LaTeXParser({ 
        enabledLayers: ['document', 'math'] 
      });
      
      const input = `
        \\section{测试章节}
        这是文本 $E = mc^2$
        \\begin{itemize}
        \\item 项目1
        \\end{itemize}
      `;
      
      const result = docMathParser.parseComplete(input);
      
      expect(result.success).toBe(true);
      // 应该处理 section 和数学公式
      expect(result.processedText).toContain('测试章节');
      expect(result.processedText).toMatch(/katex|math/i);
    });
  });

  describe('性能测试', () => {
    let parser;

    beforeEach(() => {
      parser = new LaTeXParser();
    });

    test('处理时间应该在合理范围内', () => {
      const input = `
        \\section{性能测试}
        \\subsection{子章节}
        
        这是一段测试文本，包含数学公式 $a^2 + b^2 = c^2$
        
        \\begin{itemize}
        \\item 第一项
        \\item 第二项 $\\int_0^1 x dx$
        \\item 第三项
        \\end{itemize}
        
        $$\\begin{matrix}
        1 & 2 & 3 \\\\
        4 & 5 & 6 \\\\
        7 & 8 & 9
        \\end{matrix}$$
      `;
      
      const startTime = Date.now();
      const result = parser.parseComplete(input);
      const endTime = Date.now();
      
      const processingTime = endTime - startTime;
      
      expect(result.success).toBe(true);
      expect(processingTime).toBeLessThan(1000); // 应该在1秒内完成
      expect(result.metadata.processingTime).toBeDefined();
    });

    test('空输入应该快速处理', () => {
      const startTime = Date.now();
      const result = parser.parseComplete('');
      const endTime = Date.now();
      
      const processingTime = endTime - startTime;
      
      expect(result.success).toBe(true);
      expect(processingTime).toBeLessThan(100); // 空输入应该非常快
    });
  });

  describe('边界条件测试', () => {
    let parser;

    beforeEach(() => {
      parser = new LaTeXParser();
    });

    test('应该处理 null 和 undefined 输入', () => {
      expect(() => parser.parseComplete(null)).not.toThrow();
      expect(() => parser.parseComplete(undefined)).not.toThrow();
      
      const nullResult = parser.parseComplete(null);
      const undefinedResult = parser.parseComplete(undefined);
      
      expect(nullResult.success).toBeDefined();
      expect(undefinedResult.success).toBeDefined();
    });

    test('应该处理非常长的输入', () => {
      const longInput = 'a'.repeat(10000) + ' $x^2$ ' + 'b'.repeat(10000);
      const result = parser.parseComplete(longInput);
      
      expect(result.success).toBe(true);
      expect(result.processedText.length).toBeGreaterThan(0);
    });

    test('应该处理包含特殊字符的输入', () => {
      const specialInput = '特殊字符: &<>"\'\\n\\t\\r 数学: $α + β = γ$';
      const result = parser.parseComplete(specialInput);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('特殊字符');
    });

    test('应该处理多行输入', () => {
      const multilineInput = `第一行
      第二行
      第三行 $math$
      第四行`;
      
      const result = parser.parseComplete(multilineInput);
      
      expect(result.success).toBe(true);
      expect(result.processedText).toContain('第一行');
      expect(result.processedText).toContain('第四行');
    });
  });

  describe('接口向后兼容性测试', () => {
    test('parseComplete 方法应该始终存在', () => {
      const parser = new LaTeXParser();
      expect(typeof parser.parseComplete).toBe('function');
    });

    test('返回结果应该包含必需的字段', () => {
      const parser = new LaTeXParser();
      const result = parser.parseComplete('test');
      
      // 这些字段是接口承诺的，必须存在
      const requiredFields = ['success', 'processedText', 'blocks', 'metadata', 'warnings', 'errors'];
      requiredFields.forEach(field => {
        expect(result).toHaveProperty(field);
      });
    });

    test('config 对象应该有默认值', () => {
      const parser = new LaTeXParser();
      
      expect(parser.config).toBeDefined();
      expect(parser.config.mode).toBeDefined();
      expect(parser.config.enabledLayers).toBeDefined();
      expect(parser.config.maxRecursionDepth).toBeDefined();
    });
  });
});