# LaTeX Parser V3 接口测试

本目录包含 LaTeX Parser V3 的完整对外接口测试套件，基于 interface.md 中定义的规范实现。

## 测试文件结构

### 核心测试文件

1. **interface-basic.spec.js** - 基础接口测试
   - LaTeXParser 构造函数测试
   - parseComplete 方法基础功能
   - 返回结果结构验证
   - 配置参数测试（mode, enabledLayers, maxRecursionDepth）

2. **interface-math.spec.js** - 数学公式测试
   - 行内公式 `$...$` 和 `\(...\)`
   - 显示公式 `$$...$$` 和 `\[...\]`
   - 数学环境（equation, align, matrix, cases等）
   - 数学命令和符号
   - 复杂数学表达式

3. **interface-document.spec.js** - 文档结构测试
   - 章节命令（section, subsection等）
   - 文本格式化（textbf, textit, emph等）
   - 列表环境（itemize, enumerate, description）
   - 表格环境（tabular, array）
   - 特殊环境和命令

4. **interface-browser.spec.js** - 浏览器环境测试
   - browser.js 提供的全局函数
   - parseLatex 便捷函数
   - createLaTeXParser 工厂函数
   - 浏览器兼容性测试

5. **interface-error.spec.js** - 错误处理测试
   - tolerant 模式错误恢复
   - strict 模式错误处理
   - 警告信息生成
   - 未知命令处理

6. **interface-performance.spec.js** - 性能和兼容性测试
   - 处理时间验证
   - 大文档处理测试
   - 内存使用合理性测试
   - 并发处理测试

### 测试工具

- **test-runner.js** - 测试运行器（Node.js 环境）
- **README.md** - 本说明文档

## 运行测试

### 使用 Jest（推荐）

如果已安装 Jest：

```bash
# 运行所有测试
npx jest tests/

# 运行特定测试文件
npx jest tests/interface-basic.spec.js

# 运行特定测试套件
npx jest tests/interface-math.spec.js

# 生成覆盖率报告
npx jest tests/ --coverage
```

### 使用内置测试运行器

如果未安装 Jest，可以使用内置的测试运行器：

```bash
# 运行所有测试
node tests/test-runner.js

# 查看帮助
node tests/test-runner.js --help

# 列出所有测试文件
node tests/test-runner.js --list

# 运行特定测试文件
node tests/test-runner.js --file interface-basic

# 运行性能测试
node tests/test-runner.js --file performance
```

## 浏览器环境测试

对于浏览器环境测试，需要确保按正确顺序加载文件：

```html
<!DOCTYPE html>
<html>
<head>
  <title>LaTeX Parser V3 浏览器测试</title>
</head>
<body>
  <!-- 按顺序加载核心文件 -->
  <script src="../core/DocumentParser.js"></script>
  <script src="../core/BlockElementParser.js"></script>
  <script src="../core/InlineElementParser.js"></script>
  <script src="../core/MathParser.js"></script>
  <script src="../core/SpecialParser.js"></script>
  <script src="../core/LaTeXParser.js"></script>
  <script src="../browser.js"></script>
  
  <!-- 加载测试框架 -->
  <script src="https://unpkg.com/jest@27/dist/jest.min.js"></script>
  
  <!-- 加载测试文件 -->
  <script src="interface-browser.spec.js"></script>
  
  <script>
    // 运行测试
    jest.run();
  </script>
</body>
</html>
```

## 测试覆盖范围

### 基础功能测试 (interface-basic.spec.js)
- ✅ 构造函数参数验证
- ✅ 默认配置测试
- ✅ 自定义配置测试
- ✅ parseComplete 方法测试
- ✅ 返回结果结构验证
- ✅ 错误处理基础测试
- ✅ 性能基准测试

### 数学公式测试 (interface-math.spec.js)
- ✅ 行内公式 `$...$`, `\(...\)`
- ✅ 显示公式 `$$...$$`, `\[...\]`
- ✅ equation, align, gather 环境
- ✅ matrix, pmatrix, bmatrix 环境
- ✅ cases, split 环境
- ✅ 数学操作符和符号
- ✅ 复杂嵌套表达式
- ✅ 数学错误处理

### 文档结构测试 (interface-document.spec.js)
- ✅ 章节结构 (section, subsection, chapter)
- ✅ 文本格式化 (textbf, textit, emph, underline)
- ✅ 列表环境 (itemize, enumerate, description)
- ✅ 表格环境 (tabular, array)
- ✅ 特殊环境 (center, quote, verbatim)
- ✅ document 环境
- ✅ 复杂文档结构
- ✅ 嵌套和组合测试

### 浏览器环境测试 (interface-browser.spec.js)
- ✅ 全局对象检测
- ✅ parseLatex 便捷函数
- ✅ createLaTeXParser 工厂函数
- ✅ 浏览器集成测试
- ✅ 兼容性测试
- ✅ 加载顺序检测
- ✅ 错误处理测试

### 错误处理测试 (interface-error.spec.js)
- ✅ tolerant 模式错误恢复
- ✅ strict 模式错误处理
- ✅ 未知命令处理
- ✅ 环境不匹配处理
- ✅ 警告信息生成
- ✅ 错误信息质量测试
- ✅ 错误恢复策略测试

### 性能测试 (interface-performance.spec.js)
- ✅ 基础性能指标 (< 100ms 小文档)
- ✅ 大文档处理 (< 1000ms 中型文档)
- ✅ 内存使用测试
- ✅ 并发处理测试
- ✅ 兼容性测试
- ✅ 资源清理测试

## 性能基准

### 预期性能指标
- 小型文档 (< 100 行): < 100ms
- 中型文档 (1000 行): < 1000ms  
- 大型文档 (5000+ 行): < 5000ms
- 内存使用: < 50MB (标准文档)
- 测试覆盖率: > 90%

### 实际测试结果
运行 `node tests/test-runner.js` 查看当前环境的性能测试结果。

## 故障排除

### 常见问题

**Q: 测试运行时报错 "LaTeXParser is not defined"**  
A: 检查核心文件是否存在，确保按正确顺序加载。

**Q: 数学公式测试失败**  
A: 确认 MathParser.js 文件存在且可正常加载。

**Q: 浏览器环境测试无法运行**  
A: 确保所有核心文件按正确顺序加载，browser.js 需要最后加载。

**Q: 性能测试超时**  
A: 检查系统资源，考虑调整性能基准值。

**Q: 某些测试被跳过**  
A: 检查相关核心文件是否存在，缺失文件会导致相关测试跳过。

### 调试测试

启用详细输出：

```bash
# 使用详细模式运行
DEBUG=1 node tests/test-runner.js

# 只运行失败的测试
node tests/test-runner.js --file error
```

### 检查测试环境

```bash
# 检查文件依赖
node -e "
const path = require('path');
const fs = require('fs');
const coreFiles = ['DocumentParser.js', 'BlockElementParser.js', 'InlineElementParser.js', 'MathParser.js', 'SpecialParser.js', 'LaTeXParser.js'];
coreFiles.forEach(file => {
  const exists = fs.existsSync(path.join(__dirname, '../core', file));
  console.log(\`\${file}: \${exists ? '✅' : '❌'}\`);
});
"
```

## 贡献测试

### 添加新测试

1. 选择合适的测试文件或创建新文件
2. 遵循现有测试模式
3. 包含正面和负面测试用例
4. 添加性能验证
5. 更新 README.md

### 测试模式

```javascript
describe('功能模块名称', () => {
  let parser;

  beforeEach(() => {
    parser = new LaTeXParser();
  });

  test('应该处理正常输入', () => {
    const result = parser.parseComplete('测试输入');
    expect(result.success).toBe(true);
    expect(result.processedText).toContain('期望内容');
  });

  test('应该处理错误输入', () => {
    const result = parser.parseComplete('错误输入');
    expect(result.warnings.length).toBeGreaterThan(0);
  });
});
```

## 测试报告

运行完整测试后，检查以下指标：

- [ ] 所有基础接口测试通过
- [ ] 数学公式处理正确
- [ ] 文档结构解析正常
- [ ] 浏览器环境兼容
- [ ] 错误处理健壮
- [ ] 性能满足基准
- [ ] 无内存泄漏
- [ ] 兼容性良好

## 许可证

MIT License - 与主项目保持一致