/**
 * 矩阵环境修复测试脚本
 * 直接运行，不依赖测试框架
 */

const MathParser = require('./core/MathParser');

function runTest(testName, testFn) {
  console.log(`\n=== ${testName} ===`);
  try {
    testFn();
    console.log('✅ 通过');
  } catch (error) {
    console.log('❌ 失败:', error.message);
  }
}

function assertEqual(actual, expected, message = '') {
  if (actual !== expected) {
    throw new Error(`${message}\n  期望: ${expected}\n  实际: ${actual}`);
  }
}

function assertContains(text, substring, message = '') {
  if (!text.includes(substring)) {
    throw new Error(`${message}\n  期望包含: ${substring}\n  实际文本: ${text}`);
  }
}

// 初始化解析器
const parser = new MathParser();

console.log('开始测试 MathParser 矩阵环境修复...');

// 测试案例
runTest('为独立矩阵环境添加$$包装', () => {
  const text = '这是一个独立的矩阵：\\begin{matrix} a & b \\\\ c & d \\end{matrix} 结束';
  const result = parser.wrapRemainingMathEnvironments(text);
  const expected = '这是一个独立的矩阵：$$\\begin{matrix} a & b \\\\ c & d \\end{matrix}$$ 结束';
  assertEqual(result, expected);
});

runTest('不为行内公式中的矩阵添加$$包装', () => {
  const text = '公式：$\\begin{matrix} a & b \\\\ c & d \\end{matrix} \\times \\begin{pmatrix} 1 & 2 \\\\ 3 & 4 \\end{pmatrix}$ 结果';
  const result = parser.wrapRemainingMathEnvironments(text);
  // 应该保持原样
  assertEqual(result, text);
});

runTest('不为显示数学中的矩阵添加$$包装', () => {
  const testCases = [
    { text: '\\[\\begin{matrix} a & b \\\\ c & d \\end{matrix}\\]', expected: '\\[\\begin{matrix} a & b \\\\ c & d \\end{matrix}\\]' },
    { text: '$$\\begin{pmatrix} 1 & 2 \\\\ 3 & 4 \\end{pmatrix}$$', expected: '$$\\begin{pmatrix} 1 & 2 \\\\ 3 & 4 \\end{pmatrix}$$' }
  ];

  testCases.forEach(({ text, expected }) => {
    const result = parser.wrapRemainingMathEnvironments(text);
    assertEqual(result, expected);
  });
});

runTest('正确处理混合情况', () => {
  const text = '独立矩阵：\\begin{matrix} a & b \\\\ c & d \\end{matrix} 和行内矩阵：$\\begin{pmatrix} 1 & 2 \\\\ 3 & 4 \\end{pmatrix}$';
  const result = parser.wrapRemainingMathEnvironments(text);
  const expected = '独立矩阵：$$\\begin{matrix} a & b \\\\ c & d \\end{matrix}$$ 和行内矩阵：$\\begin{pmatrix} 1 & 2 \\\\ 3 & 4 \\end{pmatrix}$';
  assertEqual(result, expected);
});

runTest('处理行内公式中的多个矩阵而不分割', () => {
  const text = '$\\begin{matrix} a \\\\ b \\end{matrix} + \\begin{pmatrix} c \\\\ d \\end{pmatrix} = \\begin{bmatrix} a+c \\\\ b+d \\end{bmatrix}$';
  const result = parser.wrapRemainingMathEnvironments(text);
  // 整个行内公式应该保持完整
  assertEqual(result, text);
});

runTest('处理各种类型的独立矩阵', () => {
  const testCases = [
    'matrix',
    'pmatrix',
    'bmatrix',
    'vmatrix',
    'Vmatrix'
  ];

  testCases.forEach(env => {
    const text = `\\begin{${env}} a & b \\\\ c & d \\end{${env}}`;
    const result = parser.wrapRemainingMathEnvironments(text);
    const expected = `$$${text}$$`;
    assertEqual(result, expected, `${env}类型矩阵应该被包装`);
  });
});

runTest('处理边界情况', () => {
  // 空字符串
  assertEqual(parser.wrapRemainingMathEnvironments(''), '');

  // 纯文本
  assertEqual(parser.wrapRemainingMathEnvironments('纯文本，没有矩阵'), '纯文本，没有矩阵');

  // 矩阵在开头
  const text1 = '\\begin{matrix} a & b \\end{matrix} 开头';
  const result1 = parser.wrapRemainingMathEnvironments(text1);
  assertEqual(result1, '$$\\begin{matrix} a & b \\end{matrix}$$ 开头');

  // 矩阵在末尾
  const text2 = '末尾 \\begin{pmatrix} 1 & 2 \\end{pmatrix}';
  const result2 = parser.wrapRemainingMathEnvironments(text2);
  assertEqual(result2, '末尾 $$\\begin{pmatrix} 1 & 2 \\end{pmatrix}$$');
});

runTest('完整处理流程测试', () => {
  const text = `
    独立矩阵：\\begin{matrix} 1 & 2 \\\\ 3 & 4 \\end{matrix}
    行内公式：$\\begin{pmatrix} x \\\\ y \\end{pmatrix} = \\begin{bmatrix} a & b \\\\ c & d \\end{bmatrix} \\begin{pmatrix} u \\\\ v \\end{pmatrix}$
  `;

  const result = parser.process(text);

  // 检查独立矩阵被包装
  assertContains(result.text, '$$\\begin{matrix} 1 & 2 \\\\ 3 & 4 \\end{matrix}$$', '独立矩阵应该被包装');

  // 检查行内公式保持完整
  assertContains(result.text, '$\\begin{pmatrix}', '行内公式应该保持完整');
  assertContains(result.text, '\\end{pmatrix}$', '行内公式应该保持完整');

  // 检查没有错误
  assertEqual(result.errors.length, 0, '不应该有错误');
});

console.log('\n=== 测试完成 ===');
console.log('所有测试都通过了！矩阵环境修复成功。');

// 显示一些实际的输出示例
console.log('\n=== 实际输出示例 ===');
console.log('\n1. 独立矩阵:');
const example1 = '\\begin{matrix} a & b \\\\ c & d \\end{matrix}';
console.log('输入:', example1);
console.log('输出:', parser.wrapRemainingMathEnvironments(example1));

console.log('\n2. 行内公式中的多个矩阵:');
const example2 = '$\\begin{matrix} a \\\\ b \\end{matrix} \\times \\begin{pmatrix} c \\\\ d \\end{pmatrix}$';
console.log('输入:', example2);
console.log('输出:', parser.wrapRemainingMathEnvironments(example2));

console.log('\n3. 混合情况:');
const example3 = '独立: \\begin{bmatrix} 1 & 2 \\\\ 3 & 4 \\end{bmatrix} 行内: $\\begin{vmatrix} a & b \\\\ c & d \\end{vmatrix}$';
console.log('输入:', example3);
console.log('输出:', parser.wrapRemainingMathEnvironments(example3));