/**
 * Renderers 模块测试配置
 * 定义测试环境的通用配置和工具函数
 */

// 通用测试配置
const TEST_CONFIG = {
  // 性能测试阈值
  performance: {
    maxRenderTime: 1000,      // 最大渲染时间（毫秒）
    maxGenerateTime: 2000,    // 最大CSS生成时间（毫秒）
    largeBatchSize: 100       // 大批量测试的数据量
  },
  
  // 输出验证配置
  output: {
    minHtmlLength: 10,        // HTML最小长度
    minCssLength: 50,         // CSS最小长度
    maxOutputSize: 1024000    // 最大输出大小（1MB）
  },
  
  // 覆盖率目标
  coverage: {
    statements: 90,
    branches: 85,
    functions: 100,
    lines: 90
  }
};

// 通用Mock数据生成器
const MockDataFactory = {
  /**
   * 创建标准的ParseResult对象
   */
  createParseResult(overrides = {}) {
    return {
      success: true,
      processedText: 'Sample processed text',
      blocks: [],
      metadata: { title: 'Test Document', author: 'Test Author' },
      warnings: [],
      errors: [],
      ...overrides
    };
  },

  /**
   * 创建HTML映射配置
   */
  createHtmlMapping() {
    return {
      'title': { tag: 'h1', className: 'document-title' },
      'author': { tag: 'div', className: 'document-author' },
      'date': { tag: 'div', className: 'document-date' },
      'maketitle': { tag: 'div', className: 'title-separator' },
      'section': { tag: 'h2', className: 'latex-section' },
      'subsection': { tag: 'h3', className: 'latex-subsection' },
      'subsubsection': { tag: 'h4', className: 'latex-subsubsection' },
      'paragraph': { tag: 'p', className: 'latex-content' },
      'documentclass': { ignore: true },
      'usepackage': { ignore: true }
    };
  },

  /**
   * 创建CSS规则配置
   */
  createCssRules() {
    return {
      'latex-document': {
        maxWidth: '8.5in',
        margin: '0 auto',
        padding: '1in',
        fontFamily: '"Times New Roman", serif',
        fontSize: '12pt',
        lineHeight: '1.6',
        color: '#333',
        background: 'white'
      },
      'document-title': {
        textAlign: 'center',
        fontSize: '18pt',
        fontWeight: 'bold',
        margin: '1.5em 0 1em 0',
        color: '#000'
      },
      'document-author': {
        textAlign: 'center',
        fontSize: '14pt',
        margin: '0.5em 0',
        color: '#333'
      },
      'document-date': {
        textAlign: 'center',
        fontSize: '12pt',
        margin: '0.5em 0 2em 0',
        color: '#666'
      },
      'title-separator': {
        borderTop: '1px solid #ccc',
        margin: '2em 0',
        height: '0'
      },
      'latex-section': {
        fontSize: '16pt',
        fontWeight: 'bold',
        margin: '2em 0 1em 0',
        color: '#000'
      },
      'latex-subsection': {
        fontSize: '14pt',
        fontWeight: 'bold',
        margin: '1.5em 0 0.8em 0',
        color: '#000'
      },
      'latex-subsubsection': {
        fontSize: '13pt',
        fontWeight: 'bold',
        margin: '1.2em 0 0.6em 0',
        color: '#000'
      },
      'latex-content': {
        margin: '0.8em 0',
        textAlign: 'justify',
        textIndent: '2em'
      }
    };
  },

  /**
   * 创建CSS工具
   */
  createCssUtils() {
    return {
      objectToCss: (className, styleObject) => {
        const cssProperties = Object.entries(styleObject)
          .map(([property, value]) => {
            // 将驼峰命名转换为短横线命名
            const cssProperty = property.replace(/[A-Z]/g, match => `-${match.toLowerCase()}`);
            return `  ${cssProperty}: ${value};`;
          })
          .join('\n');
        
        return `.${className} {\n${cssProperties}\n}`;
      },
      
      minifyCSS: (css) => {
        return css
          .replace(/\/\*[\s\S]*?\*\//g, '') // 移除注释
          .replace(/\s+/g, ' ')             // 压缩空白
          .replace(/;\s*}/g, '}')           // 移除最后一个分号
          .replace(/\s*{\s*/g, '{')         // 压缩花括号
          .replace(/\s*}\s*/g, '}')
          .replace(/\s*;\s*/g, ';')         // 压缩分号
          .trim();
      }
    };
  },

  /**
   * 创建测试用的数据块
   */
  createTestBlocks(types = ['title', 'author', 'section', 'paragraph']) {
    const blocks = [];
    
    types.forEach((type, index) => {
      switch (type) {
        case 'title':
          blocks.push({
            type: 'title',
            title: `Test Title ${index + 1}`,
            content: `Test Title ${index + 1}`
          });
          break;
        case 'author':
          blocks.push({
            type: 'author',
            author: `Test Author ${index + 1}`,
            content: `Test Author ${index + 1}`
          });
          break;
        case 'section':
          blocks.push({
            type: 'section',
            title: `Section ${index + 1}`,
            content: `Section ${index + 1}`
          });
          break;
        case 'subsection':
          blocks.push({
            type: 'subsection',
            title: `Subsection ${index + 1}.1`,
            content: `Subsection ${index + 1}.1`
          });
          break;
        case 'paragraph':
          blocks.push({
            type: 'paragraph',
            content: `This is test paragraph ${index + 1} with some sample content.`
          });
          break;
        case 'maketitle':
          blocks.push({
            type: 'maketitle'
          });
          break;
        default:
          blocks.push({
            type: type,
            content: `Test content for ${type} ${index + 1}`
          });
      }
    });
    
    return blocks;
  },

  /**
   * 创建大批量测试数据
   */
  createLargeBatch(size = 100) {
    const blocks = [];
    const types = ['section', 'paragraph', 'subsection', 'paragraph', 'paragraph'];
    
    for (let i = 0; i < size; i++) {
      const type = types[i % types.length];
      blocks.push(...this.createTestBlocks([type]));
    }
    
    return blocks;
  },

  /**
   * 创建复杂的嵌套文档结构
   */
  createNestedDocument() {
    return [
      { type: 'title', title: 'Complex Document', content: 'Complex Document' },
      { type: 'author', author: 'Test Authors', content: 'Test Authors' },
      { type: 'maketitle' },
      {
        type: 'section',
        title: '1. Introduction',
        content: '1. Introduction',
        children: [
          { type: 'paragraph', content: 'Introduction paragraph 1' },
          {
            type: 'subsection',
            title: '1.1 Background',
            content: '1.1 Background',
            children: [
              { type: 'paragraph', content: 'Background paragraph 1' },
              { type: 'paragraph', content: 'Background paragraph 2' }
            ]
          }
        ]
      },
      {
        type: 'section',
        title: '2. Methodology',
        content: '2. Methodology',
        children: [
          { type: 'paragraph', content: 'Methodology introduction' }
        ]
      }
    ];
  }
};

// 测试工具函数
const TestUtils = {
  /**
   * 测量函数执行时间
   */
  measureTime(fn) {
    const start = Date.now();
    const result = fn();
    const end = Date.now();
    return { result, time: end - start };
  },

  /**
   * 异步函数执行时间测量
   */
  async measureTimeAsync(fn) {
    const start = Date.now();
    const result = await fn();
    const end = Date.now();
    return { result, time: end - start };
  },

  /**
   * 验证HTML有效性
   */
  validateHTML(html) {
    const validation = {
      valid: true,
      errors: []
    };

    // 检查基本标签配对
    const openTags = (html.match(/<[^/!][^>]*[^/]>/g) || []).length;
    const closeTags = (html.match(/<\/[^>]+>/g) || []).length;
    const selfClosingTags = (html.match(/<[^>]+\/>/g) || []).length;

    if (openTags - selfClosingTags !== closeTags) {
      validation.valid = false;
      validation.errors.push('HTML标签配对不匹配');
    }

    // 检查是否包含必要的结构
    if (!html.includes('class=')) {
      validation.valid = false;
      validation.errors.push('HTML缺少CSS类定义');
    }

    return validation;
  },

  /**
   * 验证CSS有效性
   */
  validateCSS(css) {
    const validation = {
      valid: true,
      errors: []
    };

    // 检查括号配对
    const openBraces = (css.match(/\{/g) || []).length;
    const closeBraces = (css.match(/\}/g) || []).length;

    if (openBraces !== closeBraces) {
      validation.valid = false;
      validation.errors.push('CSS括号配对不匹配');
    }

    // 检查基本CSS格式
    if (!css.match(/\.[a-zA-Z-]+ \{/)) {
      validation.valid = false;
      validation.errors.push('CSS缺少有效的选择器');
    }

    return validation;
  },

  /**
   * 提取CSS类名
   */
  extractCSSClasses(css) {
    const matches = css.match(/\.([\w-]+)\s*\{/g) || [];
    return matches.map(match => match.replace(/^\./, '').replace(/\s*\{$/, ''));
  },

  /**
   * 提取HTML中使用的CSS类
   */
  extractHTMLClasses(html) {
    const matches = html.match(/class="([^"]+)"/g) || [];
    const classes = [];
    matches.forEach(match => {
      const className = match.match(/class="([^"]+)"/)[1];
      if (!classes.includes(className)) {
        classes.push(className);
      }
    });
    return classes;
  },

  /**
   * 比较两个数组的差异
   */
  arrayDiff(arr1, arr2) {
    return {
      inFirstOnly: arr1.filter(item => !arr2.includes(item)),
      inSecondOnly: arr2.filter(item => !arr1.includes(item)),
      common: arr1.filter(item => arr2.includes(item))
    };
  },

  /**
   * 生成随机测试数据
   */
  generateRandomString(length = 10) {
    const chars = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
    return Array.from({ length }, () => chars.charAt(Math.floor(Math.random() * chars.length))).join('');
  },

  /**
   * 深度克隆对象
   */
  deepClone(obj) {
    return JSON.parse(JSON.stringify(obj));
  }
};

// 环境检测
const Environment = {
  isNode: typeof require !== 'undefined' && typeof module !== 'undefined',
  isBrowser: typeof window !== 'undefined',
  
  getEnvironmentInfo() {
    if (this.isNode) {
      return {
        type: 'Node.js',
        version: process.version,
        platform: process.platform
      };
    } else if (this.isBrowser) {
      return {
        type: 'Browser',
        userAgent: navigator.userAgent,
        vendor: navigator.vendor
      };
    }
    return { type: 'Unknown' };
  }
};

// 导出配置和工具
if (typeof module !== 'undefined' && module.exports) {
  module.exports = {
    TEST_CONFIG,
    MockDataFactory,
    TestUtils,
    Environment
  };
} else if (typeof window !== 'undefined') {
  window.TEST_CONFIG = TEST_CONFIG;
  window.MockDataFactory = MockDataFactory;
  window.TestUtils = TestUtils;
  window.Environment = Environment;
}