#!/usr/bin/env node

/**
 * Renderers 模块测试运行器
 * 用于在Node.js环境中快速运行测试
 * 
 * 使用方法：
 * node run-tests.js [test-file]
 * 
 * 示例：
 * node run-tests.js                    # 运行所有测试
 * node run-tests.js HTMLRenderer       # 只运行HTMLRenderer测试
 * node run-tests.js CSSGenerator       # 只运行CSSGenerator测试
 * node run-tests.js integration        # 只运行集成测试
 */

const fs = require('fs');
const path = require('path');

// 简单的测试框架实现
class SimpleTestFramework {
  constructor() {
    this.tests = [];
    this.describes = [];
    this.currentDescribe = null;
    this.results = {
      passed: 0,
      failed: 0,
      errors: []
    };
  }

  describe(description, fn) {
    const oldDescribe = this.currentDescribe;
    this.currentDescribe = description;
    this.describes.push(description);
    
    console.log(`\n📋 ${description}`);
    
    try {
      fn();
    } catch (error) {
      console.error(`❌ 描述块执行失败: ${description}`);
      console.error(error);
      this.results.errors.push({ description, error: error.message });
    }
    
    this.currentDescribe = oldDescribe;
  }

  test(description, fn) {
    const fullDescription = this.currentDescribe 
      ? `${this.currentDescribe} > ${description}` 
      : description;
    
    try {
      fn();
      console.log(`  ✅ ${description}`);
      this.results.passed++;
    } catch (error) {
      console.log(`  ❌ ${description}`);
      console.log(`     错误: ${error.message}`);
      this.results.failed++;
      this.results.errors.push({ description: fullDescription, error: error.message });
    }
  }

  expect(actual) {
    return {
      toBe: (expected) => {
        if (actual !== expected) {
          throw new Error(`期望 ${expected}, 但得到 ${actual}`);
        }
      },
      toBeDefined: () => {
        if (actual === undefined) {
          throw new Error(`期望值已定义, 但得到 undefined`);
        }
      },
      toEqual: (expected) => {
        if (JSON.stringify(actual) !== JSON.stringify(expected)) {
          throw new Error(`期望 ${JSON.stringify(expected)}, 但得到 ${JSON.stringify(actual)}`);
        }
      },
      toContain: (expected) => {
        if (!actual || !actual.toString().includes(expected)) {
          throw new Error(`期望包含 "${expected}", 但在 "${actual}" 中未找到`);
        }
      },
      not: {
        toContain: (expected) => {
          if (actual && actual.toString().includes(expected)) {
            throw new Error(`期望不包含 "${expected}", 但在 "${actual}" 中找到了`);
          }
        }
      },
      toBeGreaterThan: (expected) => {
        if (!(actual > expected)) {
          throw new Error(`期望 ${actual} 大于 ${expected}`);
        }
      },
      toBeLessThan: (expected) => {
        if (!(actual < expected)) {
          throw new Error(`期望 ${actual} 小于 ${expected}`);
        }
      },
      toHaveProperty: (prop) => {
        if (!actual || !actual.hasOwnProperty(prop)) {
          throw new Error(`期望对象有属性 "${prop}"`);
        }
      },
      toMatch: (regex) => {
        if (!actual || !regex.test(actual)) {
          throw new Error(`期望 "${actual}" 匹配正则表达式 ${regex}`);
        }
      }
    };
  }

  beforeEach(fn) {
    // 简化实现，在每个test之前调用
    this._beforeEach = fn;
  }

  beforeAll(fn) {
    // 简化实现，在describe开始时调用
    this._beforeAll = fn;
  }

  runBeforeEach() {
    if (this._beforeEach) {
      this._beforeEach();
    }
  }

  runBeforeAll() {
    if (this._beforeAll) {
      this._beforeAll();
    }
  }

  printResults() {
    console.log('\n' + '='.repeat(50));
    console.log('📊 测试结果汇总');
    console.log('='.repeat(50));
    console.log(`✅ 通过: ${this.results.passed}`);
    console.log(`❌ 失败: ${this.results.failed}`);
    console.log(`📝 总计: ${this.results.passed + this.results.failed}`);
    
    if (this.results.errors.length > 0) {
      console.log('\n❌ 失败的测试详情:');
      this.results.errors.forEach((error, index) => {
        console.log(`\n${index + 1}. ${error.description}`);
        console.log(`   ${error.error}`);
      });
    }
    
    const successRate = this.results.passed / (this.results.passed + this.results.failed) * 100;
    console.log(`\n📈 成功率: ${successRate.toFixed(1)}%`);
    
    if (this.results.failed === 0) {
      console.log('\n🎉 所有测试都通过了！');
    } else {
      console.log(`\n⚠️  有 ${this.results.failed} 个测试失败`);
    }
  }
}

// 创建全局测试框架实例
const framework = new SimpleTestFramework();

// 将测试方法设置为全局
global.describe = framework.describe.bind(framework);
global.test = framework.test.bind(framework);
global.expect = framework.expect.bind(framework);
global.beforeEach = framework.beforeEach.bind(framework);
global.beforeAll = framework.beforeAll.bind(framework);

// 模拟jest对象
global.jest = {
  fn: () => ({
    // 简单的mock函数实现
    mockReturnValue: function(value) {
      this._returnValue = value;
      return this;
    },
    mockImplementation: function(fn) {
      this._implementation = fn;
      return this;
    }
  })
};

// 主函数
async function main() {
  const args = process.argv.slice(2);
  const testFilter = args[0];
  
  console.log('🚀 启动 Renderers 模块测试');
  console.log('='.repeat(50));
  
  // 获取测试文件列表
  const testFiles = [];
  const testsDir = __dirname;
  
  if (testFilter) {
    // 运行特定测试
    const testFile = `${testFilter}.test.js`;
    if (fs.existsSync(path.join(testsDir, testFile))) {
      testFiles.push(testFile);
      console.log(`🎯 运行特定测试: ${testFile}`);
    } else {
      console.error(`❌ 测试文件不存在: ${testFile}`);
      process.exit(1);
    }
  } else {
    // 运行所有测试
    const allFiles = fs.readdirSync(testsDir);
    testFiles.push(...allFiles.filter(file => file.endsWith('.test.js')));
    console.log('📚 运行所有测试文件');
  }
  
  if (testFiles.length === 0) {
    console.log('⚠️  没有找到测试文件');
    return;
  }
  
  console.log(`📄 找到 ${testFiles.length} 个测试文件:`);
  testFiles.forEach(file => console.log(`   - ${file}`));
  
  // 运行测试文件
  for (const testFile of testFiles) {
    const testPath = path.join(testsDir, testFile);
    console.log(`\n📋 执行测试文件: ${testFile}`);
    console.log('-'.repeat(30));
    
    try {
      // 重置模块缓存，确保每次都是全新加载
      delete require.cache[testPath];
      
      // 加载并执行测试文件
      require(testPath);
      
    } catch (error) {
      console.error(`❌ 加载测试文件失败: ${testFile}`);
      console.error(error);
      framework.results.errors.push({
        description: `加载文件 ${testFile}`,
        error: error.message
      });
      framework.results.failed++;
    }
  }
  
  // 打印测试结果
  framework.printResults();
  
  // 设置退出代码
  process.exit(framework.results.failed > 0 ? 1 : 0);
}

// 运行主函数
if (require.main === module) {
  main().catch(error => {
    console.error('❌ 测试运行器执行失败:', error);
    process.exit(1);
  });
}

module.exports = { SimpleTestFramework };