/**
 * Renderers 模块集成测试
 * 测试 HTMLRenderer 和 CSSGenerator 的协同工作
 * 基于interface.md规范验证完整的渲染流程
 */

// 创建完整的测试数据
const createCompleteParseResult = () => ({
  success: true,
  processedText: 'Complete LaTeX document with multiple sections and content.',
  blocks: [
    { type: 'documentclass', content: '\\documentclass{article}' },
    { type: 'title', title: 'Advanced LaTeX Document', content: 'Advanced LaTeX Document' },
    { type: 'author', author: 'John Doe and Jane Smith', content: 'John Doe and Jane Smith' },
    { type: 'maketitle' },
    { type: 'section', title: '1 Introduction', content: '1 Introduction' },
    { type: 'paragraph', content: 'This document demonstrates the comprehensive LaTeX parsing capabilities.' },
    { type: 'subsection', title: '1.1 Background', content: '1.1 Background' },
    { type: 'paragraph', content: 'LaTeX is a high-quality typesetting system widely used in academia.' },
    { type: 'section', title: '2 Methodology', content: '2 Methodology' },
    { type: 'paragraph', content: 'Our approach involves a multi-layered parsing architecture.' },
    { type: 'subsubsection', title: '2.1.1 Parser Design', content: '2.1.1 Parser Design' },
    { type: 'paragraph', content: 'The parser uses a five-layer architecture for optimal performance.' }
  ],
  metadata: {
    title: 'Advanced LaTeX Document',
    author: 'John Doe and Jane Smith',
    documentClass: 'article',
    packages: ['amsmath', 'graphicx']
  },
  warnings: [
    'Some advanced features may not be fully supported'
  ],
  errors: []
});

// Mock配置
const createTestConfig = () => ({
  htmlMapping: {
    'title': { tag: 'h1', className: 'document-title' },
    'author': { tag: 'div', className: 'document-author' },
    'section': { tag: 'h2', className: 'latex-section' },
    'subsection': { tag: 'h3', className: 'latex-subsection' },
    'subsubsection': { tag: 'h4', className: 'latex-subsubsection' },
    'documentclass': { ignore: true }
  },
  cssRules: {
    'latex-document': {
      maxWidth: '8.5in',
      margin: '0 auto',
      padding: '1in',
      fontFamily: '"Times New Roman", serif',
      fontSize: '12pt',
      lineHeight: '1.6'
    },
    'document-title': {
      textAlign: 'center',
      fontSize: '18pt',
      fontWeight: 'bold',
      margin: '1.5em 0 1em 0'
    },
    'document-author': {
      textAlign: 'center',
      fontSize: '14pt',
      margin: '0.5em 0'
    },
    'latex-section': {
      fontSize: '16pt',
      fontWeight: 'bold',
      margin: '2em 0 1em 0'
    },
    'latex-subsection': {
      fontSize: '14pt',
      fontWeight: 'bold',
      margin: '1.5em 0 0.8em 0'
    },
    'latex-subsubsection': {
      fontSize: '13pt',
      fontWeight: 'bold',
      margin: '1.2em 0 0.6em 0'
    },
    'latex-content': {
      margin: '0.8em 0',
      textAlign: 'justify',
      textIndent: '2em'
    }
  },
  cssUtils: {
    objectToCss: (className, styleObject) => {
      const cssProperties = Object.entries(styleObject)
        .map(([property, value]) => {
          const cssProperty = property.replace(/[A-Z]/g, match => `-${match.toLowerCase()}`);
          return `  ${cssProperty}: ${value};`;
        })
        .join('\n');
      
      return `.${className} {\n${cssProperties}\n}`;
    }
  }
});

describe('Renderers 模块集成测试', () => {
  let HTMLRenderer, CSSGenerator;
  let htmlRenderer, cssGenerator;
  let parseResult, config;

  beforeAll(() => {
    // 加载模块
    if (typeof require !== 'undefined') {
      try {
        HTMLRenderer = require('../HTMLRenderer.js');
        CSSGenerator = require('../CSSGenerator.js');
      } catch (e) {
        HTMLRenderer = window.HTMLRenderer;
        CSSGenerator = window.CSSGenerator;
      }
    } else {
      HTMLRenderer = window.HTMLRenderer;
      CSSGenerator = window.CSSGenerator;
    }
  });

  beforeEach(() => {
    config = createTestConfig();
    parseResult = createCompleteParseResult();
    
    htmlRenderer = new HTMLRenderer({
      htmlMapping: config.htmlMapping,
      includeContainer: true,
      escapeContent: true,
      addComments: false
    });
    
    cssGenerator = new CSSGenerator({
      cssRules: config.cssRules,
      cssUtils: config.cssUtils,
      includeComments: true,
      minify: false
    });
  });

  describe('完整渲染流程测试', () => {
    test('应该生成匹配的HTML和CSS', () => {
      const html = htmlRenderer.render(parseResult);
      const css = cssGenerator.generate(parseResult);
      
      // 验证HTML结构
      expect(html).toContain('<div class="latex-document">');
      expect(html).toContain('<h1 class="document-title">Advanced LaTeX Document</h1>');
      expect(html).toContain('<div class="document-author">John Doe and Jane Smith</div>');
      expect(html).toContain('<h2 class="latex-section">1 Introduction</h2>');
      expect(html).toContain('</div>');
      
      // 验证CSS包含相应的样式
      expect(css).toContain('.latex-document {');
      expect(css).toContain('.document-title {');
      expect(css).toContain('.document-author {');
      expect(css).toContain('.latex-section {');
      
      // 验证CSS样式值
      expect(css).toContain('max-width: 8.5in');
      expect(css).toContain('text-align: center');
      expect(css).toContain('font-size: 18pt');
    });

    test('应该处理复杂的文档结构', () => {
      const html = htmlRenderer.render(parseResult);
      const css = cssGenerator.generate(parseResult);
      
      // 验证多级标题的处理
      expect(html).toContain('<h2 class="latex-section">'); // section
      expect(html).toContain('<h3 class="latex-subsection">'); // subsection (假设渲染器支持)
      expect(html).toContain('<p class="latex-content">'); // paragraph
      
      // 验证对应的CSS样式
      expect(css).toContain('.latex-section {');
      expect(css).toContain('font-size: 16pt');
    });

    test('应该正确处理忽略的元素', () => {
      const html = htmlRenderer.render(parseResult);
      
      // documentclass 应该被忽略
      expect(html).not.toContain('documentclass');
      expect(html).not.toContain('\\documentclass{article}');
    });

    test('应该生成语义化和可访问的输出', () => {
      const html = htmlRenderer.render(parseResult);
      
      // 验证语义化标签的使用
      expect(html).toMatch(/<h[1-6][^>]*>/); // 标题标签
      expect(html).toContain('<div class="latex-document">'); // 文档容器
      expect(html).toContain('<p class="latex-content">'); // 段落标签
      
      // 验证类名遵循语义化原则
      expect(html).toContain('class="document-title"');
      expect(html).toContain('class="document-author"');
      expect(html).toContain('class="latex-section"');
    });
  });

  describe('CSS-HTML 一致性测试', () => {
    test('HTML中使用的CSS类应该在CSS中定义', () => {
      const html = htmlRenderer.render(parseResult);
      const css = cssGenerator.generate(parseResult);
      
      // 提取HTML中的CSS类
      const htmlClasses = [];
      const classMatches = html.match(/class="([^"]+)"/g);
      if (classMatches) {
        classMatches.forEach(match => {
          const className = match.match(/class="([^"]+)"/)[1];
          if (!htmlClasses.includes(className)) {
            htmlClasses.push(className);
          }
        });
      }
      
      // 验证每个CSS类都在CSS中定义
      htmlClasses.forEach(className => {
        expect(css).toContain(`.${className} {`);
      });
    });

    test('应该提供完整的样式覆盖', () => {
      const html = htmlRenderer.render(parseResult);
      const css = cssGenerator.generate(parseResult);
      
      // 验证核心样式类的存在
      const requiredClasses = [
        'latex-document',
        'document-title',
        'document-author',
        'latex-section',
        'latex-content'
      ];
      
      requiredClasses.forEach(className => {
        if (html.includes(className)) {
          expect(css).toContain(`.${className} {`);
        }
      });
    });
  });

  describe('配置一致性测试', () => {
    test('应该尊重HTML渲染器的容器配置', () => {
      const noContainerRenderer = new HTMLRenderer({
        htmlMapping: config.htmlMapping,
        includeContainer: false
      });
      
      const html = noContainerRenderer.render(parseResult);
      expect(html).not.toContain('<div class="latex-document">');
      expect(html).toContain('<h1 class="document-title">');
    });

    test('应该尊重CSS生成器的注释配置', () => {
      const noCommentGenerator = new CSSGenerator({
        cssRules: config.cssRules,
        cssUtils: config.cssUtils,
        includeComments: false
      });
      
      const css = noCommentGenerator.generate(parseResult);
      expect(css).not.toContain('/* LaTeX Parser V3');
    });

    test('应该支持自定义样式规则', () => {
      const customCSSGenerator = new CSSGenerator({
        cssRules: config.cssRules,
        cssUtils: config.cssUtils,
        customRules: {
          'custom-highlight': {
            backgroundColor: '#ffff00',
            padding: '2px 4px'
          }
        }
      });
      
      const css = customCSSGenerator.generate(parseResult);
      expect(css).toContain('.custom-highlight {');
      expect(css).toContain('background-color: #ffff00');
    });
  });

  describe('错误恢复和降级测试', () => {
    test('应该处理不完整的解析结果', () => {
      const incompleteResult = {
        success: false,
        blocks: [
          { type: 'title', title: 'Incomplete Document' }
        ],
        errors: ['Parse error at line 10']
      };
      
      const html = htmlRenderer.render(incompleteResult);
      const css = cssGenerator.generate(incompleteResult);
      
      expect(html).toContain('<h1 class="document-title">Incomplete Document</h1>');
      expect(css).toContain('.document-title {');
    });

    test('应该处理缺失配置的情况', () => {
      const minimalRenderer = new HTMLRenderer();
      const minimalGenerator = new CSSGenerator();
      
      const html = minimalRenderer.render(parseResult);
      const css = minimalGenerator.generate(parseResult);
      
      expect(html).toBeDefined();
      expect(css).toBeDefined();
      expect(html.length).toBeGreaterThan(0);
      expect(css.length).toBeGreaterThan(0);
    });

    test('应该处理空的输入数据', () => {
      const emptyResult = {
        success: true,
        blocks: [],
        metadata: {},
        warnings: [],
        errors: []
      };
      
      const html = htmlRenderer.render(emptyResult);
      const css = cssGenerator.generate(emptyResult);
      
      expect(html).toContain('<div class="latex-document">');
      expect(html).toContain('</div>');
      expect(css).toContain('.latex-document {');
    });
  });

  describe('性能和扩展性测试', () => {
    test('应该高效处理大型文档', () => {
      // 创建大型文档
      const largeBlocks = Array.from({ length: 200 }, (_, i) => ({
        type: i % 3 === 0 ? 'section' : 'paragraph',
        title: i % 3 === 0 ? `Section ${Math.floor(i/3)}` : undefined,
        content: `Content for block ${i}`
      }));
      
      const largeParseResult = {
        ...parseResult,
        blocks: largeBlocks
      };
      
      const startTime = Date.now();
      const html = htmlRenderer.render(largeParseResult);
      const css = cssGenerator.generate(largeParseResult);
      const endTime = Date.now();
      
      expect(html).toBeDefined();
      expect(css).toBeDefined();
      expect(endTime - startTime).toBeLessThan(2000); // 应该在2秒内完成
    });

    test('应该支持多个独立的渲染实例', () => {
      const renderer1 = new HTMLRenderer({ includeContainer: true });
      const renderer2 = new HTMLRenderer({ includeContainer: false });
      const generator1 = new CSSGenerator({ minify: false });
      const generator2 = new CSSGenerator({ minify: true });
      
      const html1 = renderer1.render(parseResult);
      const html2 = renderer2.render(parseResult);
      const css1 = generator1.generate(parseResult);
      const css2 = generator2.generate(parseResult);
      
      // 验证实例独立性
      expect(html1).toContain('<div class="latex-document">');
      expect(html2).not.toContain('<div class="latex-document">');
      expect(css1.length).toBeGreaterThan(css2.length); // 未压缩应该更长
    });
  });

  describe('输出质量验证', () => {
    test('应该生成有效的HTML', () => {
      const html = htmlRenderer.render(parseResult);
      
      // 基础HTML有效性检查
      expect(html).toMatch(/^<div class="latex-document">/);
      expect(html).toMatch(/<\/div>$/);
      
      // 标签配对检查
      const openTags = (html.match(/<[^/][^>]*>/g) || []).length;
      const closeTags = (html.match(/<\/[^>]+>/g) || []).length;
      const selfClosingTags = (html.match(/<[^>]+\/>/g) || []).length;
      
      // 简单的配对检查
      expect(openTags - selfClosingTags).toBe(closeTags);
    });

    test('应该生成有效的CSS', () => {
      const css = cssGenerator.generate(parseResult);
      
      // 基础CSS有效性检查
      expect(css).toMatch(/\.[a-zA-Z-]+ \{/); // CSS选择器格式
      expect(css).toMatch(/[a-zA-Z-]+: [^;]+;/); // CSS属性格式
      
      // 括号配对检查
      const openBraces = (css.match(/\{/g) || []).length;
      const closeBraces = (css.match(/\}/g) || []).length;
      expect(openBraces).toBe(closeBraces);
    });

    test('应该提供一致的输出格式', () => {
      const result1 = htmlRenderer.render(parseResult);
      const result2 = htmlRenderer.render(parseResult);
      
      // 相同输入应该产生相同输出
      expect(result1).toBe(result2);
      
      const css1 = cssGenerator.generate(parseResult);
      const css2 = cssGenerator.generate(parseResult);
      
      // 相同输入应该产生相同CSS（除了时间戳）
      const css1NoTime = css1.replace(/\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}.\d{3}Z/, 'TIME');
      const css2NoTime = css2.replace(/\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}.\d{3}Z/, 'TIME');
      expect(css1NoTime).toBe(css2NoTime);
    });
  });

  describe('扩展性和自定义测试', () => {
    test('应该支持自定义HTML映射', () => {
      const customMapping = {
        'custom-block': { tag: 'article', className: 'custom-article' }
      };
      
      const customRenderer = new HTMLRenderer({
        htmlMapping: { ...config.htmlMapping, ...customMapping }
      });
      
      const customParseResult = {
        ...parseResult,
        blocks: [{ type: 'custom-block', content: 'Custom content' }]
      };
      
      const html = customRenderer.render(customParseResult);
      expect(html).toContain('custom-article');
    });

    test('应该支持主题化CSS生成', () => {
      const darkTheme = {
        'latex-document': {
          backgroundColor: '#1a1a1a',
          color: '#ffffff'
        },
        'document-title': {
          color: '#66d9ef'
        }
      };
      
      const themedGenerator = new CSSGenerator({
        cssRules: { ...config.cssRules, ...darkTheme },
        cssUtils: config.cssUtils
      });
      
      const css = themedGenerator.generate(parseResult);
      expect(css).toContain('background-color: #1a1a1a');
      expect(css).toContain('color: #66d9ef');
    });
  });
});