/**
 * HTMLRenderer 测试用例
 * 基于interface.md规范进行全面测试
 */

// 模拟环境配置
const createMockConfig = () => ({
  'title': { tag: 'h1', className: 'document-title' },
  'author': { tag: 'div', className: 'document-author' },
  'section': { tag: 'h2', className: 'latex-section' },
  'documentclass': { ignore: true }
});

// 模拟解析结果数据
const createMockParseResult = (blocks = []) => ({
  success: true,
  processedText: 'Sample processed text',
  blocks: blocks,
  metadata: { title: 'Test Document', author: 'Test Author' },
  warnings: [],
  errors: []
});

// 创建测试块数据
const createTestBlocks = () => [
  { type: 'title', content: 'Test Title', title: 'Test Title' },
  { type: 'author', content: 'Test Author', author: 'Test Author' },
  { type: 'maketitle' },
  { type: 'section', content: 'Introduction', title: 'Introduction' },
  { type: 'paragraph', content: 'This is a test paragraph.' },
  { type: 'documentclass', content: '\\documentclass{article}' }
];

describe('HTMLRenderer 接口规范测试', () => {
  let HTMLRenderer;
  let renderer;

  beforeAll(() => {
    // 加载HTMLRenderer (适配Node.js和浏览器环境)
    if (typeof require !== 'undefined') {
      try {
        HTMLRenderer = require('../HTMLRenderer.js');
      } catch (e) {
        // 如果在浏览器环境中运行
        HTMLRenderer = window.HTMLRenderer;
      }
    } else {
      HTMLRenderer = window.HTMLRenderer;
    }
  });

  beforeEach(() => {
    renderer = new HTMLRenderer({
      htmlMapping: createMockConfig()
    });
  });

  describe('构造函数测试', () => {
    test('应该使用默认配置创建实例', () => {
      const defaultRenderer = new HTMLRenderer();
      expect(defaultRenderer).toBeDefined();
      expect(defaultRenderer.options).toBeDefined();
      expect(defaultRenderer.options.includeContainer).toBe(true);
      expect(defaultRenderer.options.escapeContent).toBe(true);
      expect(defaultRenderer.options.addComments).toBe(false);
    });

    test('应该接受自定义配置选项', () => {
      const customRenderer = new HTMLRenderer({
        includeContainer: false,
        escapeContent: false,
        addComments: true
      });
      
      expect(customRenderer.options.includeContainer).toBe(false);
      expect(customRenderer.options.escapeContent).toBe(false);
      expect(customRenderer.options.addComments).toBe(true);
    });

    test('应该接受自定义HTML映射配置', () => {
      const customMapping = { 'test': { tag: 'span' } };
      const customRenderer = new HTMLRenderer({
        htmlMapping: customMapping
      });
      
      expect(customRenderer.htmlMapping).toEqual(customMapping);
    });
  });

  describe('核心方法：render(parseResult)', () => {
    test('应该正确渲染完整的解析结果', () => {
      const parseResult = createMockParseResult(createTestBlocks());
      const html = renderer.render(parseResult);
      
      expect(html).toContain('<div class="latex-document">');
      expect(html).toContain('<h1 class="document-title">Test Title</h1>');
      expect(html).toContain('<div class="document-author">Test Author</div>');
      expect(html).toContain('<h2 class="latex-section">Introduction</h2>');
      expect(html).toContain('<p class="latex-content">This is a test paragraph.</p>');
      expect(html).toContain('</div>');
    });

    test('应该处理空的解析结果', () => {
      const html = renderer.render(null);
      expect(html).toBe('');
    });

    test('应该处理没有blocks的解析结果', () => {
      const parseResult = { success: true };
      const html = renderer.render(parseResult);
      expect(html).toBe('');
    });

    test('应该支持不包含容器的渲染', () => {
      const noContainerRenderer = new HTMLRenderer({
        includeContainer: false
      });
      const parseResult = createMockParseResult([
        { type: 'title', content: 'Test', title: 'Test' }
      ]);
      
      const html = noContainerRenderer.render(parseResult);
      expect(html).not.toContain('<div class="latex-document">');
      expect(html).toContain('<h1 class="document-title">Test</h1>');
    });
  });

  describe('核心方法：renderBlocks(blocks)', () => {
    test('应该正确渲染数据块集合', () => {
      // 注意：当前实现中没有独立的renderBlocks方法
      // 测试通过render方法来验证块渲染功能
      const blocks = [
        { type: 'title', title: 'Block Title' },
        { type: 'section', title: 'Block Section' }
      ];
      const parseResult = createMockParseResult(blocks);
      const html = renderer.render(parseResult);
      
      expect(html).toContain('<h1 class="document-title">Block Title</h1>');
      expect(html).toContain('<h2 class="latex-section">Block Section</h2>');
    });
  });

  describe('核心方法：renderMetadata(metadata)', () => {
    test('应该能够处理元数据相关的块', () => {
      const metadataBlocks = [
        { type: 'title', title: 'Document Title' },
        { type: 'author', author: 'John Doe' }
      ];
      const parseResult = createMockParseResult(metadataBlocks);
      const html = renderer.render(parseResult);
      
      expect(html).toContain('<h1 class="document-title">Document Title</h1>');
      expect(html).toContain('<div class="document-author">John Doe</div>');
    });
  });

  describe('数据块类型处理测试', () => {
    test('应该正确渲染title块', () => {
      const titleBlock = { type: 'title', title: 'My Title', content: 'My Title' };
      const html = renderer.renderBlock(titleBlock);
      expect(html).toBe('<h1 class="document-title">My Title</h1>');
    });

    test('应该正确渲染author块', () => {
      const authorBlock = { type: 'author', author: 'John Doe', content: 'John Doe' };
      const html = renderer.renderBlock(authorBlock);
      expect(html).toBe('<div class="document-author">John Doe</div>');
    });

    test('应该正确渲染section块', () => {
      const sectionBlock = { type: 'section', title: 'Introduction', content: 'Introduction' };
      const html = renderer.renderBlock(sectionBlock);
      expect(html).toBe('<h2 class="latex-section">Introduction</h2>');
    });

    test('应该正确渲染maketitle块', () => {
      const maketitleBlock = { type: 'maketitle' };
      const html = renderer.renderBlock(maketitleBlock);
      expect(html).toBe('<div class="title-separator"></div>');
    });

    test('应该正确渲染paragraph块', () => {
      const paragraphBlock = { type: 'paragraph', content: 'Test content' };
      const html = renderer.renderBlock(paragraphBlock);
      expect(html).toBe('<p class="latex-content">Test content</p>');
    });

    test('应该忽略documentclass块', () => {
      const docclassBlock = { type: 'documentclass', content: '\\documentclass{article}' };
      const html = renderer.renderBlock(docclassBlock);
      expect(html).toBe('');
    });
  });

  describe('忽略类型和注释处理', () => {
    test('应该忽略标记为ignore的类型', () => {
      const ignoredBlock = { type: 'documentclass', content: '\\documentclass{article}' };
      const html = renderer.renderBlock(ignoredBlock);
      expect(html).toBe('');
    });

    test('应该在启用注释时为忽略的类型添加HTML注释', () => {
      const commentRenderer = new HTMLRenderer({
        htmlMapping: createMockConfig(),
        addComments: true
      });
      const ignoredBlock = { type: 'documentclass', content: '\\documentclass{article}' };
      const html = commentRenderer.renderBlock(ignoredBlock);
      expect(html).toContain('<!-- LaTeX命令已忽略: documentclass -->');
    });
  });

  describe('通用块处理', () => {
    test('应该将未知类型但有内容的块作为段落处理', () => {
      const unknownBlock = { type: 'unknown', content: 'Some content' };
      const html = renderer.renderBlock(unknownBlock);
      expect(html).toBe('<p class="latex-content">Some content</p>');
    });

    test('应该返回空字符串对于没有内容的未知块', () => {
      const emptyBlock = { type: 'unknown' };
      const html = renderer.renderBlock(emptyBlock);
      expect(html).toBe('');
    });
  });

  describe('HTML转义功能', () => {
    test('应该默认转义HTML特殊字符', () => {
      const content = '<script>alert("xss")</script> & "quotes"';
      const escaped = renderer.escapeHtml(content);
      expect(escaped).toBe('&lt;script&gt;alert(&quot;xss&quot;)&lt;/script&gt; &amp; &quot;quotes&quot;');
    });

    test('应该在禁用转义时保持原内容', () => {
      const noEscapeRenderer = new HTMLRenderer({
        escapeContent: false
      });
      const content = '<b>bold</b> & "quotes"';
      const result = noEscapeRenderer.escapeHtml(content);
      expect(result).toBe(content);
    });

    test('应该处理空内容和null值', () => {
      expect(renderer.escapeHtml('')).toBe('');
      expect(renderer.escapeHtml(null)).toBe('');
      expect(renderer.escapeHtml(undefined)).toBe('');
    });
  });

  describe('配置和选项管理', () => {
    test('应该支持运行时设置选项', () => {
      renderer.setOptions({ includeContainer: false });
      expect(renderer.options.includeContainer).toBe(false);
      
      const parseResult = createMockParseResult([
        { type: 'title', title: 'Test' }
      ]);
      const html = renderer.render(parseResult);
      expect(html).not.toContain('<div class="latex-document">');
    });

    test('应该支持链式调用', () => {
      const result = renderer.setOptions({ addComments: true });
      expect(result).toBe(renderer);
    });
  });

  describe('CSS类命名规范验证', () => {
    test('应该使用规范的CSS类名', () => {
      const blocks = createTestBlocks();
      const parseResult = createMockParseResult(blocks);
      const html = renderer.render(parseResult);
      
      // 验证使用了规范的latex-前缀类名
      expect(html).toMatch(/class="latex-document"/);
      expect(html).toMatch(/class="document-title"/);
      expect(html).toMatch(/class="document-author"/);
      expect(html).toMatch(/class="latex-section"/);
      expect(html).toMatch(/class="latex-content"/);
      expect(html).toMatch(/class="title-separator"/);
    });
  });

  describe('错误处理和边界条件', () => {
    test('应该处理null或undefined的块', () => {
      expect(renderer.renderBlock(null)).toBe('');
      expect(renderer.renderBlock(undefined)).toBe('');
      expect(renderer.renderBlock({})).toBe('');
    });

    test('应该处理没有type的块', () => {
      const blockWithoutType = { content: 'some content' };
      expect(renderer.renderBlock(blockWithoutType)).toBe('');
    });

    test('应该优雅处理缺失的htmlMapping', () => {
      const noMappingRenderer = new HTMLRenderer({ htmlMapping: null });
      const block = { type: 'title', title: 'Test' };
      const html = noMappingRenderer.renderBlock(block);
      expect(html).toBe('<h1 class="document-title">Test</h1>');
    });
  });

  describe('语义化HTML和可访问性', () => {
    test('应该生成语义化的HTML结构', () => {
      const parseResult = createMockParseResult([
        { type: 'title', title: 'Main Title' },
        { type: 'section', title: 'Section 1' }
      ]);
      const html = renderer.render(parseResult);
      
      // 验证使用了语义化标签
      expect(html).toContain('<h1'); // 主标题
      expect(html).toContain('<h2'); // 章节标题
      expect(html).toContain('<div'); // 容器和其他块级元素
    });
  });

  describe('兼容性测试', () => {
    test('应该在Node.js环境中正常工作', () => {
      // 这个测试验证模块导出机制
      expect(HTMLRenderer).toBeDefined();
      expect(typeof HTMLRenderer).toBe('function');
    });

    test('应该能够创建多个独立的实例', () => {
      const renderer1 = new HTMLRenderer({ includeContainer: true });
      const renderer2 = new HTMLRenderer({ includeContainer: false });
      
      expect(renderer1.options.includeContainer).toBe(true);
      expect(renderer2.options.includeContainer).toBe(false);
      
      // 确保实例间不相互影响
      renderer1.setOptions({ addComments: true });
      expect(renderer2.options.addComments).toBe(false);
    });
  });
});