/**
 * CSSGenerator 测试用例
 * 基于interface.md规范进行全面测试
 */

// 模拟CSS规则配置
const createMockCSSRules = () => ({
  'latex-document': {
    maxWidth: '8.5in',
    margin: '0 auto',
    padding: '1in',
    fontFamily: 'Times New Roman, serif'
  },
  'document-title': {
    textAlign: 'center',
    fontSize: '18pt',
    fontWeight: 'bold'
  },
  'latex-section': {
    fontSize: '16pt',
    fontWeight: 'bold',
    margin: '2em 0 1em 0'
  }
});

// 模拟CSS工具
const createMockCSSUtils = () => ({
  objectToCss: (className, styleObject) => {
    const cssProperties = Object.entries(styleObject)
      .map(([property, value]) => {
        const cssProperty = property.replace(/[A-Z]/g, match => `-${match.toLowerCase()}`);
        return `  ${cssProperty}: ${value};`;
      })
      .join('\n');
    
    return `.${className} {\n${cssProperties}\n}`;
  }
});

// 创建测试解析结果
const createMockParseResult = (blocks = []) => ({
  success: true,
  processedText: 'Sample processed text',
  blocks: blocks,
  metadata: { title: 'Test Document' },
  warnings: [],
  errors: []
});

// 创建测试块数据
const createTestBlocks = () => [
  { type: 'title', content: 'Test Title' },
  { type: 'author', content: 'Test Author' },
  { type: 'section', content: 'Introduction' },
  { type: 'subsection', content: 'Background' },
  { type: 'paragraph', content: 'Sample paragraph' }
];

describe('CSSGenerator 接口规范测试', () => {
  let CSSGenerator;
  let generator;

  beforeAll(() => {
    // 加载CSSGenerator (适配Node.js和浏览器环境)
    if (typeof require !== 'undefined') {
      try {
        CSSGenerator = require('../CSSGenerator.js');
      } catch (e) {
        CSSGenerator = window.CSSGenerator;
      }
    } else {
      CSSGenerator = window.CSSGenerator;
    }
  });

  beforeEach(() => {
    generator = new CSSGenerator({
      cssRules: createMockCSSRules(),
      cssUtils: createMockCSSUtils()
    });
  });

  describe('构造函数测试', () => {
    test('应该使用默认配置创建实例', () => {
      const defaultGenerator = new CSSGenerator();
      expect(defaultGenerator).toBeDefined();
      expect(defaultGenerator.options).toBeDefined();
      expect(defaultGenerator.options.includeComments).toBe(true);
      expect(defaultGenerator.options.minify).toBe(false);
    });

    test('应该接受自定义配置选项', () => {
      const customGenerator = new CSSGenerator({
        includeComments: false,
        minify: true,
        customRules: { 'custom-class': { color: 'red' } }
      });
      
      expect(customGenerator.options.includeComments).toBe(false);
      expect(customGenerator.options.minify).toBe(true);
      expect(customGenerator.options.customRules['custom-class']).toEqual({ color: 'red' });
    });

    test('应该使用自定义CSS规则和工具', () => {
      const customRules = { 'test-class': { fontSize: '14px' } };
      const customUtils = { objectToCss: jest.fn() };
      
      const customGenerator = new CSSGenerator({
        cssRules: customRules,
        cssUtils: customUtils
      });
      
      expect(customGenerator.cssRules).toEqual(customRules);
      expect(customGenerator.cssUtils).toEqual(customUtils);
    });

    test('应该处理缺失的配置文件', () => {
      // 测试当require失败时的降级处理
      const generator = new CSSGenerator({});
      expect(generator).toBeDefined();
      expect(generator.cssRules).toBeDefined();
      expect(generator.cssUtils).toBeDefined();
    });
  });

  describe('核心方法：generate(parseResult)', () => {
    test('应该生成完整的CSS样式表', () => {
      const parseResult = createMockParseResult(createTestBlocks());
      const css = generator.generate(parseResult);
      
      expect(css).toBeDefined();
      expect(typeof css).toBe('string');
      expect(css.length).toBeGreaterThan(0);
    });

    test('应该包含CSS注释头当includeComments为true时', () => {
      const parseResult = createMockParseResult([]);
      const css = generator.generate(parseResult);
      
      expect(css).toContain('/* LaTeX Parser V3 - 自动生成的样式 */');
      expect(css).toContain('/* 生成时间:');
    });

    test('应该排除注释当includeComments为false时', () => {
      generator.setOptions({ includeComments: false });
      const parseResult = createMockParseResult([]);
      const css = generator.generate(parseResult);
      
      expect(css).not.toContain('/* LaTeX Parser V3');
    });

    test('应该处理null或undefined的解析结果', () => {
      const css1 = generator.generate(null);
      const css2 = generator.generate(undefined);
      const css3 = generator.generate({});
      
      expect(css1).toBeDefined();
      expect(css2).toBeDefined();
      expect(css3).toBeDefined();
    });

    test('应该生成压缩CSS当minify为true时', () => {
      const minifyGenerator = new CSSGenerator({
        cssRules: createMockCSSRules(),
        cssUtils: createMockCSSUtils(),
        minify: true,
        includeComments: false
      });
      
      const parseResult = createMockParseResult([]);
      const css = minifyGenerator.generate(parseResult);
      
      // 压缩CSS应该不包含多余的空白
      expect(css).not.toContain('\n\n');
      expect(css.split('\n').length).toBeLessThan(5);
    });
  });

  describe('核心方法：generateBlockStyles(blocks)', () => {
    test('应该根据块类型生成特定样式', () => {
      // 通过generate方法间接测试，因为generateSpecificStyles是内部实现
      const blocks = [
        { type: 'title' },
        { type: 'section' },
        { type: 'subsection' }
      ];
      const parseResult = createMockParseResult(blocks);
      const css = generator.generate(parseResult);
      
      expect(css).toBeDefined();
      expect(css.length).toBeGreaterThan(0);
    });
  });

  describe('核心方法：generateTheme(theme)', () => {
    test('应该支持主题生成功能', () => {
      // 当前实现没有独立的generateTheme方法，但通过配置可以实现
      const themeRules = {
        'latex-document': { background: '#f5f5f5' },
        'document-title': { color: '#2c3e50' }
      };
      
      const themedGenerator = new CSSGenerator({
        cssRules: themeRules,
        cssUtils: createMockCSSUtils()
      });
      
      const parseResult = createMockParseResult([{ type: 'title' }]);
      const css = themedGenerator.generate(parseResult);
      
      expect(css).toContain('background: #f5f5f5');
      expect(css).toContain('color: #2c3e50');
    });
  });

  describe('核心方法：generateResponsiveStyles()', () => {
    test('应该生成响应式样式', () => {
      const responsiveCSS = generator.generateResponsiveStyles();
      
      expect(responsiveCSS).toContain('@media (max-width: 768px)');
      expect(responsiveCSS).toContain('@media print');
      expect(responsiveCSS).toContain('/* 响应式样式 */');
    });
  });

  describe('核心方法：generatePrintStyles()', () => {
    test('应该在响应式样式中包含打印样式', () => {
      const responsiveCSS = generator.generateResponsiveStyles();
      
      expect(responsiveCSS).toContain('@media print');
      expect(responsiveCSS).toContain('padding: 0');
      expect(responsiveCSS).toContain('max-width: none');
      expect(responsiveCSS).toContain('background: transparent');
    });
  });

  describe('基础样式生成', () => {
    test('应该生成所有基础CSS类的样式', () => {
      const parseResult = createMockParseResult([]);
      const css = generator.generate(parseResult);
      
      // 验证基础类的存在
      expect(css).toContain('.latex-document');
      expect(css).toContain('.document-title');
      expect(css).toContain('.latex-section');
    });

    test('应该跳过不存在规则的CSS类', () => {
      const limitedRules = { 'latex-document': { margin: '0' } };
      const limitedGenerator = new CSSGenerator({
        cssRules: limitedRules,
        cssUtils: createMockCSSUtils()
      });
      
      const parseResult = createMockParseResult([]);
      const css = limitedGenerator.generate(parseResult);
      
      expect(css).toContain('.latex-document');
      expect(css).not.toContain('.document-title'); // 不存在的规则
    });
  });

  describe('特定样式生成', () => {
    test('应该为使用的块类型生成特定样式', () => {
      const parseResult = createMockParseResult([
        { type: 'title' },
        { type: 'section' }
      ]);
      const css = generator.generate(parseResult);
      
      expect(css).toContain('特定内容样式');
    });

    test('应该收集和映射块类型到CSS类', () => {
      const parseResult = createMockParseResult([
        { type: 'title' },
        { type: 'author' },
        { type: 'section' },
        { type: 'subsection' },
        { type: 'paragraph' }
      ]);
      const css = generator.generate(parseResult);
      
      expect(css).toBeDefined();
      expect(css.length).toBeGreaterThan(0);
    });

    test('应该处理嵌套子块', () => {
      const parseResult = createMockParseResult([
        { 
          type: 'section',
          children: [
            { type: 'subsection' },
            { type: 'paragraph' }
          ]
        }
      ]);
      const css = generator.generate(parseResult);
      
      expect(css).toBeDefined();
      expect(css.length).toBeGreaterThan(0);
    });

    test('应该区分基础类和特定类', () => {
      const parseResult = createMockParseResult([{ type: 'title' }]);
      const css = generator.generate(parseResult);
      
      // 基础样式应该在特定样式之前
      const baseIndex = css.indexOf('.latex-document');
      const specificIndex = css.indexOf('特定内容样式');
      
      expect(baseIndex).toBeLessThan(specificIndex);
    });
  });

  describe('自定义样式生成', () => {
    test('应该生成自定义CSS规则', () => {
      const customGenerator = new CSSGenerator({
        cssRules: createMockCSSRules(),
        cssUtils: createMockCSSUtils(),
        customRules: {
          'my-custom-class': { color: 'blue', fontSize: '14px' }
        }
      });
      
      const parseResult = createMockParseResult([]);
      const css = customGenerator.generate(parseResult);
      
      expect(css).toContain('/* 自定义样式 */');
      expect(css).toContain('.my-custom-class');
      expect(css).toContain('color: blue');
      expect(css).toContain('font-size: 14px');
    });

    test('应该支持运行时添加自定义规则', () => {
      generator.addCustomRule('runtime-class', { margin: '10px' });
      
      const parseResult = createMockParseResult([]);
      const css = generator.generate(parseResult);
      
      expect(css).toContain('.runtime-class');
      expect(css).toContain('margin: 10px');
    });

    test('应该支持链式调用添加规则', () => {
      const result = generator.addCustomRule('chain-class', { padding: '5px' });
      expect(result).toBe(generator);
    });
  });

  describe('CSS工具功能', () => {
    test('应该正确转换对象到CSS', () => {
      const styleObject = {
        fontSize: '16px',
        fontWeight: 'bold',
        textAlign: 'center',
        marginTop: '1em'
      };
      
      const css = generator.cssUtils.objectToCss('test-class', styleObject);
      
      expect(css).toContain('.test-class {');
      expect(css).toContain('font-size: 16px;');
      expect(css).toContain('font-weight: bold;');
      expect(css).toContain('text-align: center;');
      expect(css).toContain('margin-top: 1em;');
      expect(css).toContain('}');
    });
  });

  describe('CSS压缩功能', () => {
    test('应该正确压缩CSS', () => {
      const originalCSS = `
        /* 注释 */
        .test-class {
          font-size: 16px ;
          margin: 0 auto ;
        }
        
        .another-class {
          padding: 10px;
        }
      `;
      
      const minified = generator.minifyCSS(originalCSS);
      
      expect(minified).not.toContain('/* 注释 */');
      expect(minified).not.toContain('\n\n');
      expect(minified.length).toBeLessThan(originalCSS.length);
      expect(minified).toContain('.test-class{');
      expect(minified).toContain('.another-class{');
    });

    test('应该移除最后一个分号', () => {
      const css = '.test { color: red; }';
      const minified = generator.minifyCSS(css);
      expect(minified).toBe('.test{color:red}');
    });

    test('应该压缩空白字符', () => {
      const css = '.test   {  color  :  red  ;  }';
      const minified = generator.minifyCSS(css);
      expect(minified).toBe('.test{color:red}');
    });
  });

  describe('默认CSS规则和工具', () => {
    test('应该提供默认CSS规则', () => {
      const defaultGenerator = new CSSGenerator();
      const defaultRules = defaultGenerator.createDefaultCSSRules();
      
      expect(defaultRules).toHaveProperty('latex-document');
      expect(defaultRules).toHaveProperty('document-title');
      expect(defaultRules).toHaveProperty('document-author');
      expect(defaultRules).toHaveProperty('latex-section');
      expect(defaultRules).toHaveProperty('latex-content');
      
      // 验证默认样式的合理性
      expect(defaultRules['latex-document']).toHaveProperty('fontFamily');
      expect(defaultRules['document-title']).toHaveProperty('textAlign');
      expect(defaultRules['latex-section']).toHaveProperty('fontWeight');
    });

    test('应该提供默认CSS工具', () => {
      const defaultGenerator = new CSSGenerator();
      const defaultUtils = defaultGenerator.createDefaultCSSUtils();
      
      expect(defaultUtils).toHaveProperty('objectToCss');
      expect(typeof defaultUtils.objectToCss).toBe('function');
      
      // 测试默认工具的功能
      const css = defaultUtils.objectToCss('test', { fontSize: '16px' });
      expect(css).toContain('.test');
      expect(css).toContain('font-size: 16px');
    });
  });

  describe('配置和选项管理', () => {
    test('应该支持运行时设置选项', () => {
      generator.setOptions({ minify: true, includeComments: false });
      expect(generator.options.minify).toBe(true);
      expect(generator.options.includeComments).toBe(false);
    });

    test('应该支持链式调用', () => {
      const result = generator.setOptions({ minify: true });
      expect(result).toBe(generator);
    });

    test('应该合并新选项而不覆盖现有选项', () => {
      generator.setOptions({ customRules: { 'test': { color: 'red' } } });
      generator.setOptions({ minify: true });
      
      expect(generator.options.minify).toBe(true);
      expect(generator.options.customRules).toHaveProperty('test');
    });
  });

  describe('错误处理和边界条件', () => {
    test('应该处理空的块数组', () => {
      const parseResult = createMockParseResult([]);
      const css = generator.generate(parseResult);
      
      expect(css).toBeDefined();
      expect(css).toContain('.latex-document'); // 基础样式应该存在
    });

    test('应该处理无效的块类型', () => {
      const parseResult = createMockParseResult([
        { type: 'invalid-type', content: 'test' },
        { type: null, content: 'test' },
        { content: 'test' } // 没有type
      ]);
      const css = generator.generate(parseResult);
      
      expect(css).toBeDefined();
      // 不应该因为无效块而崩溃
    });

    test('应该处理缺失的CSS规则', () => {
      const limitedGenerator = new CSSGenerator({
        cssRules: {}, // 空规则
        cssUtils: createMockCSSUtils()
      });
      
      const parseResult = createMockParseResult([{ type: 'title' }]);
      const css = limitedGenerator.generate(parseResult);
      
      expect(css).toBeDefined();
      // 应该优雅处理缺失的规则
    });
  });

  describe('BEM命名规范验证', () => {
    test('应该生成符合BEM规范的CSS类名', () => {
      const parseResult = createMockParseResult(createTestBlocks());
      const css = generator.generate(parseResult);
      
      // 验证latex-前缀的使用
      expect(css).toMatch(/\.latex-document/);
      expect(css).toMatch(/\.latex-section/);
      expect(css).toMatch(/\.document-title/);
      expect(css).toMatch(/\.document-author/);
    });
  });

  describe('兼容性测试', () => {
    test('应该在Node.js环境中正常工作', () => {
      expect(CSSGenerator).toBeDefined();
      expect(typeof CSSGenerator).toBe('function');
    });

    test('应该能够创建多个独立的实例', () => {
      const generator1 = new CSSGenerator({ minify: true });
      const generator2 = new CSSGenerator({ minify: false });
      
      expect(generator1.options.minify).toBe(true);
      expect(generator2.options.minify).toBe(false);
      
      // 确保实例间不相互影响
      generator1.setOptions({ includeComments: false });
      expect(generator2.options.includeComments).toBe(true);
    });
  });

  describe('性能和优化', () => {
    test('应该高效处理大量块', () => {
      const largeBlocks = Array.from({ length: 100 }, (_, i) => ({
        type: i % 2 === 0 ? 'section' : 'paragraph',
        content: `Block ${i}`
      }));
      
      const parseResult = createMockParseResult(largeBlocks);
      
      const startTime = Date.now();
      const css = generator.generate(parseResult);
      const endTime = Date.now();
      
      expect(css).toBeDefined();
      expect(endTime - startTime).toBeLessThan(1000); // 应该在1秒内完成
    });

    test('压缩应该显著减少CSS大小', () => {
      const parseResult = createMockParseResult(createTestBlocks());
      
      generator.setOptions({ minify: false, includeComments: true });
      const uncompressed = generator.generate(parseResult);
      
      generator.setOptions({ minify: true, includeComments: false });
      const compressed = generator.generate(parseResult);
      
      expect(compressed.length).toBeLessThan(uncompressed.length);
    });
  });
});