/**
 * HTML渲染器 - 专门负责将LaTeX解析结果转换为HTML
 * 职责：接收结构化数据 → 生成格式正确的HTML
 * 文件大小控制：< 200行
 */

class HTMLRenderer {
  constructor(config = {}) {
    // 加载HTML映射配置
    this.htmlMapping = config.htmlMapping || (typeof window !== 'undefined' ? 
      window.LATEX_HTML_MAPPING : require('../config/latex-html-mapping.js').LATEX_HTML_MAPPING);
    
    this.options = {
      includeContainer: config.includeContainer !== false,
      escapeContent: config.escapeContent !== false,
      addComments: config.addComments === true,
      ...config.options
    };
  }

  /**
   * 渲染LaTeX解析结果为HTML
   * @param {Object} parseResult - V3解析结果
   * @returns {string} HTML字符串
   */
  render(parseResult) {
    if (!parseResult || !parseResult.blocks) {
      return '';
    }

    const htmlParts = [];
    
    // 添加文档容器开始
    if (this.options.includeContainer) {
      htmlParts.push(this.renderDocumentStart());
    }

    // 渲染所有块
    parseResult.blocks.forEach(block => {
      const html = this.renderBlock(block);
      if (html) {
        htmlParts.push(html);
      }
    });

    // 添加文档容器结束
    if (this.options.includeContainer) {
      htmlParts.push(this.renderDocumentEnd());
    }

    return htmlParts.join('\n');
  }

  /**
   * 渲染单个块
   * @param {Object} block - 解析出的块
   * @returns {string} HTML字符串
   */
  renderBlock(block) {
    if (!block || !block.type) {
      return '';
    }

    // 检查是否应该忽略此类型
    const mapping = this.htmlMapping ? this.htmlMapping[block.type] : null;
    if (mapping && mapping.ignore) {
      if (this.options.addComments) {
        return `<!-- LaTeX命令已忽略: ${block.type} -->`;
      }
      return '';
    }

    // 根据类型渲染
    switch (block.type) {
      case 'title':
        return this.renderTitle(block);
      case 'author':
        return this.renderAuthor(block);
      case 'section':
        return this.renderSection(block);
      case 'maketitle':
        return this.renderMaketitle(block);
      case 'paragraph':
        return this.renderParagraph(block);
      case 'documentclass':
        return '';
      default:
        return this.renderGeneric(block);
    }
  }

  /**
   * 渲染标题
   */
  renderTitle(block) {
    const content = this.escapeHtml(block.title || block.content);
    return `<h1 class="document-title">${content}</h1>`;
  }

  /**
   * 渲染作者
   */
  renderAuthor(block) {
    const content = this.escapeHtml(block.author || block.content);
    return `<div class="document-author">${content}</div>`;
  }

  /**
   * 渲染章节标题
   */
  renderSection(block) {
    const content = this.escapeHtml(block.title || block.content);
    return `<h2 class="latex-section">${content}</h2>`;
  }

  /**
   * 渲染maketitle分隔线
   */
  renderMaketitle(block) {
    return '<div class="title-separator"></div>';
  }

  /**
   * 渲染段落
   */
  renderParagraph(block) {
    const content = this.escapeHtml(block.content);
    return `<p class="latex-content">${content}</p>`;
  }

  /**
   * 渲染通用块
   */
  renderGeneric(block) {
    if (block.content) {
      // 如果有内容但没有特定映射，作为段落处理
      return this.renderParagraph(block);
    }
    return '';
  }

  /**
   * 渲染文档开始
   */
  renderDocumentStart() {
    return '<div class="latex-document">';
  }

  /**
   * 渲染文档结束
   */
  renderDocumentEnd() {
    return '</div>';
  }

  /**
   * HTML转义
   */
  escapeHtml(text) {
    if (!this.options.escapeContent || !text) {
      return text || '';
    }
    
    return text
      .replace(/&/g, '&amp;')
      .replace(/</g, '&lt;')
      .replace(/>/g, '&gt;')
      .replace(/"/g, '&quot;')
      .replace(/'/g, '&#39;');
  }

  /**
   * 设置选项
   */
  setOptions(options) {
    this.options = { ...this.options, ...options };
    return this;
  }
}

// 导出
if (typeof module !== 'undefined' && module.exports) {
  module.exports = HTMLRenderer;
} else if (typeof window !== 'undefined') {
  window.HTMLRenderer = HTMLRenderer;
}