/**
 * CSS生成器 - 专门负责生成LaTeX文档的CSS样式
 * 职责：根据解析结果和配置生成完整的CSS样式
 * 文件大小控制：< 200行
 */

class CSSGenerator {
  constructor(config = {}) {
    // 加载CSS规则配置
    let cssConfig;
    try {
      cssConfig = require('../config/latex-css-rules.js');
    } catch (e) {
      console.warn('CSS配置加载失败，使用默认配置:', e.message);
      cssConfig = { LATEX_CSS_RULES: {}, CSS_UTILS: {} };
    }
    
    this.cssRules = config.cssRules || (typeof window !== 'undefined' ? 
      window.LATEX_CSS_RULES : cssConfig.LATEX_CSS_RULES) || this.createDefaultCSSRules();
    
    this.cssUtils = config.cssUtils || (typeof window !== 'undefined' ? 
      window.CSS_UTILS : cssConfig.CSS_UTILS) || this.createDefaultCSSUtils();
    
    this.options = {
      includeComments: config.includeComments !== false,
      minify: config.minify === true,
      customRules: config.customRules || {},
      ...config.options
    };
  }

  /**
   * 生成完整的CSS样式
   * @param {Object} parseResult - V3解析结果
   * @returns {string} CSS字符串
   */
  generate(parseResult) {
    const cssBlocks = [];
    
    // 添加注释头
    if (this.options.includeComments) {
      cssBlocks.push(this.generateHeader());
    }

    // 生成基础样式
    cssBlocks.push(this.generateBaseStyles());

    // 根据解析结果生成特定样式
    if (parseResult && parseResult.blocks) {
      cssBlocks.push(this.generateSpecificStyles(parseResult));
    }

    // 添加自定义规则
    if (Object.keys(this.options.customRules).length > 0) {
      cssBlocks.push(this.generateCustomStyles());
    }

    const css = cssBlocks.filter(block => block).join('\n\n');
    
    return this.options.minify ? this.minifyCSS(css) : css;
  }

  /**
   * 生成CSS注释头
   */
  generateHeader() {
    return `/* LaTeX Parser V3 - 自动生成的样式 */
/* 生成时间: ${new Date().toISOString()} */`;
  }

  /**
   * 生成基础样式（所有LaTeX文档都需要的样式）
   */
  generateBaseStyles() {
    const baseClasses = [
      'latex-document',
      'document-title', 
      'document-author',
      'document-date',
      'title-separator',
      'latex-section',
      'latex-subsection', 
      'latex-subsubsection',
      'latex-content'
    ];

    const cssBlocks = baseClasses
      .filter(className => this.cssRules[className])
      .map(className => this.cssUtils.objectToCss(className, this.cssRules[className]));

    return cssBlocks.join('\n\n');
  }

  /**
   * 根据解析结果生成特定样式
   */
  generateSpecificStyles(parseResult) {
    const specificStyles = [];
    const usedClasses = new Set();

    // 分析解析结果中使用的类
    parseResult.blocks.forEach(block => {
      if (block.type) {
        this.collectUsedClasses(block, usedClasses);
      }
    });

    // 为使用的类生成样式
    usedClasses.forEach(className => {
      if (this.cssRules[className] && !this.isBaseClass(className)) {
        specificStyles.push(
          this.cssUtils.objectToCss(className, this.cssRules[className])
        );
      }
    });

    return specificStyles.length > 0 ? 
      `/* 特定内容样式 */\n${specificStyles.join('\n\n')}` : '';
  }

  /**
   * 生成自定义样式
   */
  generateCustomStyles() {
    const customBlocks = Object.entries(this.options.customRules)
      .map(([className, styles]) => this.cssUtils.objectToCss(className, styles));

    return customBlocks.length > 0 ?
      `/* 自定义样式 */\n${customBlocks.join('\n\n')}` : '';
  }

  /**
   * 收集使用的CSS类
   */
  collectUsedClasses(block, usedClasses) {
    // 映射block类型到CSS类
    const typeToClassMap = {
      title: 'document-title',
      author: 'document-author', 
      date: 'document-date',
      section: 'latex-section',
      subsection: 'latex-subsection',
      subsubsection: 'latex-subsubsection',
      paragraph: 'latex-content'
    };

    if (typeToClassMap[block.type]) {
      usedClasses.add(typeToClassMap[block.type]);
    }

    // 检查子块
    if (block.children && Array.isArray(block.children)) {
      block.children.forEach(child => this.collectUsedClasses(child, usedClasses));
    }
  }

  /**
   * 检查是否为基础类
   */
  isBaseClass(className) {
    const baseClasses = [
      'latex-document', 'document-title', 'document-author', 
      'document-date', 'title-separator', 'latex-section',
      'latex-subsection', 'latex-subsubsection', 'latex-content'
    ];
    return baseClasses.includes(className);
  }

  /**
   * CSS压缩
   */
  minifyCSS(css) {
    return css
      .replace(/\/\*[\s\S]*?\*\//g, '') // 移除注释
      .replace(/\s+/g, ' ')             // 压缩空白
      .replace(/;\s*}/g, '}')           // 移除最后一个分号
      .replace(/\s*{\s*/g, '{')         // 压缩花括号
      .replace(/\s*}\s*/g, '}')
      .replace(/\s*;\s*/g, ';')         // 压缩分号
      .trim();
  }

  /**
   * 添加自定义规则
   */
  addCustomRule(className, styles) {
    this.options.customRules[className] = styles;
    return this;
  }

  /**
   * 设置选项
   */
  setOptions(options) {
    this.options = { ...this.options, ...options };
    return this;
  }

  /**
   * 创建默认CSS规则
   */
  createDefaultCSSRules() {
    return {
      'latex-document': {
        maxWidth: '8.5in',
        margin: '0 auto',
        padding: '1in',
        fontFamily: '"Times New Roman", "SimSun", serif',
        fontSize: '12pt',
        lineHeight: '1.6',
        color: '#333',
        background: 'white'
      },
      'document-title': {
        textAlign: 'center',
        fontSize: '18pt',
        fontWeight: 'bold',
        margin: '1.5em 0 1em 0',
        color: '#000'
      },
      'document-author': {
        textAlign: 'center',
        fontSize: '14pt',
        margin: '0.5em 0',
        color: '#333'
      },
      'latex-section': {
        fontSize: '16pt',
        fontWeight: 'bold',
        margin: '2em 0 1em 0',
        color: '#000'
      },
      'latex-content': {
        margin: '0.8em 0',
        textAlign: 'justify',
        textIndent: '2em'
      }
    };
  }

  /**
   * 创建默认CSS工具
   */
  createDefaultCSSUtils() {
    return {
      objectToCss: (className, styleObject) => {
        const cssProperties = Object.entries(styleObject)
          .map(([property, value]) => {
            // 将驼峰命名转换为短横线命名
            const cssProperty = property.replace(/[A-Z]/g, match => `-${match.toLowerCase()}`);
            return `  ${cssProperty}: ${value};`;
          })
          .join('\n');
        
        return `.${className} {\n${cssProperties}\n}`;
      }
    };
  }

  /**
   * 生成响应式样式（可选功能）
   */
  generateResponsiveStyles() {
    return `
/* 响应式样式 */
@media (max-width: 768px) {
  .latex-document {
    padding: 0.5in;
    font-size: 11pt;
  }
  
  .document-title {
    font-size: 16pt;
  }
  
  .latex-section {
    font-size: 14pt;
  }
}

@media print {
  .latex-document {
    padding: 0;
    max-width: none;
    background: transparent;
  }
}`;
  }
}

// 导出
if (typeof module !== 'undefined' && module.exports) {
  module.exports = CSSGenerator;
} else if (typeof window !== 'undefined') {
  window.CSSGenerator = CSSGenerator;
}