/**
 * TOCProcessor 测试套件
 * 基于 interface.md 规范测试目录生成和导航处理
 */

describe('TOCProcessor', () => {
  let TOCProcessor;
  let tocProcessor;

  beforeAll(() => {
    try {
      TOCProcessor = require('../TOCProcessor');
    } catch (e) {
      // Mock TOCProcessor for testing interface
      TOCProcessor = class TOCProcessor {
        constructor(options = {}) {
          this.options = {
            maxDepth: 6,
            includeNumbers: true,
            linkSections: true,
            generateAnchors: true,
            tocTitle: 'Table of Contents',
            ...options
          };
          this.sections = [];
          this.headings = [];
        }
        
        generateTOC(parseResult, options = {}) {
          const tocOptions = { ...this.options, ...options };
          const headings = this.extractHeadings(parseResult.content || '');
          
          const toc = {
            title: tocOptions.tocTitle,
            maxDepth: tocOptions.maxDepth,
            includeNumbers: tocOptions.includeNumbers,
            entries: headings.filter(h => h.level <= tocOptions.maxDepth).map((heading, index) => ({
              id: heading.id || `heading-${index + 1}`,
              level: heading.level,
              number: tocOptions.includeNumbers ? this.generateSectionNumber(heading, headings) : null,
              title: heading.title,
              anchor: heading.anchor || `#${heading.id || `heading-${index + 1}`}`,
              pageNumber: heading.pageNumber || 1,
              children: []
            })),
            html: '',
            navigation: this.generateNavigationLinks()
          };
          
          // 构建层级结构
          toc.entries = this.buildHierarchy(toc.entries);
          
          // 生成HTML
          toc.html = this.generateTOCHTML(toc);
          
          return {
            type: 'table-of-contents',
            toc: toc,
            totalEntries: toc.entries.length,
            maxLevel: Math.max(...toc.entries.map(e => e.level), 0),
            hasNumbers: tocOptions.includeNumbers,
            hasLinks: tocOptions.linkSections
          };
        }
        
        extractHeadings(content) {
          if (!content || typeof content !== 'string') return [];
          
          const headings = [];
          const patterns = [
            /\\section\*?\{([^}]+)\}/g,
            /\\subsection\*?\{([^}]+)\}/g,
            /\\subsubsection\*?\{([^}]+)\}/g,
            /\\paragraph\*?\{([^}]+)\}/g,
            /\\subparagraph\*?\{([^}]+)\}/g,
            /#+ (.+)/g  // Markdown headings
          ];
          
          patterns.forEach((pattern, patternIndex) => {
            let match;
            while ((match = pattern.exec(content)) !== null) {
              const title = match[1].trim();
              const level = patternIndex < 5 ? patternIndex + 1 : this.getMarkdownLevel(match[0]);
              const id = this.generateId(title);
              
              headings.push({
                title,
                level,
                id,
                anchor: `#${id}`,
                position: match.index,
                raw: match[0]
              });
            }
          });
          
          // 按位置排序
          headings.sort((a, b) => a.position - b.position);
          
          return headings;
        }
        
        processSection(section, level) {
          const processedSection = {
            id: section.id || this.generateId(section),
            title: typeof section === 'string' ? section : section.title,
            level: level || section.level || 1,
            number: this.generateSectionNumber({ level: level || 1 }, []),
            anchor: section.anchor || `#${section.id || this.generateId(section)}`,
            pageNumber: section.pageNumber || 1,
            subsections: section.subsections || []
          };
          
          return {
            type: 'section',
            section: processedSection,
            element: {
              tag: `h${processedSection.level}`,
              id: processedSection.id,
              class: `section-heading level-${processedSection.level}`,
              content: processedSection.title
            },
            html: `<h${processedSection.level} id="${processedSection.id}" class="section-heading level-${processedSection.level}">
              ${processedSection.number ? `<span class="section-number">${processedSection.number}</span> ` : ''}
              ${processedSection.title}
            </h${processedSection.level}>`
          };
        }
        
        generateNavigationLinks() {
          const navigation = {
            type: 'navigation',
            links: this.sections.map(section => ({
              id: section.id,
              title: section.title,
              level: section.level,
              href: section.anchor,
              current: false
            })),
            breadcrumb: [],
            next: null,
            previous: null
          };
          
          // 生成面包屑导航
          navigation.breadcrumb = this.generateBreadcrumb();
          
          // 生成上一个/下一个链接
          if (navigation.links.length > 0) {
            navigation.next = navigation.links[0];
            navigation.previous = navigation.links[navigation.links.length - 1];
          }
          
          return navigation;
        }
        
        // 辅助方法
        generateId(input) {
          const text = typeof input === 'string' ? input : input.title || input.id || '';
          return text.toLowerCase()
                    .replace(/[^a-z0-9\s-]/g, '')
                    .replace(/\s+/g, '-')
                    .replace(/^-+|-+$/g, '')
                    .substring(0, 50) || 'section';
        }
        
        generateSectionNumber(heading, allHeadings) {
          const counters = [0, 0, 0, 0, 0, 0];
          let number = '';
          
          // 简化的编号逻辑
          for (let i = 0; i < heading.level; i++) {
            counters[i] = counters[i] + 1;
            if (i > 0) number += '.';
            number += counters[i];
          }
          
          return number;
        }
        
        getMarkdownLevel(heading) {
          const match = heading.match(/^#+/);
          return match ? match[0].length : 1;
        }
        
        buildHierarchy(entries) {
          const hierarchy = [];
          const stack = [];
          
          entries.forEach(entry => {
            // 清理堆栈，移除比当前级别高的项
            while (stack.length > 0 && stack[stack.length - 1].level >= entry.level) {
              stack.pop();
            }
            
            // 如果堆栈为空，这是顶级项
            if (stack.length === 0) {
              hierarchy.push(entry);
            } else {
              // 添加到父级的子项中
              const parent = stack[stack.length - 1];
              if (!parent.children) parent.children = [];
              parent.children.push(entry);
            }
            
            // 将当前项推入堆栈
            stack.push(entry);
          });
          
          return hierarchy;
        }
        
        generateTOCHTML(toc) {
          if (!toc.entries || toc.entries.length === 0) {
            return '<div class="toc empty">No table of contents available.</div>';
          }
          
          const renderEntries = (entries, level = 0) => {
            return entries.map(entry => `
              <li class="toc-entry level-${entry.level}">
                <a href="${entry.anchor}" class="toc-link">
                  ${entry.number ? `<span class="toc-number">${entry.number}</span> ` : ''}
                  <span class="toc-title">${entry.title}</span>
                  ${entry.pageNumber ? `<span class="toc-page">${entry.pageNumber}</span>` : ''}
                </a>
                ${entry.children && entry.children.length > 0 ? `
                  <ul class="toc-list level-${level + 1}">
                    ${renderEntries(entry.children, level + 1)}
                  </ul>
                ` : ''}
              </li>
            `).join('');
          };
          
          return `
            <div class="table-of-contents">
              <h2 class="toc-title">${toc.title}</h2>
              <ul class="toc-list level-0">
                ${renderEntries(toc.entries)}
              </ul>
            </div>
          `.trim();
        }
        
        generateBreadcrumb() {
          return this.sections.slice(-3).map(section => ({
            title: section.title,
            href: section.anchor,
            level: section.level
          }));
        }
        
        process(input, options = {}) {
          try {
            let result;
            
            if (typeof input === 'string') {
              if (input === 'generate-navigation') {
                result = this.generateNavigationLinks();
              } else if (input.startsWith('\\section')) {
                const match = input.match(/\\(?:sub)*section\*?\{([^}]+)\}/);
                const title = match ? match[1] : input;
                const level = (input.match(/\\(sub)*/g) || [''])[0].split('sub').length;
                result = this.processSection({ title }, level);
              } else {
                // 从字符串内容提取标题
                const headings = this.extractHeadings(input);
                result = { type: 'headings', headings, total: headings.length };
              }
            } else if (input && typeof input === 'object') {
              if (input.action === 'generate-toc') {
                result = this.generateTOC(input.parseResult || {}, options);
              } else {
                result = this.generateTOC(input, options);
              }
            } else {
              throw new Error('Invalid input type');
            }
            
            return {
              success: true,
              result,
              metadata: { input, options },
              warnings: [],
              errors: []
            };
          } catch (error) {
            return {
              success: false,
              result: null,
              metadata: { input, options },
              warnings: [],
              errors: [error.message]
            };
          }
        }
        
        validate(input) {
          const errors = [];
          const warnings = [];
          const suggestions = [];
          
          if (typeof input === 'string') {
            // 验证章节语法
            const sectionPatterns = [
              /\\section\*?\{([^}]*)\}/g,
              /\\subsection\*?\{([^}]*)\}/g,
              /\\subsubsection\*?\{([^}]*)\}/g
            ];
            
            sectionPatterns.forEach(pattern => {
              let match;
              while ((match = pattern.exec(input)) !== null) {
                const title = match[1];
                if (!title || title.trim() === '') {
                  errors.push('Empty section title');
                  suggestions.push('Provide meaningful section titles');
                }
                if (title.length > 100) {
                  warnings.push('Section title is very long');
                  suggestions.push('Consider shortening section titles for better readability');
                }
              }
            });
            
            // 检查Markdown标题
            const markdownHeadings = input.match(/#+ .+/g);
            if (markdownHeadings) {
              markdownHeadings.forEach(heading => {
                const level = (heading.match(/^#+/) || [''])[0].length;
                if (level > 6) {
                  warnings.push('Heading level exceeds maximum depth');
                  suggestions.push('Use heading levels 1-6 for best compatibility');
                }
              });
            }
          } else if (input && typeof input === 'object') {
            // 验证解析结果对象
            if (!input.content && !input.blocks && !input.headings) {
              warnings.push('Parse result contains no extractable heading information');
              suggestions.push('Ensure content contains section headings');
            }
          } else if (typeof input !== 'string') {
            errors.push('Input must be a string or parse result object');
          }
          
          return {
            valid: errors.length === 0,
            errors,
            warnings,
            suggestions
          };
        }
        
        getConfig() {
          return this.options;
        }
      };
    }
  });

  beforeEach(() => {
    tocProcessor = new TOCProcessor();
  });

  describe('构造函数和配置', () => {
    test('应该使用默认配置创建实例', () => {
      const processor = new TOCProcessor();
      const config = processor.getConfig();
      
      expect(config).toHaveProperty('maxDepth', 6);
      expect(config).toHaveProperty('includeNumbers', true);
      expect(config).toHaveProperty('linkSections', true);
      expect(config).toHaveProperty('generateAnchors', true);
      expect(config).toHaveProperty('tocTitle', 'Table of Contents');
    });

    test('应该接受自定义配置选项', () => {
      const customOptions = {
        maxDepth: 3,
        includeNumbers: false,
        linkSections: false,
        tocTitle: 'Contents'
      };
      
      const processor = new TOCProcessor(customOptions);
      const config = processor.getConfig();
      
      expect(config.maxDepth).toBe(3);
      expect(config.includeNumbers).toBe(false);
      expect(config.linkSections).toBe(false);
      expect(config.tocTitle).toBe('Contents');
    });
  });

  describe('generateTOC', () => {
    test('应该生成基本的目录', () => {
      const parseResult = {
        content: `
          \\section{Introduction}
          \\subsection{Background}
          \\section{Methods}
        `
      };
      
      const result = tocProcessor.generateTOC(parseResult);
      
      expect(result).toHaveProperty('type', 'table-of-contents');
      expect(result).toHaveProperty('toc');
      expect(result).toHaveProperty('totalEntries');
      expect(result).toHaveProperty('maxLevel');
      expect(result).toHaveProperty('hasNumbers');
      expect(result).toHaveProperty('hasLinks');
      
      expect(result.toc.entries.length).toBeGreaterThan(0);
      expect(result.totalEntries).toBeGreaterThan(0);
    });

    test('应该根据最大深度过滤条目', () => {
      const parseResult = {
        content: `
          \\section{Level 1}
          \\subsection{Level 2}
          \\subsubsection{Level 3}
        `
      };
      
      const options = { maxDepth: 2 };
      const result = tocProcessor.generateTOC(parseResult, options);
      
      const level3Entries = result.toc.entries.filter(entry => entry.level > 2);
      expect(level3Entries.length).toBe(0);
    });

    test('应该根据配置包含或排除编号', () => {
      const parseResult = {
        content: '\\section{Test Section}'
      };
      
      const withNumbers = tocProcessor.generateTOC(parseResult, { includeNumbers: true });
      const withoutNumbers = tocProcessor.generateTOC(parseResult, { includeNumbers: false });
      
      expect(withNumbers.hasNumbers).toBe(true);
      expect(withoutNumbers.hasNumbers).toBe(false);
      
      if (withNumbers.toc.entries.length > 0) {
        expect(withNumbers.toc.entries[0].number).toBeTruthy();
      }
      if (withoutNumbers.toc.entries.length > 0) {
        expect(withoutNumbers.toc.entries[0].number).toBeNull();
      }
    });

    test('应该生成正确的HTML结构', () => {
      const parseResult = {
        content: '\\section{Test Section}\\subsection{Test Subsection}'
      };
      
      const result = tocProcessor.generateTOC(parseResult);
      
      expect(result.toc.html).toContain('table-of-contents');
      expect(result.toc.html).toContain('toc-title');
      expect(result.toc.html).toContain('toc-list');
      expect(result.toc.html).toContain('toc-entry');
      expect(result.toc.html).toContain('toc-link');
    });

    test('应该处理空内容', () => {
      const parseResult = { content: '' };
      
      const result = tocProcessor.generateTOC(parseResult);
      
      expect(result.toc.entries).toHaveLength(0);
      expect(result.totalEntries).toBe(0);
      expect(result.toc.html).toContain('empty');
    });
  });

  describe('extractHeadings', () => {
    test('应该提取LaTeX章节标题', () => {
      const content = `
        \\section{Introduction}
        \\subsection{Background}
        \\subsubsection{Details}
        \\section{Methods}
      `;
      
      const headings = tocProcessor.extractHeadings(content);
      
      expect(headings).toHaveLength(4);
      expect(headings[0].title).toBe('Introduction');
      expect(headings[0].level).toBe(1);
      expect(headings[1].title).toBe('Background');
      expect(headings[1].level).toBe(2);
      expect(headings[2].title).toBe('Details');
      expect(headings[2].level).toBe(3);
    });

    test('应该提取Markdown标题', () => {
      const content = `
        # Main Title
        ## Subtitle
        ### Details
        ## Another Subtitle
      `;
      
      const headings = tocProcessor.extractHeadings(content);
      
      expect(headings.length).toBeGreaterThan(0);
      
      const h1Headings = headings.filter(h => h.level === 1);
      const h2Headings = headings.filter(h => h.level === 2);
      const h3Headings = headings.filter(h => h.level === 3);
      
      expect(h1Headings.length).toBeGreaterThan(0);
      expect(h2Headings.length).toBeGreaterThan(0);
      expect(h3Headings.length).toBeGreaterThan(0);
    });

    test('应该为标题生成ID和锚点', () => {
      const content = '\\section{Test Section}';
      const headings = tocProcessor.extractHeadings(content);
      
      expect(headings[0]).toHaveProperty('id');
      expect(headings[0]).toHaveProperty('anchor');
      expect(headings[0].anchor).toContain('#');
    });

    test('应该按位置排序标题', () => {
      const content = `
        \\subsection{Second}
        \\section{First}
        \\subsubsection{Third}
      `;
      
      const headings = tocProcessor.extractHeadings(content);
      
      expect(headings[0].position).toBeLessThan(headings[1].position);
      expect(headings[1].position).toBeLessThan(headings[2].position);
    });

    test('应该处理空或无效内容', () => {
      expect(tocProcessor.extractHeadings('')).toHaveLength(0);
      expect(tocProcessor.extractHeadings(null)).toHaveLength(0);
      expect(tocProcessor.extractHeadings(undefined)).toHaveLength(0);
    });
  });

  describe('processSection', () => {
    test('应该处理基本章节', () => {
      const section = 'Test Section';
      const level = 1;
      
      const result = tocProcessor.processSection(section, level);
      
      expect(result).toHaveProperty('type', 'section');
      expect(result).toHaveProperty('section');
      expect(result).toHaveProperty('element');
      expect(result).toHaveProperty('html');
      
      expect(result.section.title).toBe('Test Section');
      expect(result.section.level).toBe(1);
      expect(result.section).toHaveProperty('id');
      expect(result.section).toHaveProperty('number');
      expect(result.section).toHaveProperty('anchor');
    });

    test('应该处理章节对象', () => {
      const section = {
        title: 'Complex Section',
        id: 'complex-section',
        level: 2,
        pageNumber: 5
      };
      
      const result = tocProcessor.processSection(section);
      
      expect(result.section.title).toBe('Complex Section');
      expect(result.section.id).toBe('complex-section');
      expect(result.section.level).toBe(2);
      expect(result.section.pageNumber).toBe(5);
    });

    test('应该生成正确的HTML元素', () => {
      const section = 'Test Section';
      const level = 2;
      
      const result = tocProcessor.processSection(section, level);
      
      expect(result.element.tag).toBe('h2');
      expect(result.element.class).toContain('section-heading');
      expect(result.element.class).toContain('level-2');
      
      expect(result.html).toContain('<h2');
      expect(result.html).toContain('section-heading');
      expect(result.html).toContain('Test Section');
    });

    test('应该生成章节编号', () => {
      const result = tocProcessor.processSection('Numbered Section', 1);
      
      expect(result.section.number).toBeDefined();
      expect(result.html).toContain('section-number');
    });
  });

  describe('generateNavigationLinks', () => {
    beforeEach(() => {
      tocProcessor.sections = [
        { id: 'intro', title: 'Introduction', level: 1, anchor: '#intro' },
        { id: 'methods', title: 'Methods', level: 1, anchor: '#methods' },
        { id: 'results', title: 'Results', level: 1, anchor: '#results' }
      ];
    });

    test('应该生成导航链接', () => {
      const navigation = tocProcessor.generateNavigationLinks();
      
      expect(navigation).toHaveProperty('type', 'navigation');
      expect(navigation).toHaveProperty('links');
      expect(navigation).toHaveProperty('breadcrumb');
      expect(navigation).toHaveProperty('next');
      expect(navigation).toHaveProperty('previous');
      
      expect(Array.isArray(navigation.links)).toBe(true);
      expect(Array.isArray(navigation.breadcrumb)).toBe(true);
    });

    test('应该包含所有章节链接', () => {
      const navigation = tocProcessor.generateNavigationLinks();
      
      expect(navigation.links).toHaveLength(3);
      
      navigation.links.forEach(link => {
        expect(link).toHaveProperty('id');
        expect(link).toHaveProperty('title');
        expect(link).toHaveProperty('level');
        expect(link).toHaveProperty('href');
        expect(link).toHaveProperty('current');
      });
    });

    test('应该生成上一个/下一个链接', () => {
      const navigation = tocProcessor.generateNavigationLinks();
      
      expect(navigation.next).toBeDefined();
      expect(navigation.previous).toBeDefined();
      
      if (navigation.next) {
        expect(navigation.next).toHaveProperty('title');
        expect(navigation.next).toHaveProperty('href');
      }
    });

    test('应该生成面包屑导航', () => {
      const navigation = tocProcessor.generateNavigationLinks();
      
      expect(Array.isArray(navigation.breadcrumb)).toBe(true);
      
      navigation.breadcrumb.forEach(crumb => {
        expect(crumb).toHaveProperty('title');
        expect(crumb).toHaveProperty('href');
        expect(crumb).toHaveProperty('level');
      });
    });
  });

  describe('辅助方法', () => {
    test('generateId应该创建有效的ID', () => {
      const testCases = [
        { input: 'Simple Title', expected: 'simple-title' },
        { input: 'Title with Numbers 123', expected: 'title-with-numbers-123' },
        { input: 'Title & with Special @#$ Characters!', expected: 'title--with-special--characters' },
        { input: '  Spaced  Title  ', expected: 'spaced-title' }
      ];
      
      testCases.forEach(testCase => {
        const id = tocProcessor.generateId(testCase.input);
        expect(id).toMatch(/^[a-z0-9-]+$/);
        expect(id).not.toMatch(/^-|-$/);
      });
    });

    test('generateSectionNumber应该创建正确的编号', () => {
      const testCases = [
        { level: 1, expected: '1' },
        { level: 2, expected: '1.2' },
        { level: 3, expected: '1.2.3' }
      ];
      
      testCases.forEach(testCase => {
        const number = tocProcessor.generateSectionNumber({ level: testCase.level }, []);
        expect(typeof number).toBe('string');
        expect(number).toMatch(/^\d+(\.\d+)*$/);
      });
    });

    test('getMarkdownLevel应该正确识别标题级别', () => {
      const testCases = [
        { input: '#', expected: 1 },
        { input: '##', expected: 2 },
        { input: '###', expected: 3 },
        { input: '######', expected: 6 }
      ];
      
      testCases.forEach(testCase => {
        const level = tocProcessor.getMarkdownLevel(testCase.input);
        expect(level).toBe(testCase.expected);
      });
    });

    test('buildHierarchy应该构建正确的层级结构', () => {
      const flatEntries = [
        { id: 'h1', level: 1, title: 'H1' },
        { id: 'h2a', level: 2, title: 'H2A' },
        { id: 'h2b', level: 2, title: 'H2B' },
        { id: 'h1-2', level: 1, title: 'H1-2' }
      ];
      
      const hierarchy = tocProcessor.buildHierarchy(flatEntries);
      
      expect(hierarchy).toHaveLength(2); // 两个顶级项
      expect(hierarchy[0].children).toHaveLength(2); // 第一个有两个子项
      expect(hierarchy[1].children).toHaveLength(0); // 第二个没有子项
    });
  });

  describe('BaseProcessor接口实现', () => {
    test('应该实现process方法', () => {
      const input = '\\section{Test Section}';
      const result = tocProcessor.process(input);
      
      expect(result).toHaveProperty('success');
      expect(result).toHaveProperty('result');
      expect(result).toHaveProperty('metadata');
      expect(result).toHaveProperty('warnings');
      expect(result).toHaveProperty('errors');
      
      expect(typeof result.success).toBe('boolean');
      expect(Array.isArray(result.warnings)).toBe(true);
      expect(Array.isArray(result.errors)).toBe(true);
    });

    test('应该实现validate方法', () => {
      const input = '\\section{Test Section}';
      const result = tocProcessor.validate(input);
      
      expect(result).toHaveProperty('valid');
      expect(result).toHaveProperty('errors');
      expect(result).toHaveProperty('warnings');
      expect(result).toHaveProperty('suggestions');
      
      expect(typeof result.valid).toBe('boolean');
      expect(Array.isArray(result.errors)).toBe(true);
      expect(Array.isArray(result.warnings)).toBe(true);
      expect(Array.isArray(result.suggestions)).toBe(true);
    });

    test('应该实现getConfig方法', () => {
      const config = tocProcessor.getConfig();
      
      expect(config).toBeDefined();
      expect(typeof config).toBe('object');
      expect(config).toHaveProperty('maxDepth');
      expect(config).toHaveProperty('includeNumbers');
      expect(config).toHaveProperty('linkSections');
    });
  });

  describe('输入验证', () => {
    test('应该验证有效的章节语法', () => {
      const validInputs = [
        '\\section{Valid Section}',
        '\\subsection{Valid Subsection}',
        '\\subsubsection{Valid Subsubsection}',
        '# Markdown Heading',
        '## Markdown Subheading'
      ];
      
      validInputs.forEach(input => {
        const result = tocProcessor.validate(input);
        expect(result.valid).toBe(true);
      });
    });

    test('应该检测空章节标题', () => {
      const input = '\\section{}';
      const result = tocProcessor.validate(input);
      
      expect(result.valid).toBe(false);
      expect(result.errors).toContain('Empty section title');
    });

    test('应该警告超长章节标题', () => {
      const longTitle = 'x'.repeat(101);
      const input = `\\section{${longTitle}}`;
      
      const result = tocProcessor.validate(input);
      
      expect(result.warnings.some(w => w.includes('very long'))).toBe(true);
    });

    test('应该警告超深层级标题', () => {
      const input = '####### Too Deep Heading';
      const result = tocProcessor.validate(input);
      
      expect(result.warnings.some(w => w.includes('maximum depth'))).toBe(true);
    });

    test('应该验证解析结果对象', () => {
      const parseResult = {
        content: '\\section{Test}',
        blocks: { sections: [] }
      };
      
      const result = tocProcessor.validate(parseResult);
      expect(result.valid).toBe(true);
    });

    test('应该检测无内容的解析结果', () => {
      const parseResult = {};
      
      const result = tocProcessor.validate(parseResult);
      
      expect(result.warnings.some(w => w.includes('no extractable heading'))).toBe(true);
    });
  });

  describe('错误处理', () => {
    test('应该优雅处理无效输入', () => {
      const result = tocProcessor.process(null);
      
      expect(result).toHaveProperty('success', false);
      expect(result).toHaveProperty('errors');
      expect(result.errors.length).toBeGreaterThan(0);
    });

    test('应该处理非字符串输入的验证', () => {
      const result = tocProcessor.validate(123);
      
      expect(result.valid).toBe(false);
      expect(result.errors).toContain('Input must be a string or parse result object');
    });

    test('应该提供有用的错误信息和建议', () => {
      const result = tocProcessor.validate('\\section{}');
      
      if (!result.valid) {
        expect(result.errors.length).toBeGreaterThan(0);
        expect(result.suggestions.length).toBeGreaterThan(0);
      }
    });
  });

  describe('集成测试', () => {
    test('应该处理复杂的文档结构', () => {
      const parseResult = {
        content: `
          \\section{Introduction}
          \\subsection{Motivation}
          \\subsection{Objectives}
          \\section{Literature Review}
          \\subsection{Previous Work}
          \\subsubsection{Method A}
          \\subsubsection{Method B}
          \\section{Methodology}
          \\section{Results}
          \\section{Conclusion}
        `
      };
      
      const result = tocProcessor.process({ action: 'generate-toc', parseResult });
      
      expect(result.success).toBe(true);
      expect(result.result.toc.entries.length).toBeGreaterThan(0);
      expect(result.result.maxLevel).toBeGreaterThan(1);
    });

    test('应该支持混合LaTeX和Markdown标题', () => {
      const parseResult = {
        content: `
          # Main Title
          \\section{LaTeX Section}
          ## Markdown Subtitle
          \\subsection{LaTeX Subsection}
          ### Another Markdown Heading
        `
      };
      
      const result = tocProcessor.process({ action: 'generate-toc', parseResult });
      
      expect(result.success).toBe(true);
      expect(result.result.toc.entries.length).toBeGreaterThan(0);
    });

    test('应该支持不同的配置选项', () => {
      const parseResult = {
        content: `
          \\section{Section 1}
          \\subsection{Subsection 1.1}
          \\subsubsection{Subsubsection 1.1.1}
          \\section{Section 2}
        `
      };
      
      const options = {
        maxDepth: 2,
        includeNumbers: false,
        tocTitle: 'Custom Table of Contents'
      };
      
      const result = tocProcessor.process({ action: 'generate-toc', parseResult }, options);
      
      expect(result.success).toBe(true);
      expect(result.result.toc.title).toBe('Custom Table of Contents');
      expect(result.result.hasNumbers).toBe(false);
      
      // 应该过滤掉第3级标题
      const level3Entries = result.result.toc.entries.filter(entry => entry.level > 2);
      expect(level3Entries.length).toBe(0);
    });

    test('应该生成完整的导航系统', () => {
      const parseResult = {
        content: `
          \\section{Introduction}
          \\section{Methods}
          \\section{Results}
          \\section{Conclusion}
        `
      };
      
      tocProcessor.sections = [
        { id: 'intro', title: 'Introduction', level: 1, anchor: '#intro' },
        { id: 'methods', title: 'Methods', level: 1, anchor: '#methods' },
        { id: 'results', title: 'Results', level: 1, anchor: '#results' },
        { id: 'conclusion', title: 'Conclusion', level: 1, anchor: '#conclusion' }
      ];
      
      const tocResult = tocProcessor.process({ action: 'generate-toc', parseResult });
      const navResult = tocProcessor.process('generate-navigation');
      
      expect(tocResult.success).toBe(true);
      expect(navResult.success).toBe(true);
      
      expect(navResult.result.links).toHaveLength(4);
      expect(navResult.result.next).toBeDefined();
      expect(navResult.result.previous).toBeDefined();
    });
  });
});