/**
 * ReferenceProcessor 测试套件
 * 基于 interface.md 规范测试交叉引用和链接处理
 */

describe('ReferenceProcessor', () => {
  let ReferenceProcessor;
  let referenceProcessor;

  beforeAll(() => {
    try {
      ReferenceProcessor = require('../ReferenceProcessor');
    } catch (e) {
      // Mock ReferenceProcessor for testing interface
      ReferenceProcessor = class ReferenceProcessor {
        constructor(options = {}) {
          this.options = {
            citationStyle: 'numeric',
            linkStyle: 'inline',
            generateLinks: true,
            ...options
          };
          this.references = new Map();
          this.labels = new Map();
          this.citations = new Map();
        }
        
        processRef(refId, type = 'general') {
          const ref = this.labels.get(refId);
          
          if (!ref) {
            return {
              type: 'reference',
              refId,
              targetType: type,
              found: false,
              element: {
                id: refId,
                text: `[${refId}]`,
                href: `#${refId}`,
                class: 'latex-ref-missing'
              },
              html: `<span class="latex-ref-missing" data-ref="${refId}">[${refId}]</span>`
            };
          }
          
          return {
            type: 'reference',
            refId,
            targetType: type,
            found: true,
            element: {
              id: refId,
              text: ref.displayText || refId,
              href: `#${refId}`,
              class: `latex-ref latex-ref-${type}`
            },
            html: `<a href="#${refId}" class="latex-ref latex-ref-${type}">${ref.displayText || refId}</a>`
          };
        }
        
        processCite(citeKey, style = null) {
          const citationStyle = style || this.options.citationStyle;
          const citation = this.citations.get(citeKey);
          
          if (!citation) {
            return {
              type: 'citation',
              citeKey,
              style: citationStyle,
              found: false,
              element: {
                id: citeKey,
                text: `[${citeKey}]`,
                class: 'latex-cite-missing'
              },
              html: `<span class="latex-cite-missing" data-cite="${citeKey}">[${citeKey}]</span>`
            };
          }
          
          let displayText;
          switch (citationStyle) {
            case 'numeric':
              displayText = `[${citation.number || '?'}]`;
              break;
            case 'author-year':
              displayText = `(${citation.author || 'Unknown'}, ${citation.year || 'n.d.'})`;
              break;
            case 'author':
              displayText = citation.author || 'Unknown';
              break;
            default:
              displayText = `[${citeKey}]`;
          }
          
          return {
            type: 'citation',
            citeKey,
            style: citationStyle,
            found: true,
            element: {
              id: citeKey,
              text: displayText,
              class: `latex-cite latex-cite-${citationStyle}`
            },
            html: `<span class="latex-cite latex-cite-${citationStyle}" data-cite="${citeKey}">${displayText}</span>`
          };
        }
        
        processLabel(labelId, context = {}) {
          const label = {
            id: labelId,
            type: context.type || 'general',
            section: context.section || null,
            number: context.number || null,
            displayText: context.displayText || labelId
          };
          
          this.labels.set(labelId, label);
          
          return {
            type: 'label',
            labelId,
            context,
            element: {
              id: labelId,
              type: label.type,
              anchor: `<span id="${labelId}" class="latex-label"></span>`
            },
            html: `<span id="${labelId}" class="latex-label" data-type="${label.type}"></span>`
          };
        }
        
        generateReferenceList() {
          const refs = Array.from(this.citations.values());
          
          return {
            type: 'reference-list',
            total: refs.length,
            entries: refs.map((ref, index) => ({
              id: ref.key,
              number: index + 1,
              text: this.formatReferenceEntry(ref),
              html: `<div class="latex-ref-entry" id="ref-${ref.key}">
                [${index + 1}] ${this.formatReferenceEntry(ref)}
              </div>`
            })),
            html: refs.length > 0 ? `
              <div class="latex-reference-list">
                <h3>References</h3>
                ${refs.map((ref, index) => `
                  <div class="latex-ref-entry" id="ref-${ref.key}">
                    [${index + 1}] ${this.formatReferenceEntry(ref)}
                  </div>
                `).join('')}
              </div>
            ` : '<div class="latex-reference-list empty">No references found.</div>'
          };
        }
        
        formatReferenceEntry(ref) {
          const parts = [];
          if (ref.author) parts.push(ref.author);
          if (ref.title) parts.push(`"${ref.title}"`);
          if (ref.journal) parts.push(ref.journal);
          if (ref.year) parts.push(`(${ref.year})`);
          return parts.join(', ') || ref.key;
        }
        
        // 添加引用数据的方法
        addCitation(key, data) {
          this.citations.set(key, { key, ...data });
        }
        
        process(input, options = {}) {
          try {
            let result;
            
            if (typeof input === 'string') {
              if (input.startsWith('\\ref{')) {
                const refMatch = input.match(/\\ref\{(.+?)\}/);
                const refId = refMatch ? refMatch[1] : input;
                result = this.processRef(refId, options.type);
              } else if (input.startsWith('\\cite{')) {
                const citeMatch = input.match(/\\cite\{(.+?)\}/);
                const citeKey = citeMatch ? citeMatch[1] : input;
                result = this.processCite(citeKey, options.style);
              } else if (input.startsWith('\\label{')) {
                const labelMatch = input.match(/\\label\{(.+?)\}/);
                const labelId = labelMatch ? labelMatch[1] : input;
                result = this.processLabel(labelId, options.context);
              } else {
                // 默认作为引用处理
                result = this.processRef(input, options.type);
              }
            } else if (input && typeof input === 'object') {
              // 处理引用列表生成
              result = this.generateReferenceList();
            } else {
              throw new Error('Invalid input type');
            }
            
            return {
              success: true,
              result,
              metadata: { input, options },
              warnings: [],
              errors: []
            };
          } catch (error) {
            return {
              success: false,
              result: null,
              metadata: { input, options },
              warnings: [],
              errors: [error.message]
            };
          }
        }
        
        validate(input) {
          const errors = [];
          const warnings = [];
          const suggestions = [];
          
          if (typeof input !== 'string') {
            errors.push('Input must be a string');
            return { valid: false, errors, warnings, suggestions };
          }
          
          // 验证引用语法
          if (input.includes('\\ref{')) {
            const refMatches = input.match(/\\ref\{(.+?)\}/g);
            if (refMatches) {
              refMatches.forEach(match => {
                const refId = match.match(/\\ref\{(.+?)\}/)[1];
                if (!refId) {
                  errors.push('Empty reference ID');
                } else if (!this.labels.has(refId)) {
                  warnings.push(`Reference '${refId}' not found`);
                  suggestions.push(`Define label '${refId}' with \\label{${refId}}`);
                }
              });
            }
          }
          
          // 验证引用语法
          if (input.includes('\\cite{')) {
            const citeMatches = input.match(/\\cite\{(.+?)\}/g);
            if (citeMatches) {
              citeMatches.forEach(match => {
                const citeKey = match.match(/\\cite\{(.+?)\}/)[1];
                if (!citeKey) {
                  errors.push('Empty citation key');
                } else if (!this.citations.has(citeKey)) {
                  warnings.push(`Citation '${citeKey}' not found`);
                  suggestions.push(`Add bibliography entry for '${citeKey}'`);
                }
              });
            }
          }
          
          // 验证标签语法
          if (input.includes('\\label{')) {
            const labelMatches = input.match(/\\label\{(.+?)\}/g);
            if (labelMatches) {
              labelMatches.forEach(match => {
                const labelId = match.match(/\\label\{(.+?)\}/)[1];
                if (!labelId) {
                  errors.push('Empty label ID');
                } else if (this.labels.has(labelId)) {
                  warnings.push(`Label '${labelId}' already exists`);
                  suggestions.push(`Use unique label ID for '${labelId}'`);
                }
              });
            }
          }
          
          return {
            valid: errors.length === 0,
            errors,
            warnings,
            suggestions
          };
        }
        
        getConfig() {
          return this.options;
        }
      };
    }
  });

  beforeEach(() => {
    referenceProcessor = new ReferenceProcessor();
    
    // 添加一些测试数据
    referenceProcessor.addCitation('smith2020', {
      author: 'Smith, J.',
      title: 'A Study of LaTeX',
      journal: 'TeX Journal',
      year: '2020'
    });
    
    referenceProcessor.processLabel('fig:example', {
      type: 'figure',
      number: '1',
      displayText: 'Figure 1'
    });
  });

  describe('构造函数和配置', () => {
    test('应该使用默认配置创建实例', () => {
      const processor = new ReferenceProcessor();
      const config = processor.getConfig();
      
      expect(config).toHaveProperty('citationStyle', 'numeric');
      expect(config).toHaveProperty('linkStyle', 'inline');
      expect(config).toHaveProperty('generateLinks', true);
    });

    test('应该接受自定义配置选项', () => {
      const customOptions = {
        citationStyle: 'author-year',
        linkStyle: 'footnote',
        generateLinks: false
      };
      
      const processor = new ReferenceProcessor(customOptions);
      const config = processor.getConfig();
      
      expect(config.citationStyle).toBe('author-year');
      expect(config.linkStyle).toBe('footnote');
      expect(config.generateLinks).toBe(false);
    });
  });

  describe('processRef', () => {
    test('应该处理存在的引用', () => {
      const result = referenceProcessor.processRef('fig:example', 'figure');
      
      expect(result).toHaveProperty('type', 'reference');
      expect(result).toHaveProperty('refId', 'fig:example');
      expect(result).toHaveProperty('targetType', 'figure');
      expect(result).toHaveProperty('found', true);
      expect(result).toHaveProperty('element');
      expect(result).toHaveProperty('html');
      
      expect(result.element).toHaveProperty('id');
      expect(result.element).toHaveProperty('text');
      expect(result.element).toHaveProperty('href');
      expect(result.element).toHaveProperty('class');
    });

    test('应该处理不存在的引用', () => {
      const result = referenceProcessor.processRef('nonexistent', 'figure');
      
      expect(result).toHaveProperty('type', 'reference');
      expect(result).toHaveProperty('refId', 'nonexistent');
      expect(result).toHaveProperty('found', false);
      
      expect(result.element.class).toContain('missing');
      expect(result.html).toContain('latex-ref-missing');
    });

    test('应该根据类型生成正确的CSS类', () => {
      const figureRef = referenceProcessor.processRef('fig:example', 'figure');
      const tableRef = referenceProcessor.processRef('fig:example', 'table');
      
      expect(figureRef.element.class).toContain('latex-ref-figure');
      expect(tableRef.element.class).toContain('latex-ref-table');
    });

    test('应该生成正确的链接', () => {
      const result = referenceProcessor.processRef('fig:example', 'figure');
      
      expect(result.element.href).toBe('#fig:example');
      expect(result.html).toContain('href="#fig:example"');
    });
  });

  describe('processCite', () => {
    test('应该处理存在的引用', () => {
      const result = referenceProcessor.processCite('smith2020');
      
      expect(result).toHaveProperty('type', 'citation');
      expect(result).toHaveProperty('citeKey', 'smith2020');
      expect(result).toHaveProperty('found', true);
      expect(result).toHaveProperty('element');
      expect(result).toHaveProperty('html');
      
      expect(result.element).toHaveProperty('id');
      expect(result.element).toHaveProperty('text');
      expect(result.element).toHaveProperty('class');
    });

    test('应该处理不存在的引用', () => {
      const result = referenceProcessor.processCite('unknown2021');
      
      expect(result).toHaveProperty('citeKey', 'unknown2021');
      expect(result).toHaveProperty('found', false);
      
      expect(result.element.class).toContain('missing');
      expect(result.html).toContain('latex-cite-missing');
    });

    test('应该支持不同的引用样式', () => {
      const styles = ['numeric', 'author-year', 'author'];
      
      styles.forEach(style => {
        const result = referenceProcessor.processCite('smith2020', style);
        
        expect(result.style).toBe(style);
        expect(result.element.class).toContain(`latex-cite-${style}`);
        expect(result.element.text).toBeDefined();
      });
    });

    test('应该生成正确的数字引用格式', () => {
      const result = referenceProcessor.processCite('smith2020', 'numeric');
      
      expect(result.element.text).toMatch(/^\[\d*\]$/);
    });

    test('应该生成正确的作者-年份格式', () => {
      const result = referenceProcessor.processCite('smith2020', 'author-year');
      
      expect(result.element.text).toContain('Smith, J.');
      expect(result.element.text).toContain('2020');
    });
  });

  describe('processLabel', () => {
    test('应该处理基本标签', () => {
      const result = referenceProcessor.processLabel('sec:intro');
      
      expect(result).toHaveProperty('type', 'label');
      expect(result).toHaveProperty('labelId', 'sec:intro');
      expect(result).toHaveProperty('element');
      expect(result).toHaveProperty('html');
      
      expect(result.element).toHaveProperty('id', 'sec:intro');
      expect(result.element).toHaveProperty('type');
      expect(result.element).toHaveProperty('anchor');
    });

    test('应该处理带上下文的标签', () => {
      const context = {
        type: 'section',
        section: 'Introduction',
        number: '1',
        displayText: 'Section 1'
      };
      
      const result = referenceProcessor.processLabel('sec:intro', context);
      
      expect(result.context).toEqual(context);
      expect(result.element.type).toBe('section');
    });

    test('应该生成正确的锚点HTML', () => {
      const result = referenceProcessor.processLabel('eq:pythagoras');
      
      expect(result.element.anchor).toContain('id="eq:pythagoras"');
      expect(result.html).toContain('id="eq:pythagoras"');
      expect(result.html).toContain('class="latex-label"');
    });

    test('应该在内部存储标签信息', () => {
      const labelId = 'test:label';
      referenceProcessor.processLabel(labelId, { type: 'test' });
      
      // 测试标签是否可以被引用
      const refResult = referenceProcessor.processRef(labelId);
      expect(refResult.found).toBe(true);
    });
  });

  describe('generateReferenceList', () => {
    test('应该生成引用列表', () => {
      const result = referenceProcessor.generateReferenceList();
      
      expect(result).toHaveProperty('type', 'reference-list');
      expect(result).toHaveProperty('total');
      expect(result).toHaveProperty('entries');
      expect(result).toHaveProperty('html');
      
      expect(Array.isArray(result.entries)).toBe(true);
      expect(typeof result.total).toBe('number');
    });

    test('应该包含正确的引用条目', () => {
      const result = referenceProcessor.generateReferenceList();
      
      expect(result.total).toBeGreaterThan(0);
      
      if (result.entries.length > 0) {
        const entry = result.entries[0];
        expect(entry).toHaveProperty('id');
        expect(entry).toHaveProperty('number');
        expect(entry).toHaveProperty('text');
        expect(entry).toHaveProperty('html');
      }
    });

    test('应该正确格式化引用条目', () => {
      const result = referenceProcessor.generateReferenceList();
      
      const smithEntry = result.entries.find(entry => entry.id === 'smith2020');
      if (smithEntry) {
        expect(smithEntry.text).toContain('Smith, J.');
        expect(smithEntry.text).toContain('A Study of LaTeX');
        expect(smithEntry.text).toContain('2020');
      }
    });

    test('应该生成正确的HTML结构', () => {
      const result = referenceProcessor.generateReferenceList();
      
      if (result.total > 0) {
        expect(result.html).toContain('class="latex-reference-list"');
        expect(result.html).toContain('<h3>References</h3>');
        expect(result.html).toContain('class="latex-ref-entry"');
      } else {
        expect(result.html).toContain('empty');
      }
    });
  });

  describe('BaseProcessor接口实现', () => {
    test('应该实现process方法', () => {
      const input = '\\ref{fig:example}';
      const result = referenceProcessor.process(input);
      
      expect(result).toHaveProperty('success');
      expect(result).toHaveProperty('result');
      expect(result).toHaveProperty('metadata');
      expect(result).toHaveProperty('warnings');
      expect(result).toHaveProperty('errors');
      
      expect(typeof result.success).toBe('boolean');
      expect(Array.isArray(result.warnings)).toBe(true);
      expect(Array.isArray(result.errors)).toBe(true);
    });

    test('应该实现validate方法', () => {
      const input = '\\ref{fig:example}';
      const result = referenceProcessor.validate(input);
      
      expect(result).toHaveProperty('valid');
      expect(result).toHaveProperty('errors');
      expect(result).toHaveProperty('warnings');
      expect(result).toHaveProperty('suggestions');
      
      expect(typeof result.valid).toBe('boolean');
      expect(Array.isArray(result.errors)).toBe(true);
      expect(Array.isArray(result.warnings)).toBe(true);
      expect(Array.isArray(result.suggestions)).toBe(true);
    });

    test('应该实现getConfig方法', () => {
      const config = referenceProcessor.getConfig();
      
      expect(config).toBeDefined();
      expect(typeof config).toBe('object');
      expect(config).toHaveProperty('citationStyle');
      expect(config).toHaveProperty('linkStyle');
      expect(config).toHaveProperty('generateLinks');
    });
  });

  describe('输入验证', () => {
    test('应该验证有效的引用语法', () => {
      const validInputs = [
        '\\ref{fig:1}',
        '\\cite{smith2020}',
        '\\label{sec:intro}',
        'text with \\ref{fig:1} reference',
        'multiple \\cite{a2020} and \\cite{b2021} citations'
      ];
      
      validInputs.forEach(input => {
        const result = referenceProcessor.validate(input);
        expect(result).toHaveProperty('valid');
      });
    });

    test('应该检测缺失的引用', () => {
      const input = '\\ref{nonexistent}';
      const result = referenceProcessor.validate(input);
      
      expect(result.warnings.length).toBeGreaterThan(0);
      expect(result.suggestions.length).toBeGreaterThan(0);
    });

    test('应该检测空的引用ID', () => {
      const input = '\\ref{}';
      const result = referenceProcessor.validate(input);
      
      expect(result.valid).toBe(false);
      expect(result.errors).toContain('Empty reference ID');
    });

    test('应该检测重复的标签', () => {
      // 先创建一个标签
      referenceProcessor.processLabel('duplicate', { type: 'test' });
      
      // 验证包含相同标签的输入
      const input = '\\label{duplicate}';
      const result = referenceProcessor.validate(input);
      
      expect(result.warnings.some(w => w.includes('already exists'))).toBe(true);
    });
  });

  describe('错误处理', () => {
    test('应该优雅处理无效输入', () => {
      const result = referenceProcessor.process(null);
      
      expect(result).toHaveProperty('success', false);
      expect(result).toHaveProperty('errors');
      expect(result.errors.length).toBeGreaterThan(0);
    });

    test('应该处理非字符串输入的验证', () => {
      const result = referenceProcessor.validate(123);
      
      expect(result.valid).toBe(false);
      expect(result.errors).toContain('Input must be a string');
    });

    test('应该提供有用的错误信息', () => {
      const result = referenceProcessor.validate('\\ref{}');
      
      if (!result.valid) {
        expect(result.errors.length).toBeGreaterThan(0);
        expect(result.errors[0]).toContain('Empty reference ID');
      }
    });
  });

  describe('集成测试', () => {
    test('应该处理复杂的引用文档', () => {
      const complexInput = `
        See \\ref{fig:results} for details. 
        According to \\cite{smith2020} and \\cite{jones2021}, 
        the method described in Section \\ref{sec:method} is effective.
      `;
      
      const result = referenceProcessor.process(complexInput);
      expect(result.success).toBeTruthy();
    });

    test('应该支持链式引用处理', () => {
      // 首先处理标签
      referenceProcessor.processLabel('sec:method', {
        type: 'section',
        displayText: 'Section 2'
      });
      
      // 然后处理引用
      const refResult = referenceProcessor.processRef('sec:method', 'section');
      expect(refResult.found).toBe(true);
      
      // 最后生成引用列表
      const listResult = referenceProcessor.generateReferenceList();
      expect(listResult.total).toBeGreaterThan(0);
    });

    test('应该保持引用和标签的关联', () => {
      const labelId = 'eq:important';
      
      // 创建标签
      referenceProcessor.processLabel(labelId, {
        type: 'equation',
        displayText: 'Equation 1'
      });
      
      // 引用该标签
      const refResult = referenceProcessor.processRef(labelId, 'equation');
      
      expect(refResult.found).toBe(true);
      expect(refResult.element.text).toBe('Equation 1');
    });
  });
});