# Processors 模块测试套件

基于 `interface.md` 规范为 Processors 模块创建的完整测试套件。

## 测试文件概览

| 测试文件 | 大小 | 描述 |
|---------|------|------|
| `MathProcessor.test.js` | 9.1KB | 数学公式处理和KaTeX集成测试 |
| `FigureProcessor.test.js` | 16.8KB | 图片和浮动环境处理测试 |
| `ReferenceProcessor.test.js` | 22.9KB | 交叉引用和链接处理测试 |
| `FootnoteProcessor.test.js` | 26.3KB | 脚注系统处理测试 |
| `MetadataProcessor.test.js` | 32.9KB | 文档元数据提取和处理测试 |
| `TOCProcessor.test.js` | 31.7KB | 目录生成和导航处理测试 |
| `BibliographyProcessor.test.js` | 36.0KB | 参考文献处理测试 |
| `AdvancedEnvironmentProcessor.test.js` | 32.8KB | 复杂LaTeX环境处理测试 |
| `DocumentClassProcessor.test.js` | 39.9KB | 文档类别和样式处理测试 |
| `index.test.js` | 8.0KB | 测试套件概览和接口一致性验证 |

**总计：** 10个测试文件，约 256KB 的测试代码

## 测试覆盖范围

### 核心功能测试

每个处理器都包含以下核心测试类别：

1. **构造函数和配置** - 验证初始化和配置选项
2. **主要处理方法** - 测试核心业务逻辑
3. **输入验证** - 验证输入格式和内容检查
4. **错误处理** - 测试异常情况和错误恢复
5. **BaseProcessor接口实现** - 确保接口一致性

### 接口一致性

所有处理器都实现统一的基础接口：

```javascript
class BaseProcessor {
  process(input, options = {}) // 返回 ProcessResult
  validate(input) // 返回 ValidationResult  
  getConfig() // 返回配置对象
}
```

### 结果格式标准化

#### ProcessResult 格式
```javascript
{
  success: Boolean,
  result: Any,
  metadata: Object,
  warnings: Array,
  errors: Array
}
```

#### ValidationResult 格式
```javascript
{
  valid: Boolean,
  errors: Array,
  warnings: Array,
  suggestions: Array
}
```

## 测试特色功能

### Mock 实现

每个测试文件都包含完整的 Mock 实现，确保：
- 完整的接口覆盖
- 现实的行为模拟
- 独立运行能力
- 错误场景模拟

### 测试数据

包含丰富的测试数据：
- 有效的LaTeX输入样本
- 无效输入和边界条件
- 复杂的嵌套结构
- 真实世界的使用场景

### 集成测试

每个处理器都包含集成测试：
- 处理器链式调用
- 复杂工作流验证
- 跨处理器数据传递
- 性能和稳定性测试

## 运行测试

### 环境要求

- Node.js 14+
- Jest 测试框架
- 支持 ES6+ 语法

### 运行单个测试文件

```bash
# 运行特定处理器测试
npx jest MathProcessor.test.js
npx jest FigureProcessor.test.js
npx jest ReferenceProcessor.test.js
# ... 其他处理器
```

### 运行所有测试

```bash
# 运行整个测试套件
npx jest processors/tests/

# 带覆盖率报告
npx jest processors/tests/ --coverage

# 详细输出
npx jest processors/tests/ --verbose
```

### 运行特定测试组

```bash
# 运行接口一致性测试
npx jest processors/tests/index.test.js

# 运行核心功能测试
npx jest --testNamePattern="BaseProcessor接口实现"

# 运行集成测试
npx jest --testNamePattern="集成测试"
```

## 测试质量指标

### 覆盖率目标

- **语句覆盖率**: ≥ 90%
- **分支覆盖率**: ≥ 85%  
- **函数覆盖率**: ≥ 95%
- **行覆盖率**: ≥ 90%

### 测试数量统计

| 处理器 | 主要测试组 | 预估测试数量 |
|-------|-----------|-------------|
| MathProcessor | 8组 | ~40个测试 |
| FigureProcessor | 9组 | ~45个测试 |
| ReferenceProcessor | 8组 | ~50个测试 |
| FootnoteProcessor | 10组 | ~55个测试 |
| MetadataProcessor | 9组 | ~60个测试 |
| TOCProcessor | 10组 | ~65个测试 |
| BibliographyProcessor | 11组 | ~70个测试 |
| AdvancedEnvironmentProcessor | 10组 | ~65个测试 |
| DocumentClassProcessor | 12组 | ~75个测试 |
| 接口一致性测试 | 7组 | ~35个测试 |

**预计总测试数量**: ~560个测试用例

## 维护指南

### 添加新测试

1. 遵循现有的测试结构和命名约定
2. 确保包含所有核心测试类别
3. 实现完整的Mock对象
4. 添加边界条件和错误处理测试

### 更新测试

1. 当接口发生变化时，更新相应的测试
2. 保持Mock实现与真实接口同步
3. 验证向后兼容性
4. 更新文档和注释

### 测试最佳实践

1. **描述性测试名称** - 清楚说明测试目的
2. **独立测试用例** - 每个测试应该独立运行
3. **有意义的断言** - 使用具体的期望值
4. **适当的测试数据** - 使用真实的LaTeX示例
5. **完整的错误测试** - 覆盖所有异常路径

## 故障排除

### 常见问题

1. **Mock对象未找到**
   - 确保测试文件中的Mock实现完整
   - 检查require路径是否正确

2. **测试超时**
   - 检查是否有无限循环
   - 增加Jest超时时间配置

3. **断言失败**
   - 检查期望值是否与实际实现匹配
   - 验证Mock对象行为是否现实

### 调试技巧

1. 使用 `console.log` 输出中间结果
2. 运行单个测试进行隔离调试
3. 检查测试数据的格式和内容
4. 验证Mock对象的方法调用

## 贡献指南

### 提交测试变更

1. 确保所有测试通过
2. 验证覆盖率没有降低
3. 添加必要的文档更新
4. 遵循代码风格约定

### 代码审查要点

1. 测试覆盖的完整性
2. Mock实现的准确性
3. 测试用例的有效性
4. 文档和注释的清晰度

---

此测试套件基于 `interface.md` 规范设计，确保 Processors 模块的所有公共接口都经过全面测试和验证。通过这些测试，我们可以保证处理器的稳定性、可靠性和接口一致性。