/**
 * FootnoteProcessor 测试套件
 * 基于 interface.md 规范测试脚注系统处理
 */

describe('FootnoteProcessor', () => {
  let FootnoteProcessor;
  let footnoteProcessor;

  beforeAll(() => {
    try {
      FootnoteProcessor = require('../FootnoteProcessor');
    } catch (e) {
      // Mock FootnoteProcessor for testing interface
      FootnoteProcessor = class FootnoteProcessor {
        constructor(options = {}) {
          this.options = {
            numberStyle: 'arabic',
            position: 'bottom',
            separator: '---',
            backLinks: true,
            ...options
          };
          this.footnotes = new Map();
          this.counter = 0;
        }
        
        processFootnote(content, id = null) {
          const footnoteId = id || `footnote-${++this.counter}`;
          const footnoteNumber = this.footnotes.size + 1;
          
          const footnote = {
            id: footnoteId,
            content,
            number: footnoteNumber,
            marker: this.formatNumber(footnoteNumber),
            backRef: `${footnoteId}-ref`
          };
          
          this.footnotes.set(footnoteId, footnote);
          
          return {
            type: 'footnote',
            id: footnoteId,
            number: footnoteNumber,
            content,
            marker: footnote.marker,
            markerHtml: `<sup><a href="#${footnoteId}" id="${footnote.backRef}" class="footnote-ref">${footnote.marker}</a></sup>`,
            footnoteHtml: `<div class="footnote" id="${footnoteId}">
              <span class="footnote-number">${footnote.marker}</span>
              <span class="footnote-content">${content}</span>
              ${this.options.backLinks ? `<a href="#${footnote.backRef}" class="footnote-backlink">↩</a>` : ''}
            </div>`,
            element: footnote
          };
        }
        
        generateFootnoteList() {
          const footnotes = Array.from(this.footnotes.values());
          
          return {
            type: 'footnote-list',
            total: footnotes.length,
            footnotes: footnotes.map(fn => ({
              id: fn.id,
              number: fn.number,
              marker: fn.marker,
              content: fn.content,
              html: `<div class="footnote-item" id="${fn.id}">
                <span class="footnote-number">${fn.marker}.</span>
                <span class="footnote-content">${fn.content}</span>
                ${this.options.backLinks ? `<a href="#${fn.backRef}" class="footnote-backlink">↩</a>` : ''}
              </div>`
            })),
            html: footnotes.length > 0 ? `
              <div class="footnotes">
                <hr class="footnotes-separator" />
                <ol class="footnote-list">
                  ${footnotes.map(fn => `
                    <li class="footnote-item" id="${fn.id}">
                      ${fn.content}
                      ${this.options.backLinks ? `<a href="#${fn.backRef}" class="footnote-backlink">↩</a>` : ''}
                    </li>
                  `).join('')}
                </ol>
              </div>
            ` : '<div class="footnotes empty">No footnotes found.</div>'
          };
        }
        
        linkFootnoteMarkers() {
          const footnotes = Array.from(this.footnotes.values());
          let linkedCount = 0;
          
          footnotes.forEach(footnote => {
            // 模拟链接标记到脚注的过程
            if (footnote.marker && footnote.id) {
              linkedCount++;
            }
          });
          
          return {
            linked: linkedCount,
            total: footnotes.length,
            success: linkedCount === footnotes.length,
            unlinked: footnotes.filter(fn => !fn.marker || !fn.id).map(fn => fn.id)
          };
        }
        
        processFootnoteRef(refId) {
          const footnote = this.footnotes.get(refId);
          
          if (!footnote) {
            return {
              type: 'footnote-reference',
              refId,
              found: false,
              element: {
                id: refId,
                text: `[${refId}]`,
                class: 'footnote-ref-missing'
              },
              html: `<span class="footnote-ref-missing" data-ref="${refId}">[${refId}]</span>`
            };
          }
          
          return {
            type: 'footnote-reference',
            refId,
            found: true,
            element: {
              id: refId,
              text: footnote.marker,
              href: `#${footnote.id}`,
              backRef: footnote.backRef,
              class: 'footnote-ref'
            },
            html: `<sup><a href="#${footnote.id}" id="${footnote.backRef}" class="footnote-ref">${footnote.marker}</a></sup>`
          };
        }
        
        formatNumber(number) {
          switch (this.options.numberStyle) {
            case 'roman':
              return this.toRoman(number);
            case 'alpha':
              return this.toAlpha(number);
            case 'symbol':
              return this.toSymbol(number);
            default:
              return number.toString();
          }
        }
        
        toRoman(num) {
          const values = [1000, 900, 500, 400, 100, 90, 50, 40, 10, 9, 5, 4, 1];
          const symbols = ['M', 'CM', 'D', 'CD', 'C', 'XC', 'L', 'XL', 'X', 'IX', 'V', 'IV', 'I'];
          let result = '';
          
          for (let i = 0; i < values.length; i++) {
            while (num >= values[i]) {
              result += symbols[i];
              num -= values[i];
            }
          }
          return result.toLowerCase();
        }
        
        toAlpha(num) {
          const letters = 'abcdefghijklmnopqrstuvwxyz';
          if (num <= 26) return letters[num - 1];
          return letters[(num - 1) % 26];
        }
        
        toSymbol(num) {
          const symbols = ['*', '†', '‡', '§', '¶', '‖'];
          return symbols[(num - 1) % symbols.length];
        }
        
        process(input, options = {}) {
          try {
            let result;
            
            if (typeof input === 'string') {
              if (input.startsWith('\\footnote{')) {
                const match = input.match(/\\footnote\{(.+?)\}/);
                const content = match ? match[1] : input;
                result = this.processFootnote(content, options.id);
              } else if (input.startsWith('\\footnotemark')) {
                const match = input.match(/\\footnotemark(?:\[(.+?)\])?/);
                const refId = match && match[1] ? match[1] : 'current';
                result = this.processFootnoteRef(refId);
              } else if (input === 'generate-list') {
                result = this.generateFootnoteList();
              } else {
                // 默认作为脚注内容处理
                result = this.processFootnote(input, options.id);
              }
            } else if (input && typeof input === 'object') {
              if (input.action === 'link-markers') {
                result = this.linkFootnoteMarkers();
              } else if (input.action === 'generate-list') {
                result = this.generateFootnoteList();
              } else {
                result = this.processFootnote(input.content || '', input.id);
              }
            } else {
              throw new Error('Invalid input type');
            }
            
            return {
              success: true,
              result,
              metadata: { input, options },
              warnings: [],
              errors: []
            };
          } catch (error) {
            return {
              success: false,
              result: null,
              metadata: { input, options },
              warnings: [],
              errors: [error.message]
            };
          }
        }
        
        validate(input) {
          const errors = [];
          const warnings = [];
          const suggestions = [];
          
          if (typeof input === 'string') {
            // 验证脚注语法
            if (input.includes('\\footnote{')) {
              const matches = input.match(/\\footnote\{(.+?)\}/g);
              if (matches) {
                matches.forEach(match => {
                  const content = match.match(/\\footnote\{(.+?)\}/)[1];
                  if (!content || content.trim() === '') {
                    errors.push('Empty footnote content');
                    suggestions.push('Provide meaningful footnote content');
                  }
                  if (content.length > 1000) {
                    warnings.push('Footnote content is very long');
                    suggestions.push('Consider shortening footnote content');
                  }
                });
              } else {
                errors.push('Invalid footnote syntax');
                suggestions.push('Use format: \\footnote{content}');
              }
            }
            
            // 验证脚注引用语法
            if (input.includes('\\footnotemark')) {
              const matches = input.match(/\\footnotemark(?:\[(.+?)\])?/g);
              if (matches) {
                matches.forEach(match => {
                  const refMatch = match.match(/\\footnotemark(?:\[(.+?)\])?/);
                  if (refMatch && refMatch[1]) {
                    const refId = refMatch[1];
                    if (!this.footnotes.has(refId)) {
                      warnings.push(`Footnote reference '${refId}' not found`);
                      suggestions.push(`Define footnote with ID '${refId}'`);
                    }
                  }
                });
              }
            }
          } else if (typeof input !== 'object') {
            errors.push('Input must be a string or object');
          }
          
          return {
            valid: errors.length === 0,
            errors,
            warnings,
            suggestions
          };
        }
        
        getConfig() {
          return this.options;
        }
      };
    }
  });

  beforeEach(() => {
    footnoteProcessor = new FootnoteProcessor();
  });

  describe('构造函数和配置', () => {
    test('应该使用默认配置创建实例', () => {
      const processor = new FootnoteProcessor();
      const config = processor.getConfig();
      
      expect(config).toHaveProperty('numberStyle', 'arabic');
      expect(config).toHaveProperty('position', 'bottom');
      expect(config).toHaveProperty('separator', '---');
      expect(config).toHaveProperty('backLinks', true);
    });

    test('应该接受自定义配置选项', () => {
      const customOptions = {
        numberStyle: 'roman',
        position: 'margin',
        separator: '___',
        backLinks: false
      };
      
      const processor = new FootnoteProcessor(customOptions);
      const config = processor.getConfig();
      
      expect(config.numberStyle).toBe('roman');
      expect(config.position).toBe('margin');
      expect(config.separator).toBe('___');
      expect(config.backLinks).toBe(false);
    });
  });

  describe('processFootnote', () => {
    test('应该处理基本脚注', () => {
      const content = 'This is a footnote.';
      const result = footnoteProcessor.processFootnote(content);
      
      expect(result).toHaveProperty('type', 'footnote');
      expect(result).toHaveProperty('id');
      expect(result).toHaveProperty('number');
      expect(result).toHaveProperty('content', content);
      expect(result).toHaveProperty('marker');
      expect(result).toHaveProperty('markerHtml');
      expect(result).toHaveProperty('footnoteHtml');
      expect(result).toHaveProperty('element');
      
      expect(typeof result.number).toBe('number');
      expect(result.number).toBeGreaterThan(0);
    });

    test('应该处理带自定义ID的脚注', () => {
      const content = 'Custom footnote';
      const customId = 'custom-footnote-1';
      
      const result = footnoteProcessor.processFootnote(content, customId);
      
      expect(result.id).toBe(customId);
      expect(result.content).toBe(content);
    });

    test('应该生成正确的标记HTML', () => {
      const result = footnoteProcessor.processFootnote('Test footnote');
      
      expect(result.markerHtml).toContain('<sup>');
      expect(result.markerHtml).toContain('<a href=');
      expect(result.markerHtml).toContain('class="footnote-ref"');
      expect(result.markerHtml).toContain(result.marker);
    });

    test('应该生成正确的脚注HTML', () => {
      const result = footnoteProcessor.processFootnote('Test footnote');
      
      expect(result.footnoteHtml).toContain('class="footnote"');
      expect(result.footnoteHtml).toContain('class="footnote-number"');
      expect(result.footnoteHtml).toContain('class="footnote-content"');
      expect(result.footnoteHtml).toContain('Test footnote');
    });

    test('应该根据配置生成返回链接', () => {
      const withBackLinks = new FootnoteProcessor({ backLinks: true });
      const withoutBackLinks = new FootnoteProcessor({ backLinks: false });
      
      const result1 = withBackLinks.processFootnote('Test');
      const result2 = withoutBackLinks.processFootnote('Test');
      
      expect(result1.footnoteHtml).toContain('footnote-backlink');
      expect(result2.footnoteHtml).not.toContain('footnote-backlink');
    });

    test('应该递增脚注编号', () => {
      const result1 = footnoteProcessor.processFootnote('First footnote');
      const result2 = footnoteProcessor.processFootnote('Second footnote');
      const result3 = footnoteProcessor.processFootnote('Third footnote');
      
      expect(result1.number).toBe(1);
      expect(result2.number).toBe(2);
      expect(result3.number).toBe(3);
    });
  });

  describe('generateFootnoteList', () => {
    beforeEach(() => {
      // 添加一些测试脚注
      footnoteProcessor.processFootnote('First footnote');
      footnoteProcessor.processFootnote('Second footnote');
      footnoteProcessor.processFootnote('Third footnote');
    });

    test('应该生成脚注列表', () => {
      const result = footnoteProcessor.generateFootnoteList();
      
      expect(result).toHaveProperty('type', 'footnote-list');
      expect(result).toHaveProperty('total');
      expect(result).toHaveProperty('footnotes');
      expect(result).toHaveProperty('html');
      
      expect(Array.isArray(result.footnotes)).toBe(true);
      expect(typeof result.total).toBe('number');
    });

    test('应该包含所有脚注', () => {
      const result = footnoteProcessor.generateFootnoteList();
      
      expect(result.total).toBe(3);
      expect(result.footnotes).toHaveLength(3);
    });

    test('应该为每个脚注生成正确的条目', () => {
      const result = footnoteProcessor.generateFootnoteList();
      
      result.footnotes.forEach((footnote, index) => {
        expect(footnote).toHaveProperty('id');
        expect(footnote).toHaveProperty('number', index + 1);
        expect(footnote).toHaveProperty('marker');
        expect(footnote).toHaveProperty('content');
        expect(footnote).toHaveProperty('html');
        
        expect(footnote.html).toContain('footnote-item');
        expect(footnote.html).toContain('footnote-number');
        expect(footnote.html).toContain('footnote-content');
      });
    });

    test('应该生成正确的HTML结构', () => {
      const result = footnoteProcessor.generateFootnoteList();
      
      expect(result.html).toContain('class="footnotes"');
      expect(result.html).toContain('footnotes-separator');
      expect(result.html).toContain('footnote-list');
      expect(result.html).toContain('<ol');
      expect(result.html).toContain('<li');
    });

    test('应该处理空的脚注列表', () => {
      const emptyProcessor = new FootnoteProcessor();
      const result = emptyProcessor.generateFootnoteList();
      
      expect(result.total).toBe(0);
      expect(result.footnotes).toHaveLength(0);
      expect(result.html).toContain('empty');
    });
  });

  describe('linkFootnoteMarkers', () => {
    beforeEach(() => {
      footnoteProcessor.processFootnote('First footnote');
      footnoteProcessor.processFootnote('Second footnote');
    });

    test('应该链接脚注标记', () => {
      const result = footnoteProcessor.linkFootnoteMarkers();
      
      expect(result).toHaveProperty('linked');
      expect(result).toHaveProperty('total');
      expect(result).toHaveProperty('success');
      expect(result).toHaveProperty('unlinked');
      
      expect(typeof result.linked).toBe('number');
      expect(typeof result.total).toBe('number');
      expect(typeof result.success).toBe('boolean');
      expect(Array.isArray(result.unlinked)).toBe(true);
    });

    test('应该报告正确的链接数量', () => {
      const result = footnoteProcessor.linkFootnoteMarkers();
      
      expect(result.linked).toBe(2);
      expect(result.total).toBe(2);
      expect(result.success).toBe(true);
      expect(result.unlinked).toHaveLength(0);
    });

    test('应该检测未链接的脚注', () => {
      // 创建一个没有标记的脚注（模拟情况）
      const emptyProcessor = new FootnoteProcessor();
      const result = emptyProcessor.linkFootnoteMarkers();
      
      expect(result.linked).toBe(0);
      expect(result.total).toBe(0);
      expect(result.success).toBe(true);
    });
  });

  describe('processFootnoteRef', () => {
    beforeEach(() => {
      footnoteProcessor.processFootnote('Referenced footnote', 'ref-footnote');
    });

    test('应该处理存在的脚注引用', () => {
      const result = footnoteProcessor.processFootnoteRef('ref-footnote');
      
      expect(result).toHaveProperty('type', 'footnote-reference');
      expect(result).toHaveProperty('refId', 'ref-footnote');
      expect(result).toHaveProperty('found', true);
      expect(result).toHaveProperty('element');
      expect(result).toHaveProperty('html');
      
      expect(result.element).toHaveProperty('id');
      expect(result.element).toHaveProperty('text');
      expect(result.element).toHaveProperty('href');
      expect(result.element).toHaveProperty('class');
    });

    test('应该处理不存在的脚注引用', () => {
      const result = footnoteProcessor.processFootnoteRef('nonexistent');
      
      expect(result.found).toBe(false);
      expect(result.element.class).toContain('missing');
      expect(result.html).toContain('footnote-ref-missing');
    });

    test('应该生成正确的引用HTML', () => {
      const result = footnoteProcessor.processFootnoteRef('ref-footnote');
      
      expect(result.html).toContain('<sup>');
      expect(result.html).toContain('<a href=');
      expect(result.html).toContain('class="footnote-ref"');
      expect(result.html).toContain('#ref-footnote');
    });

    test('应该包含返回引用信息', () => {
      const result = footnoteProcessor.processFootnoteRef('ref-footnote');
      
      if (result.found) {
        expect(result.element).toHaveProperty('backRef');
        expect(result.html).toContain('id=');
      }
    });
  });

  describe('编号样式', () => {
    test('应该支持阿拉伯数字编号', () => {
      const processor = new FootnoteProcessor({ numberStyle: 'arabic' });
      const result = processor.processFootnote('Test');
      
      expect(result.marker).toMatch(/^\d+$/);
    });

    test('应该支持罗马数字编号', () => {
      const processor = new FootnoteProcessor({ numberStyle: 'roman' });
      const result = processor.processFootnote('Test');
      
      expect(result.marker).toMatch(/^[ivxlcdm]+$/);
    });

    test('应该支持字母编号', () => {
      const processor = new FootnoteProcessor({ numberStyle: 'alpha' });
      const result = processor.processFootnote('Test');
      
      expect(result.marker).toMatch(/^[a-z]$/);
    });

    test('应该支持符号编号', () => {
      const processor = new FootnoteProcessor({ numberStyle: 'symbol' });
      const result = processor.processFootnote('Test');
      
      expect(['*', '†', '‡', '§', '¶', '‖']).toContain(result.marker);
    });

    test('应该正确转换罗马数字', () => {
      const processor = new FootnoteProcessor({ numberStyle: 'roman' });
      
      // 测试几个罗马数字转换
      expect(processor.formatNumber(1)).toBe('i');
      expect(processor.formatNumber(4)).toBe('iv');
      expect(processor.formatNumber(5)).toBe('v');
      expect(processor.formatNumber(9)).toBe('ix');
      expect(processor.formatNumber(10)).toBe('x');
    });

    test('应该正确转换字母', () => {
      const processor = new FootnoteProcessor({ numberStyle: 'alpha' });
      
      expect(processor.formatNumber(1)).toBe('a');
      expect(processor.formatNumber(26)).toBe('z');
      expect(processor.formatNumber(27)).toBe('a'); // 循环
    });
  });

  describe('BaseProcessor接口实现', () => {
    test('应该实现process方法', () => {
      const input = '\\footnote{Test footnote}';
      const result = footnoteProcessor.process(input);
      
      expect(result).toHaveProperty('success');
      expect(result).toHaveProperty('result');
      expect(result).toHaveProperty('metadata');
      expect(result).toHaveProperty('warnings');
      expect(result).toHaveProperty('errors');
      
      expect(typeof result.success).toBe('boolean');
      expect(Array.isArray(result.warnings)).toBe(true);
      expect(Array.isArray(result.errors)).toBe(true);
    });

    test('应该实现validate方法', () => {
      const input = '\\footnote{Test footnote}';
      const result = footnoteProcessor.validate(input);
      
      expect(result).toHaveProperty('valid');
      expect(result).toHaveProperty('errors');
      expect(result).toHaveProperty('warnings');
      expect(result).toHaveProperty('suggestions');
      
      expect(typeof result.valid).toBe('boolean');
      expect(Array.isArray(result.errors)).toBe(true);
      expect(Array.isArray(result.warnings)).toBe(true);
      expect(Array.isArray(result.suggestions)).toBe(true);
    });

    test('应该实现getConfig方法', () => {
      const config = footnoteProcessor.getConfig();
      
      expect(config).toBeDefined();
      expect(typeof config).toBe('object');
      expect(config).toHaveProperty('numberStyle');
      expect(config).toHaveProperty('position');
      expect(config).toHaveProperty('backLinks');
    });
  });

  describe('输入验证', () => {
    test('应该验证有效的脚注语法', () => {
      const validInputs = [
        '\\footnote{Simple footnote}',
        '\\footnote{Footnote with \\emph{formatting}}',
        '\\footnotemark',
        '\\footnotemark[custom-id]'
      ];
      
      validInputs.forEach(input => {
        const result = footnoteProcessor.validate(input);
        expect(result).toHaveProperty('valid');
      });
    });

    test('应该检测空的脚注内容', () => {
      const input = '\\footnote{}';
      const result = footnoteProcessor.validate(input);
      
      expect(result.valid).toBe(false);
      expect(result.errors).toContain('Empty footnote content');
    });

    test('应该检测无效的脚注语法', () => {
      const invalidInputs = [
        '\\footnote',
        '\\footnote{',
        '\\footnote unclosed'
      ];
      
      invalidInputs.forEach(input => {
        const result = footnoteProcessor.validate(input);
        if (input.includes('\\footnote{')) {
          expect(result.errors.some(e => e.includes('syntax'))).toBe(true);
        }
      });
    });

    test('应该警告超长脚注内容', () => {
      const longContent = 'x'.repeat(1001);
      const input = `\\footnote{${longContent}}`;
      
      const result = footnoteProcessor.validate(input);
      
      expect(result.warnings.some(w => w.includes('very long'))).toBe(true);
      expect(result.suggestions.some(s => s.includes('shortening'))).toBe(true);
    });

    test('应该检测缺失的脚注引用', () => {
      const input = '\\footnotemark[nonexistent]';
      const result = footnoteProcessor.validate(input);
      
      expect(result.warnings.some(w => w.includes('not found'))).toBe(true);
    });
  });

  describe('错误处理', () => {
    test('应该优雅处理无效输入', () => {
      const result = footnoteProcessor.process(null);
      
      expect(result).toHaveProperty('success', false);
      expect(result).toHaveProperty('errors');
      expect(result.errors.length).toBeGreaterThan(0);
    });

    test('应该处理非字符串输入的验证', () => {
      const result = footnoteProcessor.validate(123);
      
      expect(result.valid).toBe(false);
      expect(result.errors).toContain('Input must be a string or object');
    });

    test('应该提供有用的错误信息和建议', () => {
      const result = footnoteProcessor.validate('\\footnote{}');
      
      if (!result.valid) {
        expect(result.errors.length).toBeGreaterThan(0);
        expect(result.suggestions.length).toBeGreaterThan(0);
      }
    });
  });

  describe('集成测试', () => {
    test('应该处理复杂的脚注文档', () => {
      const inputs = [
        '\\footnote{First footnote with details}',
        '\\footnote{Second footnote with \\emph{emphasis}}',
        '\\footnotemark[custom-ref]'
      ];
      
      inputs.forEach(input => {
        const result = footnoteProcessor.process(input);
        expect(result.success).toBeTruthy();
      });
    });

    test('应该支持完整的脚注工作流', () => {
      // 创建脚注
      const footnote1 = footnoteProcessor.processFootnote('First footnote');
      const footnote2 = footnoteProcessor.processFootnote('Second footnote');
      
      expect(footnote1.number).toBe(1);
      expect(footnote2.number).toBe(2);
      
      // 链接标记
      const linkResult = footnoteProcessor.linkFootnoteMarkers();
      expect(linkResult.success).toBe(true);
      
      // 生成列表
      const listResult = footnoteProcessor.generateFootnoteList();
      expect(listResult.total).toBe(2);
      expect(listResult.footnotes).toHaveLength(2);
    });

    test('应该保持脚注与引用的一致性', () => {
      const footnoteId = 'important-note';
      
      // 创建脚注
      const footnote = footnoteProcessor.processFootnote('Important note', footnoteId);
      
      // 引用该脚注
      const reference = footnoteProcessor.processFootnoteRef(footnoteId);
      
      expect(reference.found).toBe(true);
      expect(reference.element.text).toBe(footnote.marker);
      expect(reference.element.href).toBe(`#${footnoteId}`);
    });
  });
});