/**
 * DocumentClassProcessor 测试套件
 * 基于 interface.md 规范测试文档类别和样式处理
 */

describe('DocumentClassProcessor', () => {
  let DocumentClassProcessor;
  let docClassProcessor;

  beforeAll(() => {
    try {
      DocumentClassProcessor = require('../DocumentClassProcessor');
    } catch (e) {
      // Mock DocumentClassProcessor for testing interface
      DocumentClassProcessor = class DocumentClassProcessor {
        constructor(options = {}) {
          this.options = {
            defaultClass: 'article',
            defaultOptions: [],
            supportedClasses: ['article', 'book', 'report', 'beamer', 'memoir'],
            packageRegistry: {},
            ...options
          };
          this.currentClass = null;
          this.classOptions = [];
          this.packages = new Map();
        }
        
        processDocumentClass(className, options = []) {
          if (!this.options.supportedClasses.includes(className)) {
            className = this.options.defaultClass;
          }
          
          const documentClass = {
            name: className,
            options: Array.isArray(options) ? options : [],
            fullCommand: this.generateClassCommand(className, options),
            supported: this.options.supportedClasses.includes(className),
            features: this.getClassFeatures(className),
            layout: this.getClassLayout(className),
            sections: this.getClassSections(className)
          };
          
          this.currentClass = documentClass;
          this.classOptions = documentClass.options;
          
          return {
            type: 'document-class',
            documentClass,
            styleConfig: this.applyClassStyles(className),
            layoutConfig: this.generateLayoutConfig(className),
            element: {
              tag: 'meta',
              name: 'document-class',
              content: className,
              options: options
            },
            html: `<!-- Document Class: ${className} -->`
          };
        }
        
        applyClassStyles(className) {
          const styles = {
            article: {
              fontSize: '12pt',
              margin: '1in',
              spacing: 'single',
              columns: 1,
              twoSide: false,
              titlePage: false
            },
            book: {
              fontSize: '12pt',
              margin: '1in',
              spacing: 'single',
              columns: 1,
              twoSide: true,
              titlePage: true,
              chapters: true
            },
            report: {
              fontSize: '12pt',
              margin: '1in',
              spacing: 'single',
              columns: 1,
              twoSide: false,
              titlePage: true,
              chapters: true
            },
            beamer: {
              fontSize: '11pt',
              margin: '0.5in',
              spacing: 'single',
              columns: 1,
              presentation: true,
              frames: true
            },
            memoir: {
              fontSize: '12pt',
              margin: '1in',
              spacing: 'single',
              columns: 1,
              twoSide: true,
              titlePage: true,
              chapters: true,
              flexible: true
            }
          };
          
          return {
            type: 'style-config',
            className,
            baseStyles: styles[className] || styles[this.options.defaultClass],
            customStyles: this.processClassOptions(this.classOptions),
            css: this.generateCSS(className),
            html: this.generateStyleHTML(className)
          };
        }
        
        processPackageOptions(packageName, options = []) {
          const packageConfig = {
            name: packageName,
            options: Array.isArray(options) ? options : [],
            fullCommand: this.generatePackageCommand(packageName, options),
            supported: this.isSupportedPackage(packageName),
            features: this.getPackageFeatures(packageName),
            dependencies: this.getPackageDependencies(packageName),
            conflicts: this.getPackageConflicts(packageName)
          };
          
          this.packages.set(packageName, packageConfig);
          
          return {
            type: 'package-config',
            package: packageConfig,
            enabled: packageConfig.supported,
            warnings: this.validatePackage(packageName, options),
            element: {
              tag: 'meta',
              name: 'package',
              content: packageName,
              options: options
            },
            html: `<!-- Package: ${packageName} -->`
          };
        }
        
        generateLayoutConfig(className) {
          const layouts = {
            article: {
              hasChapters: false,
              hasParts: false,
              maxSectionLevel: 6,
              numberedSections: true,
              toc: true,
              bibliography: true,
              index: false,
              appendix: true
            },
            book: {
              hasChapters: true,
              hasParts: true,
              maxSectionLevel: 6,
              numberedSections: true,
              toc: true,
              bibliography: true,
              index: true,
              appendix: true,
              frontMatter: true,
              backMatter: true
            },
            report: {
              hasChapters: true,
              hasParts: false,
              maxSectionLevel: 6,
              numberedSections: true,
              toc: true,
              bibliography: true,
              index: false,
              appendix: true,
              abstract: true
            },
            beamer: {
              hasChapters: false,
              hasParts: false,
              hasFrames: true,
              maxSectionLevel: 3,
              numberedSections: false,
              toc: false,
              bibliography: false,
              navigation: true,
              themes: true
            },
            memoir: {
              hasChapters: true,
              hasParts: true,
              maxSectionLevel: 6,
              numberedSections: true,
              toc: true,
              bibliography: true,
              index: true,
              appendix: true,
              frontMatter: true,
              backMatter: true,
              flexible: true
            }
          };
          
          return {
            type: 'layout-config',
            className,
            layout: layouts[className] || layouts[this.options.defaultClass],
            hierarchy: this.generateSectionHierarchy(className),
            pageLayout: this.generatePageLayout(className)
          };
        }
        
        // 辅助方法
        generateClassCommand(className, options) {
          if (options.length === 0) {
            return `\\documentclass{${className}}`;
          }
          return `\\documentclass[${options.join(',')}]{${className}}`;
        }
        
        generatePackageCommand(packageName, options) {
          if (options.length === 0) {
            return `\\usepackage{${packageName}}`;
          }
          return `\\usepackage[${options.join(',')}]{${packageName}}`;
        }
        
        getClassFeatures(className) {
          const features = {
            article: ['sections', 'figures', 'tables', 'bibliography'],
            book: ['chapters', 'parts', 'sections', 'figures', 'tables', 'index', 'bibliography'],
            report: ['chapters', 'sections', 'abstract', 'figures', 'tables', 'bibliography'],
            beamer: ['frames', 'slides', 'navigation', 'themes', 'overlays'],
            memoir: ['chapters', 'parts', 'sections', 'figures', 'tables', 'index', 'bibliography', 'flexible-layout']
          };
          
          return features[className] || features[this.options.defaultClass] || [];
        }
        
        getClassLayout(className) {
          const layouts = {
            article: { type: 'single-column', titlePage: false },
            book: { type: 'book', titlePage: true, twoSide: true },
            report: { type: 'report', titlePage: true },
            beamer: { type: 'presentation', aspectRatio: '4:3' },
            memoir: { type: 'flexible', titlePage: true, twoSide: true }
          };
          
          return layouts[className] || layouts[this.options.defaultClass] || {};
        }
        
        getClassSections(className) {
          const sections = {
            article: ['section', 'subsection', 'subsubsection', 'paragraph', 'subparagraph'],
            book: ['part', 'chapter', 'section', 'subsection', 'subsubsection', 'paragraph', 'subparagraph'],
            report: ['chapter', 'section', 'subsection', 'subsubsection', 'paragraph', 'subparagraph'],
            beamer: ['section', 'subsection', 'frame'],
            memoir: ['part', 'chapter', 'section', 'subsection', 'subsubsection', 'paragraph', 'subparagraph']
          };
          
          return sections[className] || sections[this.options.defaultClass] || [];
        }
        
        processClassOptions(options) {
          const customStyles = {};
          
          options.forEach(option => {
            switch (option) {
              case '10pt':
              case '11pt':
              case '12pt':
                customStyles.fontSize = option;
                break;
              case 'a4paper':
              case 'letterpaper':
                customStyles.paperSize = option;
                break;
              case 'twoside':
                customStyles.twoSide = true;
                break;
              case 'oneside':
                customStyles.twoSide = false;
                break;
              case 'twocolumn':
                customStyles.columns = 2;
                break;
              case 'onecolumn':
                customStyles.columns = 1;
                break;
              case 'titlepage':
                customStyles.titlePage = true;
                break;
              case 'notitlepage':
                customStyles.titlePage = false;
                break;
              default:
                customStyles[option] = true;
            }
          });
          
          return customStyles;
        }
        
        isSupportedPackage(packageName) {
          const supportedPackages = [
            'amsmath', 'amsfonts', 'amssymb', 'graphicx', 'float', 'caption',
            'hyperref', 'url', 'cite', 'natbib', 'geometry', 'fancyhdr',
            'babel', 'inputenc', 'fontenc', 'xcolor', 'listings', 'tikz'
          ];
          
          return supportedPackages.includes(packageName) ||
                 this.options.packageRegistry[packageName];
        }
        
        getPackageFeatures(packageName) {
          const features = {
            'amsmath': ['math-environments', 'equations', 'matrices'],
            'graphicx': ['include-graphics', 'image-scaling', 'rotation'],
            'hyperref': ['links', 'bookmarks', 'cross-references'],
            'geometry': ['page-layout', 'margins', 'paper-size'],
            'babel': ['internationalization', 'hyphenation'],
            'xcolor': ['colors', 'color-mixing', 'named-colors'],
            'listings': ['code-highlighting', 'syntax-coloring'],
            'tikz': ['graphics', 'diagrams', 'plots']
          };
          
          return features[packageName] || [];
        }
        
        getPackageDependencies(packageName) {
          const dependencies = {
            'hyperref': ['url'],
            'natbib': ['cite'],
            'tikz': ['xcolor'],
            'listings': ['xcolor']
          };
          
          return dependencies[packageName] || [];
        }
        
        getPackageConflicts(packageName) {
          const conflicts = {
            'cite': ['natbib'],
            'natbib': ['cite']
          };
          
          return conflicts[packageName] || [];
        }
        
        validatePackage(packageName, options) {
          const warnings = [];
          
          // 检查包冲突
          const conflicts = this.getPackageConflicts(packageName);
          conflicts.forEach(conflict => {
            if (this.packages.has(conflict)) {
              warnings.push(`Package '${packageName}' conflicts with '${conflict}'`);
            }
          });
          
          // 检查依赖
          const dependencies = this.getPackageDependencies(packageName);
          dependencies.forEach(dep => {
            if (!this.packages.has(dep)) {
              warnings.push(`Package '${packageName}' requires '${dep}'`);
            }
          });
          
          // 检查包是否受支持
          if (!this.isSupportedPackage(packageName)) {
            warnings.push(`Package '${packageName}' may not be fully supported`);
          }
          
          return warnings;
        }
        
        generateSectionHierarchy(className) {
          const sections = this.getClassSections(className);
          
          return sections.map((section, index) => ({
            name: section,
            level: index + 1,
            numbered: index < 5, // 前5级默认编号
            toc: index < 3 // 前3级在目录中显示
          }));
        }
        
        generatePageLayout(className) {
          const layout = this.getClassLayout(className);
          
          return {
            type: layout.type || 'standard',
            margins: {
              top: '1in',
              bottom: '1in',
              left: '1in',
              right: '1in'
            },
            header: layout.twoSide ? 'alternating' : 'single',
            footer: 'page-numbers',
            columns: 1,
            spacing: 'single'
          };
        }
        
        generateCSS(className) {
          const styles = this.applyClassStyles(className);
          const baseStyles = styles.baseStyles;
          const customStyles = styles.customStyles;
          
          return `
            .document-${className} {
              font-size: ${baseStyles.fontSize};
              margin: ${baseStyles.margin};
              line-height: ${baseStyles.spacing === 'double' ? '2' : '1.5'};
              columns: ${baseStyles.columns};
            }
            
            ${customStyles.twoSide ? '.document-' + className + ' { page-break-inside: avoid; }' : ''}
            ${customStyles.columns > 1 ? '.document-' + className + ' { column-gap: 2em; }' : ''}
          `.trim();
        }
        
        generateStyleHTML(className) {
          return `<style type="text/css">${this.generateCSS(className)}</style>`;
        }
        
        // 公共方法
        getCurrentClass() {
          return this.currentClass;
        }
        
        getLoadedPackages() {
          return Array.from(this.packages.values());
        }
        
        isPackageLoaded(packageName) {
          return this.packages.has(packageName);
        }
        
        process(input, options = {}) {
          try {
            let result;
            
            if (typeof input === 'string') {
              if (input.startsWith('\\documentclass')) {
                const match = input.match(/\\documentclass(?:\[([^\]]+)\])?\{([^}]+)\}/);
                if (match) {
                  const classOptions = match[1] ? match[1].split(',').map(opt => opt.trim()) : [];
                  const className = match[2];
                  result = this.processDocumentClass(className, classOptions);
                } else {
                  throw new Error('Invalid document class syntax');
                }
              } else if (input.startsWith('\\usepackage')) {
                const match = input.match(/\\usepackage(?:\[([^\]]+)\])?\{([^}]+)\}/);
                if (match) {
                  const packageOptions = match[1] ? match[1].split(',').map(opt => opt.trim()) : [];
                  const packageName = match[2];
                  result = this.processPackageOptions(packageName, packageOptions);
                } else {
                  throw new Error('Invalid package syntax');
                }
              } else {
                // 假设是类名
                result = this.processDocumentClass(input, options.classOptions || []);
              }
            } else if (input && typeof input === 'object') {
              if (input.type === 'document-class') {
                result = this.processDocumentClass(input.name, input.options);
              } else if (input.type === 'package') {
                result = this.processPackageOptions(input.name, input.options);
              } else {
                throw new Error('Invalid input object');
              }
            } else {
              throw new Error('Invalid input type');
            }
            
            return {
              success: true,
              result,
              metadata: { input, options },
              warnings: [],
              errors: []
            };
          } catch (error) {
            return {
              success: false,
              result: null,
              metadata: { input, options },
              warnings: [],
              errors: [error.message]
            };
          }
        }
        
        validate(input) {
          const errors = [];
          const warnings = [];
          const suggestions = [];
          
          if (typeof input === 'string') {
            // 验证文档类语法
            if (input.includes('\\documentclass')) {
              const matches = input.match(/\\documentclass(?:\[([^\]]*)\])?\{([^}]*)\}/g);
              if (matches) {
                matches.forEach(match => {
                  const parsed = match.match(/\\documentclass(?:\[([^\]]*)\])?\{([^}]*)\}/);
                  const className = parsed[2];
                  
                  if (!className) {
                    errors.push('Empty document class name');
                    suggestions.push('Specify a valid document class name');
                  } else if (!this.options.supportedClasses.includes(className)) {
                    warnings.push(`Document class '${className}' may not be supported`);
                    suggestions.push(`Use supported classes: ${this.options.supportedClasses.join(', ')}`);
                  }
                });
              } else {
                errors.push('Invalid \\documentclass syntax');
                suggestions.push('Use format: \\documentclass[options]{classname}');
              }
            }
            
            // 验证包语法
            if (input.includes('\\usepackage')) {
              const matches = input.match(/\\usepackage(?:\[([^\]]*)\])?\{([^}]*)\}/g);
              if (matches) {
                matches.forEach(match => {
                  const parsed = match.match(/\\usepackage(?:\[([^\]]*)\])?\{([^}]*)\}/);
                  const packageName = parsed[2];
                  
                  if (!packageName) {
                    errors.push('Empty package name');
                    suggestions.push('Specify a valid package name');
                  } else if (!this.isSupportedPackage(packageName)) {
                    warnings.push(`Package '${packageName}' may not be supported`);
                  }
                });
              }
            }
          } else if (input && typeof input === 'object') {
            // 验证对象
            if (input.type === 'document-class') {
              if (!input.name) {
                errors.push('Document class object is missing name');
              }
            } else if (input.type === 'package') {
              if (!input.name) {
                errors.push('Package object is missing name');
              }
            }
          } else if (typeof input !== 'string') {
            errors.push('Input must be a string or object');
          }
          
          return {
            valid: errors.length === 0,
            errors,
            warnings,
            suggestions
          };
        }
        
        getConfig() {
          return this.options;
        }
      };
    }
  });

  beforeEach(() => {
    docClassProcessor = new DocumentClassProcessor();
  });

  describe('构造函数和配置', () => {
    test('应该使用默认配置创建实例', () => {
      const processor = new DocumentClassProcessor();
      const config = processor.getConfig();
      
      expect(config).toHaveProperty('defaultClass', 'article');
      expect(config).toHaveProperty('defaultOptions');
      expect(config).toHaveProperty('supportedClasses');
      expect(config).toHaveProperty('packageRegistry');
      
      expect(Array.isArray(config.defaultOptions)).toBe(true);
      expect(Array.isArray(config.supportedClasses)).toBe(true);
      expect(typeof config.packageRegistry).toBe('object');
    });

    test('应该接受自定义配置选项', () => {
      const customOptions = {
        defaultClass: 'book',
        supportedClasses: ['article', 'book', 'custom'],
        packageRegistry: {
          'custom-package': { supported: true }
        }
      };
      
      const processor = new DocumentClassProcessor(customOptions);
      const config = processor.getConfig();
      
      expect(config.defaultClass).toBe('book');
      expect(config.supportedClasses).toContain('custom');
      expect(config.packageRegistry).toHaveProperty('custom-package');
    });
  });

  describe('processDocumentClass', () => {
    test('应该处理基本的文档类', () => {
      const result = docClassProcessor.processDocumentClass('article');
      
      expect(result).toHaveProperty('type', 'document-class');
      expect(result).toHaveProperty('documentClass');
      expect(result).toHaveProperty('styleConfig');
      expect(result).toHaveProperty('layoutConfig');
      expect(result).toHaveProperty('element');
      expect(result).toHaveProperty('html');
      
      expect(result.documentClass.name).toBe('article');
      expect(result.documentClass.supported).toBe(true);
      expect(Array.isArray(result.documentClass.options)).toBe(true);
    });

    test('应该处理带选项的文档类', () => {
      const options = ['12pt', 'a4paper', 'twoside'];
      const result = docClassProcessor.processDocumentClass('article', options);
      
      expect(result.documentClass.options).toEqual(options);
      expect(result.documentClass.fullCommand).toBe('\\documentclass[12pt,a4paper,twoside]{article}');
    });

    test('应该处理不支持的文档类', () => {
      const result = docClassProcessor.processDocumentClass('unknown');
      
      expect(result.documentClass.name).toBe('article'); // 回退到默认类
      expect(result.documentClass.supported).toBe(true);
    });

    test('应该生成正确的类特性', () => {
      const articleResult = docClassProcessor.processDocumentClass('article');
      const bookResult = docClassProcessor.processDocumentClass('book');
      
      expect(articleResult.documentClass.features).toContain('sections');
      expect(bookResult.documentClass.features).toContain('chapters');
      expect(bookResult.documentClass.features).toContain('index');
    });

    test('应该设置当前文档类', () => {
      docClassProcessor.processDocumentClass('book');
      
      const currentClass = docClassProcessor.getCurrentClass();
      expect(currentClass.name).toBe('book');
    });

    test('应该生成正确的命令字符串', () => {
      const result1 = docClassProcessor.processDocumentClass('article');
      const result2 = docClassProcessor.processDocumentClass('book', ['12pt', 'twoside']);
      
      expect(result1.documentClass.fullCommand).toBe('\\documentclass{article}');
      expect(result2.documentClass.fullCommand).toBe('\\documentclass[12pt,twoside]{book}');
    });
  });

  describe('applyClassStyles', () => {
    test('应该应用不同文档类的样式', () => {
      const classes = ['article', 'book', 'report', 'beamer', 'memoir'];
      
      classes.forEach(className => {
        const result = docClassProcessor.applyClassStyles(className);
        
        expect(result).toHaveProperty('type', 'style-config');
        expect(result).toHaveProperty('className', className);
        expect(result).toHaveProperty('baseStyles');
        expect(result).toHaveProperty('css');
        expect(result).toHaveProperty('html');
        
        expect(result.baseStyles).toHaveProperty('fontSize');
        expect(result.baseStyles).toHaveProperty('margin');
      });
    });

    test('应该处理类选项对样式的影响', () => {
      docClassProcessor.processDocumentClass('article', ['12pt', 'twoside']);
      const result = docClassProcessor.applyClassStyles('article');
      
      expect(result.customStyles).toHaveProperty('fontSize', '12pt');
      expect(result.customStyles).toHaveProperty('twoSide', true);
    });

    test('应该为特殊类生成特殊样式', () => {
      const beamerResult = docClassProcessor.applyClassStyles('beamer');
      const bookResult = docClassProcessor.applyClassStyles('book');
      
      expect(beamerResult.baseStyles).toHaveProperty('presentation', true);
      expect(bookResult.baseStyles).toHaveProperty('chapters', true);
      expect(bookResult.baseStyles).toHaveProperty('twoSide', true);
    });

    test('应该生成有效的CSS', () => {
      const result = docClassProcessor.applyClassStyles('article');
      
      expect(result.css).toContain('.document-article');
      expect(result.css).toContain('font-size:');
      expect(result.css).toContain('margin:');
    });

    test('应该生成样式HTML', () => {
      const result = docClassProcessor.applyClassStyles('article');
      
      expect(result.html).toContain('<style');
      expect(result.html).toContain('</style>');
      expect(result.html).toContain('.document-article');
    });
  });

  describe('processPackageOptions', () => {
    test('应该处理基本包', () => {
      const result = docClassProcessor.processPackageOptions('amsmath');
      
      expect(result).toHaveProperty('type', 'package-config');
      expect(result).toHaveProperty('package');
      expect(result).toHaveProperty('enabled');
      expect(result).toHaveProperty('warnings');
      expect(result).toHaveProperty('element');
      expect(result).toHaveProperty('html');
      
      expect(result.package.name).toBe('amsmath');
      expect(result.package.supported).toBe(true);
      expect(result.enabled).toBe(true);
    });

    test('应该处理带选项的包', () => {
      const options = ['utf8'];
      const result = docClassProcessor.processPackageOptions('inputenc', options);
      
      expect(result.package.options).toEqual(options);
      expect(result.package.fullCommand).toBe('\\usepackage[utf8]{inputenc}');
    });

    test('应该检测包特性', () => {
      const result = docClassProcessor.processPackageOptions('amsmath');
      
      expect(result.package.features).toContain('math-environments');
      expect(result.package.features).toContain('equations');
    });

    test('应该检测包依赖', () => {
      const result = docClassProcessor.processPackageOptions('hyperref');
      
      expect(result.package.dependencies).toContain('url');
    });

    test('应该检测包冲突', () => {
      // 先加载一个包
      docClassProcessor.processPackageOptions('cite');
      
      // 然后加载冲突的包
      const result = docClassProcessor.processPackageOptions('natbib');
      
      expect(result.warnings.some(w => w.includes('conflicts'))).toBe(true);
    });

    test('应该验证包依赖', () => {
      const result = docClassProcessor.processPackageOptions('hyperref');
      
      expect(result.warnings.some(w => w.includes('requires'))).toBe(true);
    });

    test('应该存储已加载的包', () => {
      docClassProcessor.processPackageOptions('amsmath');
      docClassProcessor.processPackageOptions('graphicx');
      
      expect(docClassProcessor.isPackageLoaded('amsmath')).toBe(true);
      expect(docClassProcessor.isPackageLoaded('graphicx')).toBe(true);
      expect(docClassProcessor.isPackageLoaded('unknown')).toBe(false);
      
      const loadedPackages = docClassProcessor.getLoadedPackages();
      expect(loadedPackages).toHaveLength(2);
    });
  });

  describe('generateLayoutConfig', () => {
    test('应该生成不同文档类的布局配置', () => {
      const classes = ['article', 'book', 'report', 'beamer', 'memoir'];
      
      classes.forEach(className => {
        const result = docClassProcessor.generateLayoutConfig(className);
        
        expect(result).toHaveProperty('type', 'layout-config');
        expect(result).toHaveProperty('className', className);
        expect(result).toHaveProperty('layout');
        expect(result).toHaveProperty('hierarchy');
        expect(result).toHaveProperty('pageLayout');
        
        expect(typeof result.layout.hasChapters).toBe('boolean');
        expect(typeof result.layout.maxSectionLevel).toBe('number');
        expect(Array.isArray(result.hierarchy)).toBe(true);
      });
    });

    test('应该为不同类生成正确的章节层级', () => {
      const articleResult = docClassProcessor.generateLayoutConfig('article');
      const bookResult = docClassProcessor.generateLayoutConfig('book');
      
      expect(articleResult.layout.hasChapters).toBe(false);
      expect(bookResult.layout.hasChapters).toBe(true);
      expect(bookResult.layout.hasParts).toBe(true);
    });

    test('应该生成章节层级结构', () => {
      const result = docClassProcessor.generateLayoutConfig('book');
      
      expect(result.hierarchy.length).toBeGreaterThan(0);
      result.hierarchy.forEach(section => {
        expect(section).toHaveProperty('name');
        expect(section).toHaveProperty('level');
        expect(section).toHaveProperty('numbered');
        expect(section).toHaveProperty('toc');
      });
    });

    test('应该生成页面布局配置', () => {
      const result = docClassProcessor.generateLayoutConfig('article');
      
      expect(result.pageLayout).toHaveProperty('type');
      expect(result.pageLayout).toHaveProperty('margins');
      expect(result.pageLayout).toHaveProperty('header');
      expect(result.pageLayout).toHaveProperty('footer');
      
      expect(result.pageLayout.margins).toHaveProperty('top');
      expect(result.pageLayout.margins).toHaveProperty('bottom');
    });
  });

  describe('辅助方法', () => {
    test('processClassOptions应该正确解析选项', () => {
      const options = ['12pt', 'a4paper', 'twoside', 'twocolumn', 'titlepage'];
      const result = docClassProcessor.processClassOptions(options);
      
      expect(result.fontSize).toBe('12pt');
      expect(result.paperSize).toBe('a4paper');
      expect(result.twoSide).toBe(true);
      expect(result.columns).toBe(2);
      expect(result.titlePage).toBe(true);
    });

    test('isSupportedPackage应该正确识别支持的包', () => {
      expect(docClassProcessor.isSupportedPackage('amsmath')).toBe(true);
      expect(docClassProcessor.isSupportedPackage('graphicx')).toBe(true);
      expect(docClassProcessor.isSupportedPackage('unknown')).toBe(false);
    });

    test('getPackageFeatures应该返回包特性', () => {
      const amsmathFeatures = docClassProcessor.getPackageFeatures('amsmath');
      const graphicxFeatures = docClassProcessor.getPackageFeatures('graphicx');
      
      expect(Array.isArray(amsmathFeatures)).toBe(true);
      expect(Array.isArray(graphicxFeatures)).toBe(true);
      
      expect(amsmathFeatures).toContain('math-environments');
      expect(graphicxFeatures).toContain('include-graphics');
    });

    test('validatePackage应该检测问题', () => {
      // 测试依赖检查
      const warnings1 = docClassProcessor.validatePackage('hyperref', []);
      expect(warnings1.some(w => w.includes('requires'))).toBe(true);
      
      // 测试不支持的包
      const warnings2 = docClassProcessor.validatePackage('unknown-package', []);
      expect(warnings2.some(w => w.includes('not be fully supported'))).toBe(true);
    });
  });

  describe('BaseProcessor接口实现', () => {
    test('应该实现process方法', () => {
      const input = '\\documentclass{article}';
      const result = docClassProcessor.process(input);
      
      expect(result).toHaveProperty('success');
      expect(result).toHaveProperty('result');
      expect(result).toHaveProperty('metadata');
      expect(result).toHaveProperty('warnings');
      expect(result).toHaveProperty('errors');
      
      expect(typeof result.success).toBe('boolean');
      expect(Array.isArray(result.warnings)).toBe(true);
      expect(Array.isArray(result.errors)).toBe(true);
    });

    test('应该实现validate方法', () => {
      const input = '\\documentclass{article}';
      const result = docClassProcessor.validate(input);
      
      expect(result).toHaveProperty('valid');
      expect(result).toHaveProperty('errors');
      expect(result).toHaveProperty('warnings');
      expect(result).toHaveProperty('suggestions');
      
      expect(typeof result.valid).toBe('boolean');
      expect(Array.isArray(result.errors)).toBe(true);
      expect(Array.isArray(result.warnings)).toBe(true);
      expect(Array.isArray(result.suggestions)).toBe(true);
    });

    test('应该实现getConfig方法', () => {
      const config = docClassProcessor.getConfig();
      
      expect(config).toBeDefined();
      expect(typeof config).toBe('object');
      expect(config).toHaveProperty('defaultClass');
      expect(config).toHaveProperty('supportedClasses');
      expect(config).toHaveProperty('packageRegistry');
    });
  });

  describe('输入处理', () => {
    test('应该解析LaTeX文档类命令', () => {
      const inputs = [
        '\\documentclass{article}',
        '\\documentclass[12pt]{book}',
        '\\documentclass[12pt,a4paper,twoside]{report}'
      ];
      
      inputs.forEach(input => {
        const result = docClassProcessor.process(input);
        expect(result.success).toBe(true);
        expect(result.result.type).toBe('document-class');
      });
    });

    test('应该解析LaTeX包命令', () => {
      const inputs = [
        '\\usepackage{amsmath}',
        '\\usepackage[utf8]{inputenc}',
        '\\usepackage[margin=1in,paper=a4paper]{geometry}'
      ];
      
      inputs.forEach(input => {
        const result = docClassProcessor.process(input);
        expect(result.success).toBe(true);
        expect(result.result.type).toBe('package-config');
      });
    });

    test('应该处理对象输入', () => {
      const classObj = { type: 'document-class', name: 'article', options: ['12pt'] };
      const packageObj = { type: 'package', name: 'amsmath', options: [] };
      
      const result1 = docClassProcessor.process(classObj);
      const result2 = docClassProcessor.process(packageObj);
      
      expect(result1.success).toBe(true);
      expect(result2.success).toBe(true);
    });

    test('应该处理简单字符串', () => {
      const result = docClassProcessor.process('book');
      
      expect(result.success).toBe(true);
      expect(result.result.documentClass.name).toBe('book');
    });
  });

  describe('输入验证', () => {
    test('应该验证有效的文档类语法', () => {
      const validInputs = [
        '\\documentclass{article}',
        '\\documentclass[12pt,a4paper]{book}',
        '\\usepackage{amsmath}',
        '\\usepackage[utf8]{inputenc}'
      ];
      
      validInputs.forEach(input => {
        const result = docClassProcessor.validate(input);
        expect(result.valid).toBe(true);
      });
    });

    test('应该检测无效的语法', () => {
      const invalidInputs = [
        '\\documentclass{}',
        '\\documentclass',
        '\\usepackage{}',
        '\\usepackage'
      ];
      
      invalidInputs.forEach(input => {
        const result = docClassProcessor.validate(input);
        expect(result.valid).toBe(false);
      });
    });

    test('应该警告不支持的类和包', () => {
      const result1 = docClassProcessor.validate('\\documentclass{unknown}');
      const result2 = docClassProcessor.validate('\\usepackage{unknown}');
      
      expect(result1.warnings.some(w => w.includes('may not be supported'))).toBe(true);
      expect(result2.warnings.some(w => w.includes('may not be supported'))).toBe(true);
    });

    test('应该验证对象输入', () => {
      const validObj = { type: 'document-class', name: 'article' };
      const invalidObj = { type: 'document-class' }; // 缺少name
      
      const result1 = docClassProcessor.validate(validObj);
      const result2 = docClassProcessor.validate(invalidObj);
      
      expect(result1.valid).toBe(true);
      expect(result2.valid).toBe(false);
    });

    test('应该检测无效输入类型', () => {
      const result = docClassProcessor.validate(123);
      
      expect(result.valid).toBe(false);
      expect(result.errors).toContain('Input must be a string or object');
    });
  });

  describe('错误处理', () => {
    test('应该优雅处理无效输入', () => {
      const result = docClassProcessor.process(null);
      
      expect(result).toHaveProperty('success', false);
      expect(result).toHaveProperty('errors');
      expect(result.errors.length).toBeGreaterThan(0);
    });

    test('应该处理无效的LaTeX语法', () => {
      const result = docClassProcessor.process('\\documentclass');
      
      expect(result.success).toBe(false);
      expect(result.errors).toContain('Invalid document class syntax');
    });

    test('应该处理无效对象', () => {
      const result = docClassProcessor.process({ invalid: true });
      
      expect(result.success).toBe(false);
      expect(result.errors).toContain('Invalid input object');
    });

    test('应该提供有用的错误信息', () => {
      const result = docClassProcessor.validate('\\documentclass{}');
      
      if (!result.valid) {
        expect(result.errors.length).toBeGreaterThan(0);
        expect(result.suggestions.length).toBeGreaterThan(0);
      }
    });
  });

  describe('集成测试', () => {
    test('应该处理完整的文档前言', () => {
      const commands = [
        '\\documentclass[12pt,a4paper]{book}',
        '\\usepackage{amsmath}',
        '\\usepackage[utf8]{inputenc}',
        '\\usepackage[margin=1in]{geometry}',
        '\\usepackage{hyperref}'
      ];
      
      const results = commands.map(cmd => docClassProcessor.process(cmd));
      
      expect(results.every(r => r.success)).toBe(true);
      expect(docClassProcessor.getCurrentClass().name).toBe('book');
      expect(docClassProcessor.getLoadedPackages()).toHaveLength(4);
    });

    test('应该检测包冲突', () => {
      // 加载冲突的包
      docClassProcessor.process('\\usepackage{cite}');
      const result = docClassProcessor.process('\\usepackage{natbib}');
      
      expect(result.result.warnings.some(w => w.includes('conflicts'))).toBe(true);
    });

    test('应该正确应用类选项到样式', () => {
      docClassProcessor.process('\\documentclass[12pt,twoside,twocolumn]{article}');
      
      const styleConfig = docClassProcessor.applyClassStyles('article');
      
      expect(styleConfig.customStyles.fontSize).toBe('12pt');
      expect(styleConfig.customStyles.twoSide).toBe(true);
      expect(styleConfig.customStyles.columns).toBe(2);
    });

    test('应该生成一致的布局配置', () => {
      docClassProcessor.process('\\documentclass{book}');
      const layoutConfig = docClassProcessor.generateLayoutConfig('book');
      
      expect(layoutConfig.layout.hasChapters).toBe(true);
      expect(layoutConfig.hierarchy.some(h => h.name === 'chapter')).toBe(true);
      expect(layoutConfig.hierarchy.some(h => h.name === 'part')).toBe(true);
    });

    test('应该支持自定义配置', () => {
      const customProcessor = new DocumentClassProcessor({
        supportedClasses: ['article', 'custom'],
        packageRegistry: {
          'custom-pkg': { supported: true }
        }
      });
      
      const classResult = customProcessor.process('\\documentclass{custom}');
      const pkgResult = customProcessor.process('\\usepackage{custom-pkg}');
      
      expect(classResult.result.documentClass.supported).toBe(true);
      expect(pkgResult.result.package.supported).toBe(true);
    });
  });
});