/**
 * BibliographyProcessor 测试套件
 * 基于 interface.md 规范测试参考文献处理
 */

describe('BibliographyProcessor', () => {
  let BibliographyProcessor;
  let bibliographyProcessor;

  beforeAll(() => {
    try {
      BibliographyProcessor = require('../BibliographyProcessor');
    } catch (e) {
      // Mock BibliographyProcessor for testing interface
      BibliographyProcessor = class BibliographyProcessor {
        constructor(options = {}) {
          this.options = {
            style: 'ieee',
            sortBy: 'author',
            includeUrls: true,
            includeDoi: true,
            maxAuthors: 6,
            abbreviateJournals: false,
            ...options
          };
          this.bibliography = new Map();
          this.citations = new Map();
        }
        
        processBibliography(bibData) {
          if (!bibData || typeof bibData !== 'object') {
            return {
              type: 'bibliography',
              entries: [],
              total: 0,
              style: this.options.style,
              html: '<div class="bibliography empty">No bibliography entries found.</div>'
            };
          }
          
          const entries = Array.isArray(bibData) ? bibData : Object.values(bibData);
          const processedEntries = entries.map(entry => this.processBibEntry(entry));
          
          // 排序
          const sortedEntries = this.sortEntries(processedEntries);
          
          return {
            type: 'bibliography',
            entries: sortedEntries,
            total: sortedEntries.length,
            style: this.options.style,
            html: this.generateBibliographyHTML(sortedEntries)
          };
        }
        
        processCitation(citation, style = null) {
          const citationStyle = style || this.options.style;
          
          // 解析引用键
          const keys = this.parseCitationKeys(citation);
          const citations = keys.map(key => {
            const entry = this.bibliography.get(key);
            if (!entry) {
              return {
                key,
                found: false,
                text: `[${key}]`,
                element: `<span class="citation-missing" data-key="${key}">[${key}]</span>`
              };
            }
            
            return {
              key,
              found: true,
              text: this.formatCitation(entry, citationStyle),
              element: this.generateCitationElement(entry, citationStyle, key)
            };
          });
          
          return {
            type: 'citation',
            style: citationStyle,
            keys,
            citations,
            formatted: this.combineCitations(citations, citationStyle),
            html: this.generateCitationHTML(citations, citationStyle)
          };
        }
        
        generateBibEntry(entry) {
          if (!entry || typeof entry !== 'object') {
            throw new Error('Invalid bibliography entry');
          }
          
          const processedEntry = {
            key: entry.key || entry.id || 'unknown',
            type: entry.type || 'article',
            title: entry.title || 'Untitled',
            author: this.processAuthors(entry.author || entry.authors),
            year: entry.year || entry.date,
            journal: entry.journal,
            volume: entry.volume,
            number: entry.number,
            pages: entry.pages,
            publisher: entry.publisher,
            address: entry.address,
            url: entry.url,
            doi: entry.doi,
            isbn: entry.isbn,
            note: entry.note,
            abstract: entry.abstract
          };
          
          // 存储到内部映射
          this.bibliography.set(processedEntry.key, processedEntry);
          
          return {
            type: 'bib-entry',
            entry: processedEntry,
            formatted: this.formatReference(processedEntry, this.options.style),
            html: this.generateEntryHTML(processedEntry)
          };
        }
        
        formatReference(ref, style = 'ieee') {
          switch (style.toLowerCase()) {
            case 'apa':
              return this.formatAPA(ref);
            case 'mla':
              return this.formatMLA(ref);
            case 'chicago':
              return this.formatChicago(ref);
            default:
              return this.formatIEEE(ref);
          }
        }
        
        // 格式化方法
        formatIEEE(ref) {
          const parts = [];
          
          if (ref.author) parts.push(`${ref.author},`);
          if (ref.title) parts.push(`"${ref.title},"`);
          if (ref.journal) parts.push(`${ref.journal},`);
          if (ref.volume) parts.push(`vol. ${ref.volume},`);
          if (ref.number) parts.push(`no. ${ref.number},`);
          if (ref.pages) parts.push(`pp. ${ref.pages},`);
          if (ref.year) parts.push(`${ref.year}.`);
          if (ref.doi && this.options.includeDoi) parts.push(`DOI: ${ref.doi}.`);
          
          return parts.join(' ').replace(/,\s*$/, '.');
        }
        
        formatAPA(ref) {
          const parts = [];
          
          if (ref.author) parts.push(`${ref.author}`);
          if (ref.year) parts.push(`(${ref.year}).`);
          if (ref.title) parts.push(`${ref.title}.`);
          if (ref.journal) parts.push(`${ref.journal},`);
          if (ref.volume) parts.push(`${ref.volume}${ref.number ? `(${ref.number})` : ''},`);
          if (ref.pages) parts.push(`${ref.pages}.`);
          if (ref.doi && this.options.includeDoi) parts.push(`https://doi.org/${ref.doi}`);
          
          return parts.join(' ');
        }
        
        formatMLA(ref) {
          const parts = [];
          
          if (ref.author) parts.push(`${ref.author}.`);
          if (ref.title) parts.push(`"${ref.title}."`);
          if (ref.journal) parts.push(`${ref.journal},`);
          if (ref.volume) parts.push(`vol. ${ref.volume},`);
          if (ref.number) parts.push(`no. ${ref.number},`);
          if (ref.year) parts.push(`${ref.year},`);
          if (ref.pages) parts.push(`pp. ${ref.pages}.`);
          
          return parts.join(' ');
        }
        
        formatChicago(ref) {
          const parts = [];
          
          if (ref.author) parts.push(`${ref.author}.`);
          if (ref.title) parts.push(`"${ref.title}."`);
          if (ref.journal) parts.push(`${ref.journal}`);
          if (ref.volume) parts.push(`${ref.volume},`);
          if (ref.number) parts.push(`no. ${ref.number}`);
          if (ref.year) parts.push(`(${ref.year}):`);
          if (ref.pages) parts.push(`${ref.pages}.`);
          
          return parts.join(' ');
        }
        
        // 辅助方法
        processAuthors(authors) {
          if (!authors) return 'Unknown';
          if (typeof authors === 'string') return authors;
          if (Array.isArray(authors)) {
            return authors.slice(0, this.options.maxAuthors).join(', ') +
                   (authors.length > this.options.maxAuthors ? ' et al.' : '');
          }
          return String(authors);
        }
        
        parseCitationKeys(citation) {
          if (typeof citation !== 'string') return [];
          
          // 解析 \cite{key1,key2,key3} 格式
          const match = citation.match(/\\cite\{([^}]+)\}/);
          if (match) {
            return match[1].split(',').map(key => key.trim());
          }
          
          // 直接的键列表
          return citation.split(',').map(key => key.trim()).filter(key => key.length > 0);
        }
        
        formatCitation(entry, style) {
          switch (style.toLowerCase()) {
            case 'numeric':
              return `[${this.getCitationNumber(entry.key)}]`;
            case 'author-year':
              return `(${entry.author || 'Unknown'}, ${entry.year || 'n.d.'})`;
            case 'author':
              return entry.author || 'Unknown';
            default:
              return `[${this.getCitationNumber(entry.key)}]`;
          }
        }
        
        getCitationNumber(key) {
          if (!this.citations.has(key)) {
            this.citations.set(key, this.citations.size + 1);
          }
          return this.citations.get(key);
        }
        
        generateCitationElement(entry, style, key) {
          const text = this.formatCitation(entry, style);
          const number = this.getCitationNumber(key);
          
          return `<a href="#ref-${key}" class="citation citation-${style}" data-key="${key}" data-number="${number}">${text}</a>`;
        }
        
        combineCitations(citations, style) {
          if (citations.length === 0) return '';
          if (citations.length === 1) return citations[0].text;
          
          const texts = citations.map(c => c.text);
          
          switch (style.toLowerCase()) {
            case 'numeric':
              // 合并数字引用 [1,2,3] 或 [1-3]
              const numbers = citations.map(c => this.getCitationNumber(c.key)).sort((a, b) => a - b);
              return `[${this.compressNumbers(numbers)}]`;
            default:
              return texts.join(', ');
          }
        }
        
        compressNumbers(numbers) {
          if (numbers.length <= 1) return numbers.join(',');
          
          const ranges = [];
          let start = numbers[0];
          let end = start;
          
          for (let i = 1; i < numbers.length; i++) {
            if (numbers[i] === end + 1) {
              end = numbers[i];
            } else {
              ranges.push(start === end ? `${start}` : `${start}-${end}`);
              start = end = numbers[i];
            }
          }
          ranges.push(start === end ? `${start}` : `${start}-${end}`);
          
          return ranges.join(',');
        }
        
        sortEntries(entries) {
          return entries.sort((a, b) => {
            switch (this.options.sortBy) {
              case 'year':
                return (b.entry.year || 0) - (a.entry.year || 0);
              case 'title':
                return (a.entry.title || '').localeCompare(b.entry.title || '');
              case 'author':
              default:
                return (a.entry.author || '').localeCompare(b.entry.author || '');
            }
          });
        }
        
        generateBibliographyHTML(entries) {
          if (entries.length === 0) {
            return '<div class="bibliography empty">No bibliography entries found.</div>';
          }
          
          return `
            <div class="bibliography">
              <h2 class="bibliography-title">References</h2>
              <ol class="bibliography-list">
                ${entries.map((entry, index) => `
                  <li class="bibliography-entry" id="ref-${entry.entry.key}" data-key="${entry.entry.key}">
                    ${entry.formatted}
                  </li>
                `).join('')}
              </ol>
            </div>
          `.trim();
        }
        
        generateCitationHTML(citations, style) {
          const elements = citations.map(c => c.element).join('');
          
          switch (style.toLowerCase()) {
            case 'numeric':
              if (citations.length > 1) {
                const numbers = citations.map(c => this.getCitationNumber(c.key));
                const compressed = this.compressNumbers(numbers.sort((a, b) => a - b));
                return `<span class="citations citations-numeric">[${compressed}]</span>`;
              }
              return `<span class="citations citations-numeric">${elements}</span>`;
            default:
              return `<span class="citations citations-${style}">${elements}</span>`;
          }
        }
        
        generateEntryHTML(entry) {
          const formatted = this.formatReference(entry, this.options.style);
          return `<div class="bib-entry" id="ref-${entry.key}" data-key="${entry.key}">${formatted}</div>`;
        }
        
        // 添加条目的便利方法
        addEntry(key, entryData) {
          const entry = { key, ...entryData };
          return this.generateBibEntry(entry);
        }
        
        process(input, options = {}) {
          try {
            let result;
            
            if (typeof input === 'string') {
              if (input.startsWith('\\cite{')) {
                result = this.processCitation(input, options.style);
              } else if (input.startsWith('\\bibliography{')) {
                // 处理 \bibliography{file} 命令
                result = { type: 'bibliography-file', file: input.match(/\\bibliography\{([^}]+)\}/)[1] };
              } else {
                // 假设是引用键
                result = this.processCitation(input, options.style);
              }
            } else if (input && typeof input === 'object') {
              if (Array.isArray(input)) {
                result = this.processBibliography(input);
              } else if (input.type === 'bibliography') {
                result = this.processBibliography(input.entries || input.data);
              } else {
                result = this.generateBibEntry(input);
              }
            } else {
              throw new Error('Invalid input type');
            }
            
            return {
              success: true,
              result,
              metadata: { input, options },
              warnings: [],
              errors: []
            };
          } catch (error) {
            return {
              success: false,
              result: null,
              metadata: { input, options },
              warnings: [],
              errors: [error.message]
            };
          }
        }
        
        validate(input) {
          const errors = [];
          const warnings = [];
          const suggestions = [];
          
          if (typeof input === 'string') {
            // 验证引用语法
            if (input.includes('\\cite{')) {
              const matches = input.match(/\\cite\{([^}]*)\}/g);
              if (matches) {
                matches.forEach(match => {
                  const keys = match.match(/\\cite\{([^}]*)\}/)[1];
                  if (!keys || keys.trim() === '') {
                    errors.push('Empty citation keys');
                    suggestions.push('Provide citation keys in \\cite{key1,key2} format');
                  } else {
                    const keyList = keys.split(',').map(k => k.trim());
                    keyList.forEach(key => {
                      if (!this.bibliography.has(key)) {
                        warnings.push(`Citation key '${key}' not found in bibliography`);
                        suggestions.push(`Add bibliography entry for '${key}'`);
                      }
                    });
                  }
                });
              }
            }
          } else if (input && typeof input === 'object') {
            if (Array.isArray(input)) {
              // 验证条目数组
              input.forEach((entry, index) => {
                if (!entry.key && !entry.id) {
                  errors.push(`Bibliography entry ${index} is missing key/id`);
                }
                if (!entry.title) {
                  warnings.push(`Bibliography entry ${index} is missing title`);
                }
                if (!entry.author && !entry.authors) {
                  warnings.push(`Bibliography entry ${index} is missing author`);
                }
                if (!entry.year && !entry.date) {
                  warnings.push(`Bibliography entry ${index} is missing year/date`);
                }
              });
            } else {
              // 验证单个条目
              if (!input.key && !input.id) {
                errors.push('Bibliography entry is missing key/id');
              }
              if (!input.title) {
                warnings.push('Bibliography entry is missing title');
                suggestions.push('Provide a meaningful title for the entry');
              }
              if (!input.author && !input.authors) {
                warnings.push('Bibliography entry is missing author');
                suggestions.push('Provide author information for the entry');
              }
            }
          } else if (typeof input !== 'string') {
            errors.push('Input must be a string, object, or array');
          }
          
          return {
            valid: errors.length === 0,
            errors,
            warnings,
            suggestions
          };
        }
        
        getConfig() {
          return this.options;
        }
      };
    }
  });

  beforeEach(() => {
    bibliographyProcessor = new BibliographyProcessor();
    
    // 添加一些测试数据
    bibliographyProcessor.addEntry('smith2020', {
      type: 'article',
      title: 'A Comprehensive Study of LaTeX Processing',
      author: 'Smith, J. and Doe, A.',
      journal: 'Journal of Document Processing',
      volume: '15',
      number: '3',
      pages: '123-145',
      year: '2020',
      doi: '10.1234/jdp.2020.15.3.123'
    });
    
    bibliographyProcessor.addEntry('jones2021', {
      type: 'book',
      title: 'Modern Bibliography Management',
      author: 'Jones, M.',
      publisher: 'Academic Press',
      address: 'New York',
      year: '2021',
      isbn: '978-0-123456-78-9'
    });
  });

  describe('构造函数和配置', () => {
    test('应该使用默认配置创建实例', () => {
      const processor = new BibliographyProcessor();
      const config = processor.getConfig();
      
      expect(config).toHaveProperty('style', 'ieee');
      expect(config).toHaveProperty('sortBy', 'author');
      expect(config).toHaveProperty('includeUrls', true);
      expect(config).toHaveProperty('includeDoi', true);
      expect(config).toHaveProperty('maxAuthors', 6);
      expect(config).toHaveProperty('abbreviateJournals', false);
    });

    test('应该接受自定义配置选项', () => {
      const customOptions = {
        style: 'apa',
        sortBy: 'year',
        includeUrls: false,
        maxAuthors: 3
      };
      
      const processor = new BibliographyProcessor(customOptions);
      const config = processor.getConfig();
      
      expect(config.style).toBe('apa');
      expect(config.sortBy).toBe('year');
      expect(config.includeUrls).toBe(false);
      expect(config.maxAuthors).toBe(3);
    });
  });

  describe('processBibliography', () => {
    test('应该处理基本的文献列表', () => {
      const bibData = [
        {
          key: 'test1',
          title: 'Test Article 1',
          author: 'Author A',
          year: '2020'
        },
        {
          key: 'test2',
          title: 'Test Article 2',
          author: 'Author B',
          year: '2021'
        }
      ];
      
      const result = bibliographyProcessor.processBibliography(bibData);
      
      expect(result).toHaveProperty('type', 'bibliography');
      expect(result).toHaveProperty('entries');
      expect(result).toHaveProperty('total');
      expect(result).toHaveProperty('style');
      expect(result).toHaveProperty('html');
      
      expect(result.entries.length).toBe(2);
      expect(result.total).toBe(2);
    });

    test('应该处理空的文献数据', () => {
      const result = bibliographyProcessor.processBibliography([]);
      
      expect(result.total).toBe(0);
      expect(result.entries.length).toBe(0);
      expect(result.html).toContain('empty');
    });

    test('应该生成正确的HTML结构', () => {
      const bibData = [{
        key: 'test1',
        title: 'Test Article',
        author: 'Test Author',
        year: '2020'
      }];
      
      const result = bibliographyProcessor.processBibliography(bibData);
      
      expect(result.html).toContain('class="bibliography"');
      expect(result.html).toContain('bibliography-title');
      expect(result.html).toContain('bibliography-list');
      expect(result.html).toContain('bibliography-entry');
    });

    test('应该根据配置排序条目', () => {
      const bibData = [
        { key: 'b', author: 'B Author', title: 'B Title', year: '2020' },
        { key: 'a', author: 'A Author', title: 'A Title', year: '2021' },
        { key: 'c', author: 'C Author', title: 'C Title', year: '2019' }
      ];
      
      // 按作者排序
      const authorSorted = new BibliographyProcessor({ sortBy: 'author' });
      const result1 = authorSorted.processBibliography(bibData);
      expect(result1.entries[0].entry.author).toBe('A Author');
      
      // 按年份排序
      const yearSorted = new BibliographyProcessor({ sortBy: 'year' });
      const result2 = yearSorted.processBibliography(bibData);
      expect(result2.entries[0].entry.year).toBe('2021');
    });
  });

  describe('processCitation', () => {
    test('应该处理单个引用', () => {
      const citation = 'smith2020';
      const result = bibliographyProcessor.processCitation(citation);
      
      expect(result).toHaveProperty('type', 'citation');
      expect(result).toHaveProperty('style');
      expect(result).toHaveProperty('keys');
      expect(result).toHaveProperty('citations');
      expect(result).toHaveProperty('formatted');
      expect(result).toHaveProperty('html');
      
      expect(result.keys).toEqual(['smith2020']);
      expect(result.citations[0].found).toBe(true);
    });

    test('应该处理多个引用', () => {
      const citation = 'smith2020,jones2021';
      const result = bibliographyProcessor.processCitation(citation);
      
      expect(result.keys).toEqual(['smith2020', 'jones2021']);
      expect(result.citations).toHaveLength(2);
      expect(result.citations[0].found).toBe(true);
      expect(result.citations[1].found).toBe(true);
    });

    test('应该处理LaTeX引用语法', () => {
      const citation = '\\cite{smith2020,jones2021}';
      const result = bibliographyProcessor.processCitation(citation);
      
      expect(result.keys).toEqual(['smith2020', 'jones2021']);
      expect(result.citations).toHaveLength(2);
    });

    test('应该处理不存在的引用', () => {
      const citation = 'nonexistent';
      const result = bibliographyProcessor.processCitation(citation);
      
      expect(result.citations[0].found).toBe(false);
      expect(result.citations[0].text).toContain('nonexistent');
      expect(result.citations[0].element).toContain('citation-missing');
    });

    test('应该支持不同的引用样式', () => {
      const citation = 'smith2020';
      
      const numericResult = bibliographyProcessor.processCitation(citation, 'numeric');
      const authorYearResult = bibliographyProcessor.processCitation(citation, 'author-year');
      const authorResult = bibliographyProcessor.processCitation(citation, 'author');
      
      expect(numericResult.citations[0].text).toMatch(/^\[\d+\]$/);
      expect(authorYearResult.citations[0].text).toContain('Smith, J. and Doe, A.');
      expect(authorYearResult.citations[0].text).toContain('2020');
      expect(authorResult.citations[0].text).toBe('Smith, J. and Doe, A.');
    });
  });

  describe('generateBibEntry', () => {
    test('应该生成基本的文献条目', () => {
      const entry = {
        key: 'test2023',
        type: 'article',
        title: 'Test Article',
        author: 'Test Author',
        journal: 'Test Journal',
        year: '2023'
      };
      
      const result = bibliographyProcessor.generateBibEntry(entry);
      
      expect(result).toHaveProperty('type', 'bib-entry');
      expect(result).toHaveProperty('entry');
      expect(result).toHaveProperty('formatted');
      expect(result).toHaveProperty('html');
      
      expect(result.entry.key).toBe('test2023');
      expect(result.entry.title).toBe('Test Article');
    });

    test('应该处理缺失字段', () => {
      const entry = {
        key: 'minimal',
        title: 'Minimal Entry'
      };
      
      const result = bibliographyProcessor.generateBibEntry(entry);
      
      expect(result.entry.author).toBe('Unknown');
      expect(result.entry.type).toBe('article');
      expect(result.formatted).toContain('Minimal Entry');
    });

    test('应该处理多个作者', () => {
      const entry = {
        key: 'multi-author',
        title: 'Multi-Author Paper',
        author: ['Author A', 'Author B', 'Author C', 'Author D']
      };
      
      const result = bibliographyProcessor.generateBibEntry(entry);
      expect(result.entry.author).toContain('Author A');
      expect(result.entry.author).toContain('Author B');
    });

    test('应该限制作者数量', () => {
      const processor = new BibliographyProcessor({ maxAuthors: 2 });
      const entry = {
        key: 'many-authors',
        title: 'Many Authors Paper',
        author: ['A1', 'A2', 'A3', 'A4', 'A5']
      };
      
      const result = processor.generateBibEntry(entry);
      expect(result.entry.author).toContain('et al.');
    });

    test('应该抛出无效条目错误', () => {
      expect(() => {
        bibliographyProcessor.generateBibEntry(null);
      }).toThrow('Invalid bibliography entry');
    });
  });

  describe('formatReference', () => {
    const testEntry = {
      key: 'test2023',
      author: 'Smith, J.',
      title: 'Test Article',
      journal: 'Test Journal',
      volume: '10',
      number: '2',
      pages: '25-40',
      year: '2023',
      doi: '10.1234/test.2023.10.2.25'
    };

    test('应该使用IEEE格式', () => {
      const result = bibliographyProcessor.formatReference(testEntry, 'ieee');
      
      expect(result).toContain('Smith, J.');
      expect(result).toContain('"Test Article"');
      expect(result).toContain('Test Journal');
      expect(result).toContain('vol. 10');
      expect(result).toContain('pp. 25-40');
      expect(result).toContain('2023');
    });

    test('应该使用APA格式', () => {
      const result = bibliographyProcessor.formatReference(testEntry, 'apa');
      
      expect(result).toContain('Smith, J.');
      expect(result).toContain('(2023)');
      expect(result).toContain('Test Article');
      expect(result).toContain('Test Journal');
    });

    test('应该使用MLA格式', () => {
      const result = bibliographyProcessor.formatReference(testEntry, 'mla');
      
      expect(result).toContain('Smith, J.');
      expect(result).toContain('"Test Article"');
      expect(result).toContain('Test Journal');
      expect(result).toContain('2023');
    });

    test('应该使用Chicago格式', () => {
      const result = bibliographyProcessor.formatReference(testEntry, 'chicago');
      
      expect(result).toContain('Smith, J.');
      expect(result).toContain('"Test Article"');
      expect(result).toContain('Test Journal');
      expect(result).toContain('(2023)');
    });

    test('应该包含DOI', () => {
      const processor = new BibliographyProcessor({ includeDoi: true });
      const result = processor.formatReference(testEntry, 'ieee');
      
      expect(result).toContain('DOI: 10.1234/test.2023.10.2.25');
    });

    test('应该排除DOI', () => {
      const processor = new BibliographyProcessor({ includeDoi: false });
      const result = processor.formatReference(testEntry, 'ieee');
      
      expect(result).not.toContain('DOI:');
    });
  });

  describe('辅助方法', () => {
    test('parseCitationKeys应该正确解析引用键', () => {
      const testCases = [
        { input: '\\cite{key1}', expected: ['key1'] },
        { input: '\\cite{key1,key2,key3}', expected: ['key1', 'key2', 'key3'] },
        { input: 'key1,key2', expected: ['key1', 'key2'] },
        { input: 'single', expected: ['single'] }
      ];
      
      testCases.forEach(testCase => {
        const result = bibliographyProcessor.parseCitationKeys(testCase.input);
        expect(result).toEqual(testCase.expected);
      });
    });

    test('getCitationNumber应该分配递增的编号', () => {
      const num1 = bibliographyProcessor.getCitationNumber('key1');
      const num2 = bibliographyProcessor.getCitationNumber('key2');
      const num1Again = bibliographyProcessor.getCitationNumber('key1');
      
      expect(num1).toBe(1);
      expect(num2).toBe(2);
      expect(num1Again).toBe(1); // 应该返回相同编号
    });

    test('compressNumbers应该压缩连续数字', () => {
      const testCases = [
        { input: [1, 2, 3], expected: '1-3' },
        { input: [1, 3, 5], expected: '1,3,5' },
        { input: [1, 2, 4, 5, 6], expected: '1-2,4-6' },
        { input: [1], expected: '1' }
      ];
      
      testCases.forEach(testCase => {
        const result = bibliographyProcessor.compressNumbers(testCase.input);
        expect(result).toBe(testCase.expected);
      });
    });

    test('processAuthors应该处理不同的作者格式', () => {
      const testCases = [
        { input: 'Single Author', expected: 'Single Author' },
        { input: ['A1', 'A2'], expected: 'A1, A2' },
        { input: null, expected: 'Unknown' },
        { input: ['A1', 'A2', 'A3', 'A4', 'A5', 'A6', 'A7'], expected: 'A1, A2, A3, A4, A5, A6 et al.' }
      ];
      
      testCases.forEach(testCase => {
        const result = bibliographyProcessor.processAuthors(testCase.input);
        expect(result).toBe(testCase.expected);
      });
    });
  });

  describe('BaseProcessor接口实现', () => {
    test('应该实现process方法', () => {
      const input = '\\cite{smith2020}';
      const result = bibliographyProcessor.process(input);
      
      expect(result).toHaveProperty('success');
      expect(result).toHaveProperty('result');
      expect(result).toHaveProperty('metadata');
      expect(result).toHaveProperty('warnings');
      expect(result).toHaveProperty('errors');
      
      expect(typeof result.success).toBe('boolean');
      expect(Array.isArray(result.warnings)).toBe(true);
      expect(Array.isArray(result.errors)).toBe(true);
    });

    test('应该实现validate方法', () => {
      const input = '\\cite{smith2020}';
      const result = bibliographyProcessor.validate(input);
      
      expect(result).toHaveProperty('valid');
      expect(result).toHaveProperty('errors');
      expect(result).toHaveProperty('warnings');
      expect(result).toHaveProperty('suggestions');
      
      expect(typeof result.valid).toBe('boolean');
      expect(Array.isArray(result.errors)).toBe(true);
      expect(Array.isArray(result.warnings)).toBe(true);
      expect(Array.isArray(result.suggestions)).toBe(true);
    });

    test('应该实现getConfig方法', () => {
      const config = bibliographyProcessor.getConfig();
      
      expect(config).toBeDefined();
      expect(typeof config).toBe('object');
      expect(config).toHaveProperty('style');
      expect(config).toHaveProperty('sortBy');
      expect(config).toHaveProperty('includeDoi');
    });
  });

  describe('输入验证', () => {
    test('应该验证有效的引用语法', () => {
      const validInputs = [
        '\\cite{valid}',
        '\\cite{key1,key2}',
        'key1',
        'key1,key2'
      ];
      
      validInputs.forEach(input => {
        const result = bibliographyProcessor.validate(input);
        expect(result).toHaveProperty('valid');
      });
    });

    test('应该检测空的引用键', () => {
      const input = '\\cite{}';
      const result = bibliographyProcessor.validate(input);
      
      expect(result.valid).toBe(false);
      expect(result.errors).toContain('Empty citation keys');
    });

    test('应该检测缺失的引用', () => {
      const input = '\\cite{nonexistent}';
      const result = bibliographyProcessor.validate(input);
      
      expect(result.warnings.some(w => w.includes('not found'))).toBe(true);
    });

    test('应该验证文献条目', () => {
      const validEntry = {
        key: 'test',
        title: 'Test Title',
        author: 'Test Author',
        year: '2023'
      };
      
      const invalidEntry = {};
      
      const result1 = bibliographyProcessor.validate(validEntry);
      const result2 = bibliographyProcessor.validate(invalidEntry);
      
      expect(result1.valid).toBe(true);
      expect(result2.valid).toBe(false);
    });

    test('应该检测缺失的必要字段', () => {
      const entryWithoutTitle = {
        key: 'test',
        author: 'Author'
      };
      
      const result = bibliographyProcessor.validate(entryWithoutTitle);
      
      expect(result.warnings.some(w => w.includes('missing title'))).toBe(true);
    });

    test('应该验证条目数组', () => {
      const entries = [
        { key: 'valid1', title: 'Title1', author: 'Author1' },
        { title: 'No Key' }, // 缺少key
        { key: 'valid2', author: 'Author2' } // 缺少title
      ];
      
      const result = bibliographyProcessor.validate(entries);
      
      expect(result.errors.some(e => e.includes('missing key'))).toBe(true);
      expect(result.warnings.some(w => w.includes('missing title'))).toBe(true);
    });
  });

  describe('错误处理', () => {
    test('应该优雅处理无效输入', () => {
      const result = bibliographyProcessor.process(null);
      
      expect(result).toHaveProperty('success', false);
      expect(result).toHaveProperty('errors');
      expect(result.errors.length).toBeGreaterThan(0);
    });

    test('应该处理无效条目生成', () => {
      const result = bibliographyProcessor.process({ invalidEntry: true });
      
      if (!result.success) {
        expect(result.errors.length).toBeGreaterThan(0);
      }
    });

    test('应该提供有用的错误信息', () => {
      const result = bibliographyProcessor.validate('\\cite{}');
      
      if (!result.valid) {
        expect(result.errors.length).toBeGreaterThan(0);
        expect(result.suggestions.length).toBeGreaterThan(0);
      }
    });
  });

  describe('集成测试', () => {
    test('应该处理完整的文献管理工作流', () => {
      // 1. 添加文献条目
      const entry1 = bibliographyProcessor.addEntry('test2023', {
        title: 'Test Article 2023',
        author: 'Test Author',
        year: '2023'
      });
      
      expect(entry1.entry.key).toBe('test2023');
      
      // 2. 处理引用
      const citation = bibliographyProcessor.processCitation('test2023,smith2020');
      expect(citation.citations).toHaveLength(2);
      expect(citation.citations.every(c => c.found)).toBe(true);
      
      // 3. 生成文献列表
      const bibData = Array.from(bibliographyProcessor.bibliography.values());
      const bibliography = bibliographyProcessor.processBibliography(bibData);
      
      expect(bibliography.total).toBeGreaterThan(0);
      expect(bibliography.html).toContain('bibliography-list');
    });

    test('应该支持不同样式的引用和格式', () => {
      const styles = ['ieee', 'apa', 'mla', 'chicago'];
      
      styles.forEach(style => {
        const processor = new BibliographyProcessor({ style });
        
        const citation = processor.processCitation('smith2020', style);
        const formatted = processor.formatReference(
          processor.bibliography.get('smith2020'),
          style
        );
        
        expect(citation.style).toBe(style);
        expect(typeof formatted).toBe('string');
        expect(formatted.length).toBeGreaterThan(0);
      });
    });

    test('应该保持引用编号的一致性', () => {
      const citation1 = bibliographyProcessor.processCitation('smith2020');
      const citation2 = bibliographyProcessor.processCitation('jones2021');
      const citation3 = bibliographyProcessor.processCitation('smith2020'); // 重复引用
      
      const num1 = bibliographyProcessor.getCitationNumber('smith2020');
      const num2 = bibliographyProcessor.getCitationNumber('jones2021');
      const num3 = bibliographyProcessor.getCitationNumber('smith2020');
      
      expect(num1).toBe(num3); // 相同引用应该有相同编号
      expect(num1).not.toBe(num2); // 不同引用应该有不同编号
    });

    test('应该正确处理复杂的多引用格式', () => {
      const multiCitation = 'smith2020,jones2021,smith2020'; // 包含重复
      const result = bibliographyProcessor.processCitation(multiCitation, 'numeric');
      
      expect(result.formatted).toBeDefined();
      expect(result.html).toContain('citations-numeric');
      
      // 应该合并重复的引用
      const uniqueKeys = [...new Set(result.keys)];
      expect(uniqueKeys).toHaveLength(2);
    });
  });
});