/**
 * AdvancedEnvironmentProcessor 测试套件
 * 基于 interface.md 规范测试复杂LaTeX环境处理
 */

describe('AdvancedEnvironmentProcessor', () => {
  let AdvancedEnvironmentProcessor;
  let advancedEnvProcessor;

  beforeAll(() => {
    try {
      AdvancedEnvironmentProcessor = require('../AdvancedEnvironmentProcessor');
    } catch (e) {
      // Mock AdvancedEnvironmentProcessor for testing interface
      AdvancedEnvironmentProcessor = class AdvancedEnvironmentProcessor {
        constructor(options = {}) {
          this.options = {
            enableNumbering: true,
            includeProofs: true,
            theoremStyle: 'definition',
            language: 'en',
            customEnvironments: {},
            ...options
          };
          this.counters = {
            theorem: 0,
            definition: 0,
            lemma: 0,
            proposition: 0,
            corollary: 0,
            example: 0,
            proof: 0
          };
          this.theoremTypes = [
            'theorem', 'lemma', 'proposition', 'corollary',
            'definition', 'example', 'remark', 'proof'
          ];
        }
        
        processTheorem(content, theoremType = 'theorem') {
          if (!this.theoremTypes.includes(theoremType)) {
            theoremType = 'theorem';
          }
          
          const number = this.options.enableNumbering ? 
            ++this.counters[theoremType] : null;
          
          const theorem = {
            type: theoremType,
            content: content.trim(),
            number,
            id: `${theoremType}-${number || Date.now()}`,
            title: this.getTheoremTitle(theoremType, number),
            label: null,
            style: this.options.theoremStyle
          };
          
          // 解析可能的标题和标签
          const parsed = this.parseTheoremContent(content);
          if (parsed.title) theorem.title += ` (${parsed.title})`;
          if (parsed.label) theorem.label = parsed.label;
          if (parsed.content) theorem.content = parsed.content;
          
          return {
            type: 'theorem-environment',
            theoremType,
            theorem,
            element: {
              tag: 'div',
              class: `theorem ${theoremType}`,
              id: theorem.id,
              content: theorem.content
            },
            html: this.generateTheoremHTML(theorem)
          };
        }
        
        processProof(content) {
          const proofNumber = this.options.enableNumbering ? 
            ++this.counters.proof : null;
          
          const proof = {
            type: 'proof',
            content: content.trim(),
            number: proofNumber,
            id: `proof-${proofNumber || Date.now()}`,
            title: 'Proof',
            qed: true, // 是否显示QED符号
            style: this.options.theoremStyle
          };
          
          // 解析内容和标签
          const parsed = this.parseProofContent(content);
          if (parsed.title) proof.title = parsed.title;
          if (parsed.content) proof.content = parsed.content;
          if (parsed.qed !== undefined) proof.qed = parsed.qed;
          
          return {
            type: 'proof-environment',
            proof,
            element: {
              tag: 'div',
              class: 'proof',
              id: proof.id,
              content: proof.content
            },
            html: this.generateProofHTML(proof)
          };
        }
        
        processDefinition(content) {
          return this.processTheorem(content, 'definition');
        }
        
        processExample(content) {
          return this.processTheorem(content, 'example');
        }
        
        // 辅助方法
        getTheoremTitle(type, number) {
          const titles = {
            theorem: 'Theorem',
            lemma: 'Lemma',
            proposition: 'Proposition',
            corollary: 'Corollary',
            definition: 'Definition',
            example: 'Example',
            remark: 'Remark',
            proof: 'Proof'
          };
          
          const title = titles[type] || 'Theorem';
          return number ? `${title} ${number}` : title;
        }
        
        parseTheoremContent(content) {
          let parsedContent = content;
          let title = null;
          let label = null;
          
          // 解析标题 [Title]
          const titleMatch = parsedContent.match(/^\[([^\]]+)\]/);
          if (titleMatch) {
            title = titleMatch[1];
            parsedContent = parsedContent.replace(titleMatch[0], '').trim();
          }
          
          // 解析标签 \label{...}
          const labelMatch = parsedContent.match(/\\label\{([^}]+)\}/);
          if (labelMatch) {
            label = labelMatch[1];
            parsedContent = parsedContent.replace(labelMatch[0], '').trim();
          }
          
          return {
            content: parsedContent,
            title,
            label
          };
        }
        
        parseProofContent(content) {
          let parsedContent = content;
          let title = 'Proof';
          let qed = true;
          
          // 解析自定义证明标题
          const titleMatch = parsedContent.match(/^\[([^\]]+)\]/);
          if (titleMatch) {
            title = titleMatch[1];
            parsedContent = parsedContent.replace(titleMatch[0], '').trim();
          }
          
          // 检查是否禁用QED符号
          if (parsedContent.includes('\\noqed')) {
            qed = false;
            parsedContent = parsedContent.replace(/\\noqed/, '').trim();
          }
          
          return {
            content: parsedContent,
            title,
            qed
          };
        }
        
        generateTheoremHTML(theorem) {
          return `
            <div class="theorem-environment ${theorem.type}" id="${theorem.id}">
              <div class="theorem-header">
                <span class="theorem-title">${theorem.title}</span>
                ${theorem.label ? `<span class="theorem-label" id="${theorem.label}"></span>` : ''}
              </div>
              <div class="theorem-content">
                ${theorem.content}
              </div>
            </div>
          `.trim();
        }
        
        generateProofHTML(proof) {
          return `
            <div class="proof-environment" id="${proof.id}">
              <div class="proof-header">
                <span class="proof-title">${proof.title}</span>
              </div>
              <div class="proof-content">
                ${proof.content}
                ${proof.qed ? '<span class="qed-symbol">□</span>' : ''}
              </div>
            </div>
          `.trim();
        }
        
        // 处理LaTeX环境语法
        processEnvironment(envType, content, options = {}) {
          switch (envType.toLowerCase()) {
            case 'theorem':
            case 'lemma':
            case 'proposition':
            case 'corollary':
              return this.processTheorem(content, envType);
            case 'definition':
              return this.processDefinition(content);
            case 'example':
              return this.processExample(content);
            case 'proof':
              return this.processProof(content);
            default:
              // 处理自定义环境
              return this.processCustomEnvironment(envType, content, options);
          }
        }
        
        processCustomEnvironment(envType, content, options) {
          if (this.options.customEnvironments[envType]) {
            const config = this.options.customEnvironments[envType];
            return this.processTheorem(content, envType);
          }
          
          return {
            type: 'unknown-environment',
            envType,
            content,
            element: {
              tag: 'div',
              class: `environment ${envType}`,
              content
            },
            html: `<div class="environment ${envType}">${content}</div>`
          };
        }
        
        // 计数器管理
        resetCounters() {
          Object.keys(this.counters).forEach(key => {
            this.counters[key] = 0;
          });
        }
        
        getCounter(type) {
          return this.counters[type] || 0;
        }
        
        setCounter(type, value) {
          if (this.counters.hasOwnProperty(type)) {
            this.counters[type] = value;
          }
        }
        
        process(input, options = {}) {
          try {
            let result;
            
            if (typeof input === 'string') {
              // 检测LaTeX环境
              const envMatch = input.match(/\\begin\{(\w+)\}(.*?)\\end\{\1\}/s);
              if (envMatch) {
                const envType = envMatch[1];
                const content = envMatch[2].trim();
                result = this.processEnvironment(envType, content, options);
              } else if (input.startsWith('\\begin{')) {
                // 不完整的环境
                throw new Error('Incomplete environment syntax');
              } else {
                // 假设是定理内容
                const theoremType = options.type || 'theorem';
                result = this.processTheorem(input, theoremType);
              }
            } else if (input && typeof input === 'object') {
              // 处理环境对象
              if (input.type && input.content) {
                result = this.processEnvironment(input.type, input.content, options);
              } else {
                throw new Error('Invalid environment object');
              }
            } else {
              throw new Error('Invalid input type');
            }
            
            return {
              success: true,
              result,
              metadata: { input, options },
              warnings: [],
              errors: []
            };
          } catch (error) {
            return {
              success: false,
              result: null,
              metadata: { input, options },
              warnings: [],
              errors: [error.message]
            };
          }
        }
        
        validate(input) {
          const errors = [];
          const warnings = [];
          const suggestions = [];
          
          if (typeof input === 'string') {
            // 验证LaTeX环境语法
            const envMatches = input.match(/\\begin\{(\w+)\}/g);
            const endMatches = input.match(/\\end\{(\w+)\}/g);
            
            if (envMatches && endMatches) {
              if (envMatches.length !== endMatches.length) {
                errors.push('Mismatched \\begin and \\end commands');
                suggestions.push('Ensure every \\begin{env} has a corresponding \\end{env}');
              }
              
              // 检查环境名称匹配
              envMatches.forEach((beginMatch, index) => {
                const beginEnv = beginMatch.match(/\\begin\{(\w+)\}/)[1];
                if (endMatches[index]) {
                  const endEnv = endMatches[index].match(/\\end\{(\w+)\}/)[1];
                  if (beginEnv !== endEnv) {
                    errors.push(`Environment mismatch: \\begin{${beginEnv}} with \\end{${endEnv}}`);
                  }
                }
              });
            }
            
            // 检查环境类型
            if (envMatches) {
              envMatches.forEach(match => {
                const envType = match.match(/\\begin\{(\w+)\}/)[1];
                if (!this.theoremTypes.includes(envType) && 
                    !this.options.customEnvironments[envType]) {
                  warnings.push(`Unknown environment type: ${envType}`);
                  suggestions.push(`Define custom environment '${envType}' or use standard types`);
                }
              });
            }
            
            // 检查空内容
            const contentMatch = input.match(/\\begin\{\w+\}(.*?)\\end\{\w+\}/s);
            if (contentMatch && !contentMatch[1].trim()) {
              warnings.push('Environment has empty content');
              suggestions.push('Provide meaningful content for the environment');
            }
          } else if (input && typeof input === 'object') {
            // 验证环境对象
            if (!input.type) {
              errors.push('Environment object is missing type');
            }
            if (!input.content) {
              warnings.push('Environment object has no content');
            }
          } else if (typeof input !== 'string') {
            errors.push('Input must be a string or environment object');
          }
          
          return {
            valid: errors.length === 0,
            errors,
            warnings,
            suggestions
          };
        }
        
        getConfig() {
          return this.options;
        }
      };
    }
  });

  beforeEach(() => {
    advancedEnvProcessor = new AdvancedEnvironmentProcessor();
  });

  describe('构造函数和配置', () => {
    test('应该使用默认配置创建实例', () => {
      const processor = new AdvancedEnvironmentProcessor();
      const config = processor.getConfig();
      
      expect(config).toHaveProperty('enableNumbering', true);
      expect(config).toHaveProperty('includeProofs', true);
      expect(config).toHaveProperty('theoremStyle', 'definition');
      expect(config).toHaveProperty('language', 'en');
      expect(config).toHaveProperty('customEnvironments');
      
      expect(typeof config.customEnvironments).toBe('object');
    });

    test('应该接受自定义配置选项', () => {
      const customOptions = {
        enableNumbering: false,
        theoremStyle: 'plain',
        language: 'zh',
        customEnvironments: {
          'custom': { style: 'italic' }
        }
      };
      
      const processor = new AdvancedEnvironmentProcessor(customOptions);
      const config = processor.getConfig();
      
      expect(config.enableNumbering).toBe(false);
      expect(config.theoremStyle).toBe('plain');
      expect(config.language).toBe('zh');
      expect(config.customEnvironments).toHaveProperty('custom');
    });
  });

  describe('processTheorem', () => {
    test('应该处理基本定理', () => {
      const content = 'This is a theorem statement.';
      const result = advancedEnvProcessor.processTheorem(content);
      
      expect(result).toHaveProperty('type', 'theorem-environment');
      expect(result).toHaveProperty('theoremType', 'theorem');
      expect(result).toHaveProperty('theorem');
      expect(result).toHaveProperty('element');
      expect(result).toHaveProperty('html');
      
      expect(result.theorem.content).toBe(content);
      expect(result.theorem.type).toBe('theorem');
      expect(result.theorem.number).toBe(1);
      expect(result.theorem.title).toContain('Theorem 1');
    });

    test('应该处理不同类型的定理', () => {
      const theoremTypes = ['theorem', 'lemma', 'proposition', 'corollary', 'definition'];
      
      theoremTypes.forEach(type => {
        const result = advancedEnvProcessor.processTheorem('Content', type);
        
        expect(result.theoremType).toBe(type);
        expect(result.theorem.type).toBe(type);
        expect(result.theorem.title).toContain(type.charAt(0).toUpperCase() + type.slice(1));
      });
    });

    test('应该处理带标题的定理', () => {
      const content = '[Important Theorem] This theorem is very important.';
      const result = advancedEnvProcessor.processTheorem(content);
      
      expect(result.theorem.title).toContain('Important Theorem');
      expect(result.theorem.content).toBe('This theorem is very important.');
    });

    test('应该处理带标签的定理', () => {
      const content = 'This is a theorem. \\label{thm:important}';
      const result = advancedEnvProcessor.processTheorem(content);
      
      expect(result.theorem.label).toBe('thm:important');
      expect(result.theorem.content).toBe('This is a theorem.');
    });

    test('应该递增计数器', () => {
      advancedEnvProcessor.processTheorem('First theorem');
      const result2 = advancedEnvProcessor.processTheorem('Second theorem');
      const result3 = advancedEnvProcessor.processTheorem('Third theorem');
      
      expect(result2.theorem.number).toBe(2);
      expect(result3.theorem.number).toBe(3);
    });

    test('应该处理无效的定理类型', () => {
      const result = advancedEnvProcessor.processTheorem('Content', 'invalid');
      
      expect(result.theoremType).toBe('theorem'); // 应该回退到默认类型
    });

    test('应该生成正确的HTML', () => {
      const result = advancedEnvProcessor.processTheorem('Test theorem');
      
      expect(result.html).toContain('class="theorem-environment theorem"');
      expect(result.html).toContain('theorem-title');
      expect(result.html).toContain('theorem-content');
      expect(result.html).toContain('Test theorem');
    });
  });

  describe('processProof', () => {
    test('应该处理基本证明', () => {
      const content = 'This is the proof of the theorem.';
      const result = advancedEnvProcessor.processProof(content);
      
      expect(result).toHaveProperty('type', 'proof-environment');
      expect(result).toHaveProperty('proof');
      expect(result).toHaveProperty('element');
      expect(result).toHaveProperty('html');
      
      expect(result.proof.content).toBe(content);
      expect(result.proof.type).toBe('proof');
      expect(result.proof.title).toBe('Proof');
      expect(result.proof.qed).toBe(true);
    });

    test('应该处理自定义证明标题', () => {
      const content = '[Proof of Theorem 1] The proof goes here.';
      const result = advancedEnvProcessor.processProof(content);
      
      expect(result.proof.title).toBe('Proof of Theorem 1');
      expect(result.proof.content).toBe('The proof goes here.');
    });

    test('应该处理QED符号控制', () => {
      const contentWithQED = 'Proof with QED symbol.';
      const contentWithoutQED = 'Proof without QED. \\noqed';
      
      const result1 = advancedEnvProcessor.processProof(contentWithQED);
      const result2 = advancedEnvProcessor.processProof(contentWithoutQED);
      
      expect(result1.proof.qed).toBe(true);
      expect(result2.proof.qed).toBe(false);
      expect(result2.proof.content).toBe('Proof without QED.');
    });

    test('应该生成正确的HTML', () => {
      const result = advancedEnvProcessor.processProof('Test proof');
      
      expect(result.html).toContain('class="proof-environment"');
      expect(result.html).toContain('proof-title');
      expect(result.html).toContain('proof-content');
      expect(result.html).toContain('Test proof');
      expect(result.html).toContain('qed-symbol');
      expect(result.html).toContain('□');
    });

    test('应该在禁用QED时不显示符号', () => {
      const result = advancedEnvProcessor.processProof('Proof \\noqed');
      
      expect(result.html).not.toContain('qed-symbol');
      expect(result.html).not.toContain('□');
    });
  });

  describe('processDefinition', () => {
    test('应该处理定义', () => {
      const content = 'A function is continuous if...';
      const result = advancedEnvProcessor.processDefinition(content);
      
      expect(result.type).toBe('theorem-environment');
      expect(result.theoremType).toBe('definition');
      expect(result.theorem.type).toBe('definition');
      expect(result.theorem.title).toContain('Definition');
    });

    test('应该使用独立的计数器', () => {
      advancedEnvProcessor.processTheorem('Theorem 1');
      const def1 = advancedEnvProcessor.processDefinition('Definition 1');
      const def2 = advancedEnvProcessor.processDefinition('Definition 2');
      
      expect(def1.theorem.number).toBe(1);
      expect(def2.theorem.number).toBe(2);
    });
  });

  describe('processExample', () => {
    test('应该处理示例', () => {
      const content = 'Consider the function f(x) = x^2.';
      const result = advancedEnvProcessor.processExample(content);
      
      expect(result.type).toBe('theorem-environment');
      expect(result.theoremType).toBe('example');
      expect(result.theorem.type).toBe('example');
      expect(result.theorem.title).toContain('Example');
    });

    test('应该使用独立的计数器', () => {
      const ex1 = advancedEnvProcessor.processExample('Example 1');
      const ex2 = advancedEnvProcessor.processExample('Example 2');
      
      expect(ex1.theorem.number).toBe(1);
      expect(ex2.theorem.number).toBe(2);
    });
  });

  describe('processEnvironment', () => {
    test('应该正确路由到相应的处理方法', () => {
      const environments = [
        { type: 'theorem', expected: 'theorem-environment' },
        { type: 'proof', expected: 'proof-environment' },
        { type: 'definition', expected: 'theorem-environment' },
        { type: 'example', expected: 'theorem-environment' }
      ];
      
      environments.forEach(env => {
        const result = advancedEnvProcessor.processEnvironment(env.type, 'Content');
        expect(result.type).toBe(env.expected);
      });
    });

    test('应该处理自定义环境', () => {
      const processor = new AdvancedEnvironmentProcessor({
        customEnvironments: {
          'mytheorem': { style: 'custom' }
        }
      });
      
      const result = processor.processEnvironment('mytheorem', 'Custom content');
      expect(result.type).toBe('theorem-environment');
      expect(result.theoremType).toBe('mytheorem');
    });

    test('应该处理未知环境', () => {
      const result = advancedEnvProcessor.processEnvironment('unknown', 'Content');
      
      expect(result.type).toBe('unknown-environment');
      expect(result.envType).toBe('unknown');
      expect(result.content).toBe('Content');
    });
  });

  describe('计数器管理', () => {
    test('应该正确初始化计数器', () => {
      const processor = new AdvancedEnvironmentProcessor();
      
      expect(processor.getCounter('theorem')).toBe(0);
      expect(processor.getCounter('definition')).toBe(0);
      expect(processor.getCounter('example')).toBe(0);
    });

    test('应该正确递增计数器', () => {
      advancedEnvProcessor.processTheorem('First');
      advancedEnvProcessor.processTheorem('Second');
      
      expect(advancedEnvProcessor.getCounter('theorem')).toBe(2);
    });

    test('应该支持重置计数器', () => {
      advancedEnvProcessor.processTheorem('First');
      advancedEnvProcessor.processTheorem('Second');
      
      expect(advancedEnvProcessor.getCounter('theorem')).toBe(2);
      
      advancedEnvProcessor.resetCounters();
      
      expect(advancedEnvProcessor.getCounter('theorem')).toBe(0);
    });

    test('应该支持设置计数器', () => {
      advancedEnvProcessor.setCounter('theorem', 5);
      const result = advancedEnvProcessor.processTheorem('Test');
      
      expect(result.theorem.number).toBe(6);
    });

    test('应该为不同类型维护独立计数器', () => {
      advancedEnvProcessor.processTheorem('Theorem 1');
      advancedEnvProcessor.processTheorem('Theorem 2');
      advancedEnvProcessor.processDefinition('Definition 1');
      advancedEnvProcessor.processExample('Example 1');
      
      expect(advancedEnvProcessor.getCounter('theorem')).toBe(2);
      expect(advancedEnvProcessor.getCounter('definition')).toBe(1);
      expect(advancedEnvProcessor.getCounter('example')).toBe(1);
    });
  });

  describe('LaTeX环境语法处理', () => {
    test('应该解析完整的LaTeX环境', () => {
      const input = `
        \\begin{theorem}
        This is a theorem statement.
        \\end{theorem}
      `;
      
      const result = advancedEnvProcessor.process(input);
      
      expect(result.success).toBe(true);
      expect(result.result.type).toBe('theorem-environment');
      expect(result.result.theorem.content).toContain('This is a theorem statement.');
    });

    test('应该处理带标题和标签的环境', () => {
      const input = `
        \\begin{theorem}[Pythagorean Theorem]
        In a right triangle, $a^2 + b^2 = c^2$.
        \\label{thm:pythagoras}
        \\end{theorem}
      `;
      
      const result = advancedEnvProcessor.process(input);
      
      expect(result.success).toBe(true);
      expect(result.result.theorem.title).toContain('Pythagorean Theorem');
      expect(result.result.theorem.label).toBe('thm:pythagoras');
    });

    test('应该处理证明环境', () => {
      const input = `
        \\begin{proof}
        We proceed by induction...
        \\end{proof}
      `;
      
      const result = advancedEnvProcessor.process(input);
      
      expect(result.success).toBe(true);
      expect(result.result.type).toBe('proof-environment');
      expect(result.result.proof.qed).toBe(true);
    });

    test('应该检测不完整的环境语法', () => {
      const input = '\\begin{theorem}';
      
      const result = advancedEnvProcessor.process(input);
      
      expect(result.success).toBe(false);
      expect(result.errors).toContain('Incomplete environment syntax');
    });
  });

  describe('BaseProcessor接口实现', () => {
    test('应该实现process方法', () => {
      const input = 'This is a theorem.';
      const result = advancedEnvProcessor.process(input);
      
      expect(result).toHaveProperty('success');
      expect(result).toHaveProperty('result');
      expect(result).toHaveProperty('metadata');
      expect(result).toHaveProperty('warnings');
      expect(result).toHaveProperty('errors');
      
      expect(typeof result.success).toBe('boolean');
      expect(Array.isArray(result.warnings)).toBe(true);
      expect(Array.isArray(result.errors)).toBe(true);
    });

    test('应该实现validate方法', () => {
      const input = '\\begin{theorem}Content\\end{theorem}';
      const result = advancedEnvProcessor.validate(input);
      
      expect(result).toHaveProperty('valid');
      expect(result).toHaveProperty('errors');
      expect(result).toHaveProperty('warnings');
      expect(result).toHaveProperty('suggestions');
      
      expect(typeof result.valid).toBe('boolean');
      expect(Array.isArray(result.errors)).toBe(true);
      expect(Array.isArray(result.warnings)).toBe(true);
      expect(Array.isArray(result.suggestions)).toBe(true);
    });

    test('应该实现getConfig方法', () => {
      const config = advancedEnvProcessor.getConfig();
      
      expect(config).toBeDefined();
      expect(typeof config).toBe('object');
      expect(config).toHaveProperty('enableNumbering');
      expect(config).toHaveProperty('theoremStyle');
      expect(config).toHaveProperty('customEnvironments');
    });
  });

  describe('输入验证', () => {
    test('应该验证有效的LaTeX环境', () => {
      const validInputs = [
        '\\begin{theorem}Content\\end{theorem}',
        '\\begin{proof}Proof content\\end{proof}',
        '\\begin{definition}Definition content\\end{definition}'
      ];
      
      validInputs.forEach(input => {
        const result = advancedEnvProcessor.validate(input);
        expect(result.valid).toBe(true);
      });
    });

    test('应该检测不匹配的环境', () => {
      const input = '\\begin{theorem}Content\\end{proof}';
      const result = advancedEnvProcessor.validate(input);
      
      expect(result.valid).toBe(false);
      expect(result.errors.some(e => e.includes('mismatch'))).toBe(true);
    });

    test('应该检测不匹配的begin/end数量', () => {
      const input = '\\begin{theorem}\\begin{proof}Content\\end{theorem}';
      const result = advancedEnvProcessor.validate(input);
      
      expect(result.valid).toBe(false);
      expect(result.errors.some(e => e.includes('Mismatched'))).toBe(true);
    });

    test('应该警告未知环境类型', () => {
      const input = '\\begin{unknownenv}Content\\end{unknownenv}';
      const result = advancedEnvProcessor.validate(input);
      
      expect(result.warnings.some(w => w.includes('Unknown environment'))).toBe(true);
    });

    test('应该警告空环境内容', () => {
      const input = '\\begin{theorem}\\end{theorem}';
      const result = advancedEnvProcessor.validate(input);
      
      expect(result.warnings.some(w => w.includes('empty content'))).toBe(true);
    });

    test('应该验证环境对象', () => {
      const validObj = { type: 'theorem', content: 'Content' };
      const invalidObj = { content: 'Content' }; // 缺少type
      
      const result1 = advancedEnvProcessor.validate(validObj);
      const result2 = advancedEnvProcessor.validate(invalidObj);
      
      expect(result1.valid).toBe(true);
      expect(result2.valid).toBe(false);
      expect(result2.errors).toContain('Environment object is missing type');
    });

    test('应该检测无效输入类型', () => {
      const result = advancedEnvProcessor.validate(123);
      
      expect(result.valid).toBe(false);
      expect(result.errors).toContain('Input must be a string or environment object');
    });
  });

  describe('错误处理', () => {
    test('应该优雅处理无效输入', () => {
      const result = advancedEnvProcessor.process(null);
      
      expect(result).toHaveProperty('success', false);
      expect(result).toHaveProperty('errors');
      expect(result.errors.length).toBeGreaterThan(0);
    });

    test('应该处理无效环境对象', () => {
      const result = advancedEnvProcessor.process({ invalid: true });
      
      expect(result.success).toBe(false);
      expect(result.errors).toContain('Invalid environment object');
    });

    test('应该提供有用的错误信息', () => {
      const result = advancedEnvProcessor.validate('\\begin{theorem}\\end{proof}');
      
      if (!result.valid) {
        expect(result.errors.length).toBeGreaterThan(0);
        expect(result.suggestions.length).toBeGreaterThan(0);
      }
    });
  });

  describe('集成测试', () => {
    test('应该处理复杂的定理-证明组合', () => {
      const theoremResult = advancedEnvProcessor.process(`
        \\begin{theorem}[Fundamental Theorem]
        This is an important theorem.
        \\label{thm:fundamental}
        \\end{theorem}
      `);
      
      const proofResult = advancedEnvProcessor.process(`
        \\begin{proof}[Proof of Theorem 1]
        The proof follows from basic principles.
        \\end{proof}
      `);
      
      expect(theoremResult.success).toBe(true);
      expect(proofResult.success).toBe(true);
      
      expect(theoremResult.result.theorem.label).toBe('thm:fundamental');
      expect(proofResult.result.proof.title).toBe('Proof of Theorem 1');
    });

    test('应该支持多种定理类型的混合使用', () => {
      const environments = [
        { input: 'Theorem statement', type: 'theorem' },
        { input: 'Lemma statement', type: 'lemma' },
        { input: 'Definition statement', type: 'definition' },
        { input: 'Example statement', type: 'example' }
      ];
      
      environments.forEach(env => {
        const result = advancedEnvProcessor.process(env.input, { type: env.type });
        
        expect(result.success).toBe(true);
        expect(result.result.theoremType).toBe(env.type);
        expect(result.result.theorem.number).toBeGreaterThan(0);
      });
    });

    test('应该支持自定义环境配置', () => {
      const processor = new AdvancedEnvironmentProcessor({
        customEnvironments: {
          'observation': { style: 'remark' },
          'conjecture': { style: 'theorem' }
        }
      });
      
      const obsResult = processor.processEnvironment('observation', 'This is an observation');
      const conjResult = processor.processEnvironment('conjecture', 'This is a conjecture');
      
      expect(obsResult.type).toBe('theorem-environment');
      expect(conjResult.type).toBe('theorem-environment');
    });

    test('应该维护全局计数器状态', () => {
      // 创建多个定理
      const results = [];
      for (let i = 1; i <= 5; i++) {
        results.push(advancedEnvProcessor.processTheorem(`Theorem ${i}`));
      }
      
      // 验证编号递增
      results.forEach((result, index) => {
        expect(result.theorem.number).toBe(index + 1);
      });
      
      // 创建定义
      const defResult = advancedEnvProcessor.processDefinition('A definition');
      expect(defResult.theorem.number).toBe(1); // 定义有自己的计数器
    });

    test('应该正确处理复杂的嵌套内容', () => {
      const complexContent = `
        [Main Theorem] 
        For any function $f: \\mathbb{R} \\to \\mathbb{R}$, 
        if $f$ is continuous, then:
        \\begin{enumerate}
        \\item $f$ is measurable
        \\item $f$ satisfies property P
        \\end{enumerate}
        \\label{thm:main}
      `;
      
      const result = advancedEnvProcessor.processTheorem(complexContent);
      
      expect(result.success).toBe(true);
      expect(result.theorem.title).toContain('Main Theorem');
      expect(result.theorem.label).toBe('thm:main');
      expect(result.theorem.content).toContain('enumerate');
    });
  });
});