# Processors 模块接口文档

## 模块概述

Processors 模块提供LaTeX Parser V3的可选处理器扩展，用于处理特定领域的LaTeX功能。这些处理器可以独立使用，也可以与核心解析器配合，提供高级的内容处理能力。

## 公共接口

### MathProcessor

**位置**: `processors/MathProcessor.js`

**职责**: 高级数学公式处理和KaTeX集成

**构造函数**:
```javascript
new MathProcessor(options = {})
```

**配置选项**:
- `enableKaTeX` {Boolean} - 是否启用KaTeX渲染，默认true
- `displayMode` {Boolean} - 默认显示模式，默认false
- `throwOnError` {Boolean} - 遇到错误是否抛出异常，默认false

**核心方法**:
```javascript
renderInlineMath(formula: String): String
renderDisplayMath(formula: String): String
processFormula(formula: String, options: Object): String
validateSyntax(formula: String): ValidationResult
```

### FigureProcessor

**位置**: `processors/FigureProcessor.js`

**职责**: 图片和浮动环境处理

**核心方法**:
```javascript
processFigure(content: String, options: Object): ProcessedFigure
processIncludeGraphics(path: String, options: Object): ImageElement
generateCaption(content: String): CaptionElement
handleFloatPositioning(position: String): PositionConfig
```

### ReferenceProcessor

**位置**: `processors/ReferenceProcessor.js`

**职责**: 交叉引用和链接处理

**核心方法**:
```javascript
processRef(refId: String, type: String): ReferenceElement
processCite(citeKey: String, style: String): CitationElement
processLabel(labelId: String, context: Object): LabelElement
generateReferenceList(): ReferenceList
```

### FootnoteProcessor

**位置**: `processors/FootnoteProcessor.js`

**职责**: 脚注系统处理

**核心方法**:
```javascript
processFootnote(content: String, id: String): FootnoteResult
generateFootnoteList(): FootnoteList
linkFootnoteMarkers(): void
processFootnoteRef(refId: String): FootnoteReference
```

### MetadataProcessor

**位置**: `processors/MetadataProcessor.js`

**职责**: 文档元数据提取和处理

**核心方法**:
```javascript
extractMetadata(parseResult: Object): DocumentMetadata
processTitle(title: String): TitleElement
processAuthor(author: String): AuthorElement
processDate(date: String): DateElement
generateDocumentInfo(): DocumentInfo
```

### TOCProcessor

**位置**: `processors/TOCProcessor.js`

**职责**: 目录生成和导航处理

**核心方法**:
```javascript
generateTOC(parseResult: Object, options: Object): TableOfContents
extractHeadings(content: String): HeadingList
processSection(section: String, level: Number): SectionElement
generateNavigationLinks(): NavigationList
```

### BibliographyProcessor

**位置**: `processors/BibliographyProcessor.js`

**职责**: 参考文献处理

**核心方法**:
```javascript
processBibliography(bibData: Object): Bibliography
processCitation(citation: String, style: String): CitationElement
generateBibEntry(entry: Object): BibliographyEntry
formatReference(ref: Object, style: String): String
```

### AdvancedEnvironmentProcessor

**位置**: `processors/AdvancedEnvironmentProcessor.js`

**职责**: 复杂LaTeX环境处理

**核心方法**:
```javascript
processTheorem(content: String, theoremType: String): TheoremElement
processProof(content: String): ProofElement
processDefinition(content: String): DefinitionElement
processExample(content: String): ExampleElement
```

### DocumentClassProcessor

**位置**: `processors/DocumentClassProcessor.js`

**职责**: 文档类别和样式处理

**核心方法**:
```javascript
processDocumentClass(className: String, options: Array): DocumentClass
applyClassStyles(className: String): StyleConfig
processPackageOptions(packageName: String, options: Array): PackageConfig
generateLayoutConfig(className: String): LayoutConfig
```

## 统一的处理器接口

### 基础处理器接口

所有处理器都应该实现的基础接口：

```javascript
class BaseProcessor {
  constructor(options = {}) {}
  
  process(input: Any, options: Object = {}): ProcessResult
  validate(input: Any): ValidationResult
  getConfig(): ProcessorConfig
}
```

### ProcessResult 通用格式

```javascript
{
  success: Boolean,         // 处理是否成功
  result: Any,             // 处理结果
  metadata: Object,        // 处理元信息
  warnings: Array,         // 警告信息
  errors: Array           // 错误信息
}
```

### ValidationResult 格式

```javascript
{
  valid: Boolean,          // 是否有效
  errors: Array,          // 验证错误
  warnings: Array,        // 验证警告
  suggestions: Array      // 修正建议
}
```

## 集成模式

### 与核心解析器集成

处理器可以在不同阶段与核心解析器集成：

1. **预处理阶段**: 在五层解析器之前处理
2. **中间处理阶段**: 在特定层之后处理
3. **后处理阶段**: 在所有解析完成后处理

### 使用示例

```javascript
const LaTeXParser = require('../core/LaTeXParser');
const MathProcessor = require('./MathProcessor');
const FigureProcessor = require('./FigureProcessor');

const parser = new LaTeXParser();
const mathProcessor = new MathProcessor({ enableKaTeX: true });
const figureProcessor = new FigureProcessor();

// 基础解析
const parseResult = parser.parseComplete(latexContent);

// 扩展处理
const mathResult = mathProcessor.process(parseResult.blocks.mathBlocks);
const figureResult = figureProcessor.process(parseResult.blocks.figures);
```

## 依赖关系

### 输入依赖
- `core/LaTeXParser.js` - 接收核心解析结果
- `config/latex-html-mapping.js` - 使用配置规则
- `utils/` 模块 - 使用工具函数

### 输出依赖
- `renderers/HTMLRenderer.js` - 提供处理结果给渲染器
- `renderers/CSSGenerator.js` - 提供样式信息

### 可选依赖
- KaTeX库 (MathProcessor)
- 图片处理库 (FigureProcessor)
- 引用数据库 (BibliographyProcessor)

## 错误处理约定

- 所有处理器都应该实现优雅的错误处理
- 提供详细的错误信息和修复建议
- 支持降级处理，失败时不影响基础功能
- 警告信息不中断处理流程

## 扩展性设计

- 支持自定义处理器插件
- 提供标准的处理器接口
- 支持异步处理模式
- 支持处理器链式调用和组合